-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at:
--         http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

-- Description: test the global IPMI sensor interfaces of sensor.
local log = require 'mc.logging'
local bs = require 'mc.bitstring'
local ipmi = require 'sensor.ipmi.ipmi'
local common = require 'test_sensor_common'

local ipmi_interface = {}
ipmi_interface.__index = ipmi_interface

function ipmi_interface.test_global_enables(bus)
    log:info('== Testing global enables ...')
    local enables_pattern = bs.new([[<<
        revmsg_ipt:1, evt_msgful_ipt:1, evt_msgbuf:1, sel:1,
        reserved:1, oem0:1, oem1:1, oem2:1
    >>]])

    log:info('== 1. Testing get current global enables ...')
    -- ipmi command: 'ipmitool raw 0x06 0x2F'
    local netfn, cmd = 0x06, 0x2F
    local ok, ret = common.call_ipmi(bus, netfn, cmd)
    assert(ok)
    local rsp = bs.new(ipmi.GetBMCEnables.encode):unpack(ret)
    assert(rsp.CompletionCode == 0x00, 'actual: ' .. rsp.CompletionCode)
    local enable = enables_pattern:unpack(string.pack('I1', rsp.Enable))
    assert(enable.revmsg_ipt == 0, 'actual: ' .. enable.revmsg_ipt)

    log:info('== 2. Testing set global enables except host ...')
    -- ipmi command: 'ipmitool raw 0x06 0x2E <enables>'
    cmd = 0x2E
    enable.oem0 = 1
    local req = ipmi.SetBMCEnables.req.new(string.unpack('I1', enables_pattern:pack(enable)))
    local payload = bs.new(ipmi.SetBMCEnables.decode):pack(req)
    ok, ret = common.call_ipmi(bus, netfn, cmd, payload)
    assert(ok)
    rsp = bs.new(ipmi.SetBMCEnables.encode):unpack(ret)
    assert(rsp.CompletionCode == 0xc1, 'actual: ' .. rsp.CompletionCode)

    log:info('== 3. Testing set global enables in host ...')
    -- ipmi command: 'ipmitool raw 0x06 0x2E <enables>'
    cmd = 0x2E
    enable.oem0 = 1
    req = ipmi.SetBMCEnables.req.new(string.unpack('I1', enables_pattern:pack(enable)))
    payload = bs.new(ipmi.SetBMCEnables.decode):pack(req)
    ok, ret = common.call_ipmi(bus, netfn, cmd, payload, 1)
    assert(ok)
    rsp = bs.new(ipmi.SetBMCEnables.encode):unpack(ret)
    assert(rsp.CompletionCode == 0x00, 'actual: ' .. rsp.CompletionCode)

    log:info('== 4. Checking set global enables ...')
    cmd = 0x2F
    ok, ret = common.call_ipmi(bus, netfn, cmd)
    assert(ok)
    rsp = bs.new(ipmi.GetBMCEnables.encode):unpack(ret)
    assert(rsp.CompletionCode == 0x00, 'actual: ' .. rsp.CompletionCode)
    enable = enables_pattern:unpack(string.pack('I1', rsp.Enable))
    assert(enable.oem0 == 1, 'actual: ' .. enable.oem0)
end

function ipmi_interface.test_entry(bus)
    log:info('================ test global ipmi interface start ================')

    ipmi_interface.test_global_enables(bus)

    log:info('================ test global ipmi interface complete ================')
end

return ipmi_interface