-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at:
--         http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

-- Description: error code and process for sensor module.
local ex = require 'messages.custom'
local er = require 'sensor.errors'
local eb = require 'messages.base'

local sensor_err = {}
sensor_err.__index = sensor_err

-- 传感器的通用错误码
sensor_err.SUCCESS = 0
sensor_err.ERR_INVALID_ID = 1
sensor_err.ERR_INVALID_FIELD = 2
sensor_err.ERR_CANNOT_RESPONSE = 3
sensor_err.ERR_ILLEGAL = 4
sensor_err.ERR_INVALID_RSVID = 5
sensor_err.ERR_SEL_ERASED = 6
sensor_err.ERR_CANNOT_SUPPORT = 7
sensor_err.ERR_OUT_OF_RANGE = 8
sensor_err.ERR_LENGTH_LARGE = 9
sensor_err.ERR_SDR_IN_UPDATE_MODE = 10
sensor_err.ERR_OUT_OF_SPACE = 11
sensor_err.ERR_UNSPECIFIED = 12
sensor_err.ERR_INVALID_CMD = 13
sensor_err.ERR_DES_UNAVAILABLE = 14
sensor_err.ERR_INVALID_PATH = 15
sensor_err.DUPLICATE_EXPORTING_ERR = 16
sensor_err.PROPERTY_VALUE_FORMAT_ERROR = 17
sensor_err.ERR_INVALID_LENGTH = 18

-- 传感器的其他错误码
sensor_err.ERR_PEF_CANNOT_SUPPORT = 100
sensor_err.ERR_PEF_IN_PROGRESS = 101
sensor_err.ERR_PEF_READONLY = 102
sensor_err.ERR_INVALID_MOCK_NAME = 103
sensor_err.ERR_INVALID_MOCK_VALUE = 104
sensor_err.ERR_INVALID_MOCK_FAILED = 105
sensor_err.ERR_INVALID_ENABLE_FAILED = 106
sensor_err.ERR_INVALID_NOT_SUPPORTED = 107

-- sensor 错误码对应ipmi错误码
function sensor_err.ipmi_error_map(err)
    local map = {
        [sensor_err.ERR_INVALID_ID] = ex.IPMIInvaildId,
        [sensor_err.ERR_INVALID_FIELD] = ex.IPMIInvalidFieldRequest,
        [sensor_err.ERR_CANNOT_RESPONSE] = ex.IPMICommandResponseCannotProvide,
        [sensor_err.ERR_SEL_ERASED] = ex.IPMISelEraseInProgress,
        [sensor_err.ERR_ILLEGAL] = ex.IPMICommandIllegal,
        [sensor_err.ERR_INVALID_RSVID] = ex.IPMIInvalidRsvId,
        [sensor_err.ERR_CANNOT_SUPPORT] = ex.IPMICommandCannotExecute,
        [sensor_err.ERR_OUT_OF_RANGE] = ex.IPMIOutOfRange,
        [sensor_err.ERR_PEF_CANNOT_SUPPORT] = ex.IPMIPefSetNotSupported,
        [sensor_err.ERR_PEF_IN_PROGRESS] = ex.IPMIPefSetInProgress,
        [sensor_err.ERR_PEF_READONLY] = ex.IPMIPefSetReadOnly,
        [sensor_err.ERR_LENGTH_LARGE] = ex.IPMIRequestLengthLarge,
        [sensor_err.ERR_SDR_IN_UPDATE_MODE] = ex.IPMISDRUpdating,
        [sensor_err.ERR_INVALID_CMD] = ex.IPMIInvalidCommand,
        [sensor_err.ERR_OUT_OF_SPACE] = ex.IPMIOutOfSpace,
        [sensor_err.ERR_DES_UNAVAILABLE] = ex.IPMIDestinationUnavailable,
        [sensor_err.ERR_INVALID_LENGTH] = ex.IPMIRequestLengthInvalid,
        [sensor_err.ERR_UNSPECIFIED] = er.unspecified_error
    }

    if map[err] then return map[err]() end
end

function sensor_err.sensor_error_map(err, ...)
    local map = {
        [sensor_err.ERR_INVALID_MOCK_NAME] = ex.InvalidSensorName,
        [sensor_err.ERR_INVALID_MOCK_VALUE] = ex.InvalidSensorValue,
        [sensor_err.ERR_INVALID_MOCK_FAILED] = ex.SensorTestFailed,
        [sensor_err.ERR_INVALID_ENABLE_FAILED] = ex.SensorEnableFailed,
        [sensor_err.ERR_INVALID_NOT_SUPPORTED] = ex.SensorNotSupported,
        [sensor_err.ERR_INVALID_PATH] = ex.InvalidPath,
        [sensor_err.DUPLICATE_EXPORTING_ERR] = ex.DuplicateExportingErr,
        [sensor_err.PROPERTY_VALUE_FORMAT_ERROR] = eb.PropertyValueFormatError
    }

    if map[err] then return map[err](...) end
end

return sensor_err
