-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at: http://license.coscl.org.cn/MulanPSL2
--
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

local utils = require 'mc.utils'

local m = {}

-- 处理器资源树接口
local OLD_VNC_PATH = "/bmc/kepler/Managers/1/VncService"
local OLD_VNC_INTF = "bmc.kepler.Managers.VncService"
local VNC_PATH = "/bmc/kepler/Systems/%s/VncService"
local VNC_INTF = "bmc.kepler.Systems.VncService"
local SESSION_PATH = "/bmc/kepler/SessionService/Sessions"
local SESSION_INTF = "bmc.kepler.SessionService.Session"
local VNC_SESSION_PATH = "/bmc/kepler/SessionService/VNC"
local VNC_SESSION_INTF = "bmc.kepler.SessionService"
local DEFAULT_MAX_SESSION_NUM = 5
local DEFAULT_SYSTEM_ID = 1

local function get_max_session_count()
    local ok, vnc_session = pcall(mdb.get_object, bus, VNC_SESSION_PATH, VNC_SESSION_INTF)
    if not ok then
        return DEFAULT_MAX_SESSION_NUM
    end
    return vnc_session.SessionMaxCount
end

local function get_number_of_active_sessions(system_id, activated_id)
    local session_type
    local count = 0
    if activated_id and activated_id ~= cjson.null and activated_id ~= tonumber(system_id) then
        return count
    end
    local ok, sessions = pcall(mdb.get_sub_objects, bus, SESSION_PATH, SESSION_INTF)
    if not ok then
        sessions = {}
    end
    for _, session in pairs(sessions) do
        session_type = string.lower(session.SessionType)
        if string.match(session_type, 'vnc') then
            count = count + 1
        end
    end
    return count
end

local function get_vnc_settings_once(system_id, vnc_path, vnc_intf, activated_id)
    local data = cjson.json_object_new_object()
    data['SystemId'] = system_id
    local ok, vnc_obj = pcall(mdb.get_object, bus, vnc_path, vnc_intf)
    if not ok or not vnc_obj then
        data['DisableKeyboardDuringBiosStartup'] = cjson.null
    else
        data['DisableKeyboardDuringBiosStartup'] = vnc_obj.DisableKeyboardDuringBiosStartup
    end
    data['MaximumNumberOfSessions'] = get_max_session_count()
    data['NumberOfActiveSessions'] = get_number_of_active_sessions(system_id, activated_id)
    return data
end

function m.get_vnc_settings(system_ids, activated_id)
    local res = cjson.json_object_new_array()
    if not system_ids or system_ids == cjson.null or #system_ids == 0 then
        res[#res+1] = get_vnc_settings_once(DEFAULT_SYSTEM_ID, OLD_VNC_PATH, OLD_VNC_INTF, DEFAULT_SYSTEM_ID)
        return res
    end
    local vnc_path
    for _, system_id in ipairs(system_ids) do
        vnc_path = string.format(VNC_PATH, system_id)
        res[#res+1] = get_vnc_settings_once(system_id, vnc_path, VNC_INTF, activated_id)
    end
    return res
end

function m.parse_vnc_settings(input)
    local data = input[1]
    local system_ids = input[2]
    local res = cjson.json_object_new_object()
    if not data or data == cjson.null or #data == 0 then
        return res
    end
    local err
    for _, sub_data in ipairs(data) do
        if not sub_data.SystemId then
            err = base_messages.PropertyMissing('SystemId')
            err.RelatedProperties = {'#/VNCSettings/SystemId'}
            error(err)
        end
        if (system_ids == cjson.null and sub_data.SystemId ~= DEFAULT_SYSTEM_ID) or
            (system_ids ~= cjson.null and not utils.array_contains(system_ids, sub_data.SystemId)) then
            err = custom_messages.OperationNotSupport()
            err.RelatedProperties = {'#/VNCSettings/SystemId'}
            error(err)
        end
        local tmp_data = cjson.json_object_new_object()
        tmp_data['DisableKeyboardDuringBiosStartup'] = sub_data.DisableKeyboardDuringBiosStartup
        res[sub_data.SystemId] = tmp_data
    end
    return res
end

return m
