-- Copyright (c) 2025 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at: http://license.coscl.org.cn/MulanPSL2
--
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

local utils = require 'mc.utils'

local vmm = {}

local VIRTUAL_MEDIA_PATH_PATTERN = "/bmc/kepler/Managers/1/NetworkProtocol/VirtualMedia/"
local VIRTUAL_MEDIA_PATH = "/bmc/kepler/Managers/1/NetworkProtocol/VirtualMedia"
local PORT_CONFIG_INTERFACE = "bmc.kepler.Managers.NetworkProtocol.PortConfig"
local OLD_CD_PATH = "/bmc/kepler/Managers/1/VirtualMedia/CD"
local OLD_CD_INTERFACE = "bmc.kepler.Managers.VirtualMedia.CD"
local CD_PATH_PATTERN = "/bmc/kepler/Systems/%s/VirtualMedia/CD"
local CD_INTERFACE = "bmc.kepler.Systems.VirtualMedia.CD"
local DEFAULT_SYSTEM_ID = 1
local DEFAULT_MAX_SESSION_NUM = 1

local function get_virtual_media_config_once(system_id, virtual_media_path, cd_path, cd_intf)
    local data = cjson.json_object_new_object()
    local ok, vmm_obj = pcall(mdb.get_object, bus, virtual_media_path, PORT_CONFIG_INTERFACE)
    if not ok then
        return data
    end
    data['SystemId'] = system_id
    data['Port'] = vmm_obj.Port
    data['Enabled'] = vmm_obj.Enabled
    ok, vmm_obj = pcall(mdb.get_object, bus, cd_path, cd_intf)
    if not ok then
        return data
    end
    data['MaximumNumberOfSessions'] = DEFAULT_MAX_SESSION_NUM
    data['NumberOfActiveSessions'] = vmm_obj.ConnectNum
    return data
end

function vmm.get_virtual_media_config(system_ids)
    local res = cjson.json_object_new_array()
    if not system_ids or system_ids == cjson.null or #system_ids == 0 then
        res[#res+1] = get_virtual_media_config_once(DEFAULT_SYSTEM_ID, VIRTUAL_MEDIA_PATH, OLD_CD_PATH, OLD_CD_INTERFACE)
        return res
    end
    local path
    for _, system_id in ipairs(system_ids) do
        path = string.format(CD_PATH_PATTERN, system_id)
        res[#res+1] = get_virtual_media_config_once(system_id, VIRTUAL_MEDIA_PATH_PATTERN .. system_id, path, CD_INTERFACE)
    end
    return res
end

function vmm.parse_virtual_media_config(input)
    local data = input[1]
    local system_ids = input[2]
    local res = cjson.json_object_new_object()
    if not data or data == cjson.null or #data == 0 then
        return res
    end
    local err
    for _, sub_data in ipairs(data) do
        if not sub_data.SystemId then
            err = base_messages.PropertyMissing('SystemId')
            err.RelatedProperties = {'#/VirtualMediaConfig/SystemId'}
            error(err)
        end
        if (system_ids == cjson.null and sub_data.SystemId ~= DEFAULT_SYSTEM_ID) or
            (system_ids ~= cjson.null and not utils.array_contains(system_ids, sub_data.SystemId)) then
            err = custom_messages.OperationNotSupport()
            err.RelatedProperties = {'#/VirtualMediaConfig/SystemId'}
            error(err)
        end
        local tmp_data = cjson.json_object_new_object()
        tmp_data['Port'] = sub_data.Port
        tmp_data['Enabled'] = sub_data.Enabled
        res[sub_data.SystemId] = tmp_data
    end
    return res
end

return vmm
