-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at: http://license.coscl.org.cn/MulanPSL2
--
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

local utils = require 'mc.utils'

local port_config = {}

-- 处理器资源树接口
local JAR_FILE_PATH <const> = "/opt/bmc/web/htdocs/bmc/pages/jar/vconsole.jar"
local KVMIP_PATH = "/bmc/kepler/Managers/1/NetworkProtocol/KVMIP"
local KVMIP_PATH_PATTERN = "/bmc/kepler/Managers/1/NetworkProtocol/KVMIP/"
local VIRTUAL_MEDIA_PATH = "/bmc/kepler/Managers/1/NetworkProtocol/VirtualMedia"
local VIRTUAL_MEDIA_PATH_PATTERN = "/bmc/kepler/Managers/1/NetworkProtocol/VirtualMedia/"
local PORT_CONFIG_INTERFACE = "bmc.kepler.Managers.NetworkProtocol.PortConfig"
local DEFAULT_SYSTEM_ID = 1

local function get_network_protocol_once(system_id, kvmip_path)
    local data = cjson.json_object_new_object()
    local kvmip_obj = mdb.get_object(bus, kvmip_path, PORT_CONFIG_INTERFACE)
    if not kvmip_obj then
        return data
    end
    data['SystemId'] = system_id
    data['Port'] = kvmip_obj.Port
    data['Enabled'] = kvmip_obj.Enabled
    return data
end

function port_config.get_graphical_console(system_ids)
    local res = cjson.json_object_new_array()
    if not system_ids or system_ids == cjson.null or #system_ids == 0 then
        res[#res+1] = get_network_protocol_once(DEFAULT_SYSTEM_ID, KVMIP_PATH)
        return res
    end
    for _, system_id in ipairs(system_ids) do
        res[#res+1] = get_network_protocol_once(system_id, KVMIP_PATH_PATTERN .. system_id)
    end
    return res
end

function port_config.parse_graphical_console(input)
    local data = input[1]
    local system_ids = input[2]
    local res = cjson.json_object_new_object()
    if not data or data == cjson.null or #data == 0 then
        return res
    end
    local err
    for _, sub_data in ipairs(data) do
        if not sub_data.SystemId then
            err = base_messages.PropertyMissing('SystemId')
            err.RelatedProperties = {'#/KVM/SystemId'}
            error(err)
        end
        if (system_ids == cjson.null and sub_data.SystemId ~= DEFAULT_SYSTEM_ID) or
            (system_ids ~= cjson.null and not utils.array_contains(system_ids, sub_data.SystemId)) then
            err = custom_messages.OperationNotSupport()
            err.RelatedProperties = {'#/KVM/SystemId'}
            error(err)
        end
        local tmp_data = cjson.json_object_new_object()
        tmp_data['Port'] = sub_data.Port
        tmp_data['Enabled'] = sub_data.Enabled
        res[sub_data.SystemId] = tmp_data
    end
    return res
end

function port_config.get_virtual_media_config(system_ids)
    local res = cjson.json_object_new_array()
    if not system_ids or system_ids == cjson.null or #system_ids == 0 then
        res[#res+1] = get_network_protocol_once(DEFAULT_SYSTEM_ID, VIRTUAL_MEDIA_PATH)
        return res
    end
    for _, system_id in ipairs(system_ids) do
        res[#res+1] = get_network_protocol_once(system_id, VIRTUAL_MEDIA_PATH_PATTERN .. system_id)
    end
    return res
end

function port_config.parse_virtual_media_config(input)
    local data = input[1]
    local system_ids = input[2]
    local res = cjson.json_object_new_object()
    if not data or data == cjson.null or #data == 0 then
        return res
    end
    local err
    for _, sub_data in ipairs(data) do
        if not sub_data.SystemId then
            err = base_messages.PropertyMissing('SystemId')
            err.RelatedProperties = {'#/VMM/SystemId'}
            error(err)
        end
        if (system_ids == cjson.null and sub_data.SystemId ~= DEFAULT_SYSTEM_ID) or
            (system_ids ~= cjson.null and not utils.array_contains(system_ids, sub_data.SystemId)) then
            err = custom_messages.OperationNotSupport()
            err.RelatedProperties = {'#/VMM/SystemId'}
            error(err)
        end
        local tmp_data = cjson.json_object_new_object()
        tmp_data['Port'] = sub_data.Port
        tmp_data['Enabled'] = sub_data.Enabled
        res[sub_data.SystemId] = tmp_data
    end
    return res
end

return port_config
