-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at: http://license.coscl.org.cn/MulanPSL2
--
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.
local utils = require 'mc.utils'
local log = require 'mc.logging'
local file_sec = require 'utils.file'
local m = {}

local file_transfer_list = {
    {
        Id = 3,
        FilePath = '/tmp/dump_info.tar.gz',
        Maxlength = 0,
        Permission = utils.S_IRUSR | utils.S_IRGRP,
        Privilege = 'SecurityMgmt'
    },
    {
        Id = 4,
        FilePath = '',
        Maxlength = 0x5A00000,
        Permission = utils.S_IRUSR | utils.S_IRGRP,
        Privilege = 'BasicSetting'
    },
    {
        Id = 6,
        FilePath = '/tmp/systemcom.tar',
        Maxlength = 0,
        Permission = utils.S_IRUSR | utils.S_IRGRP,
        Privilege = 'DiagnoseMgmt'
    },
    {
        Id = 7,
        FilePath = '/tmp/blackbox.tar',
        Maxlength = 0,
        Permission = utils.S_IRUSR | utils.S_IRGRP,
        Privilege = 'DiagnoseMgmt'
    }
}

local function get_file_transfer_info(id)
    for _, value in ipairs(file_transfer_list) do
        if value.Id == id then
            return value
        end
    end
    error(base_messages.InternalError())
end

local function check_user_privilege(user_privilege, target_privilege)
    for _, value in ipairs(user_privilege) do
        if target_privilege == value then
            return
        end
    end

    error(base_messages.InsufficientPrivilege())
end

local function get_file_name(file_path)
    local ok, sec_file_name = pcall(file_sec.get_file_name_s, file_path)
    if not ok then
        log:error('get_file_name_s error:%s', sec_file_name)
        error(base_messages.InternalError())
    end
    return sec_file_name
end

function m.get_file_transfer_para(url, user_privilege)
    local info = utils.split(url, ';')
    local transfer_type = info[1]
    local src_url, target_url
    local file_info = get_file_transfer_info(tonumber(info[2]))
    check_user_privilege(user_privilege, file_info.Privilege)
    if transfer_type == 'download' then
        src_url = info[3]
        if file_info.FilePath == '' then
            target_url = 'file:///tmp/' .. get_file_name(info[3])
        else
            target_url = 'file://' .. file_info.FilePath
        end
    end

    if transfer_type == 'upload' then
        target_url = info[3]
        src_url = 'file://' .. file_info.FilePath
    end

    return {
        SrcUrl = src_url,
        TargetUrl = target_url,
        MaxLength = file_info.Maxlength,
        Uid = 0,
        Gid = 0,
        Permission = file_info.Permission
    }
end

function m.get_file_transfer_state(tasks)
    if #tasks == 0 then
        return 0
    end

    local task = {}
    task.Obj = nil
    task.Time = 0
    for _, value in ipairs(tasks) do
        local task_obj = mdb.get_object(bus, value, 'bmc.kepler.TaskService.Task')
        local start_time = string.sub(task_obj.StartTime, 0, 19)
        local year, month, day, hour, min, sec = string.match(start_time, "(%d+)-(%d+)-(%d+)T(%d+):(%d+):(%d+)")
        local time = tonumber(string.format('%s%s%s%s%s%s', year, month, day, hour, min, sec))
        if time > task.Time then
            task.Obj = task_obj
            task.Time = time
        end
    end
    if task.Obj.Parameters == '{}' then
        return tonumber(task.Obj.Progress)
    end

    return tonumber(string.sub(task.Obj.Parameters, 14, #task.Obj.Parameters - 1))
end

return m
