-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at: http://license.coscl.org.cn/MulanPSL2
--
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.
local cjson = require 'cjson'
local performance_collection = {}

local SENSOR_TYPE_TEMPERATURE<const> = 1

local function get_thermal_fans(list)
    local res = {}
    for _, path in pairs(list) do
        local ok, fan_obj = pcall(mdb.get_object, bus, path, 'bmc.kepler.Systems.Fan')
        if not ok then
            goto continue
        end
        res[#res + 1] = fan_obj

        ::continue::
    end
    return res
end

-- 获取温度传感器的ReadingCelsius返回值
local function get_temperature_sensor_reading(value)
    -- 传感器禁止扫描时候将返回na
    if value == 'na' then
        return cjson.null
    end
    return tonumber(string.format('%.2f', value))
end

local function get_temperatures(sensor_obj, temperature_sensor_num)
    local rsp = cjson.json_object_new_object()
    rsp['@odata.id'] = '/redfish/v1/Chassis/1/Thermal#/Temperatures/' .. temperature_sensor_num
    rsp['Name'] = sensor_obj.SensorName
    rsp['ReadingCelsius'] = get_temperature_sensor_reading(sensor_obj.SensorReading)
    return rsp
end

function performance_collection.get_fans_info(fan_path_list)
    local fan_info_table = {}
    local fan_speed_ratio_table = {}
    local max_fan_id = 0
    local cjson_rsp = cjson.json_object_new_array()
    local thermal_fans = get_thermal_fans(fan_path_list)
    -- 所有风扇按照对称风扇进行id插入排序
    for _, fan_obj in pairs(thermal_fans) do
        if fan_obj.IsTwins == true then
            fan_info_table[fan_obj.FanId * 2 - 1] = fan_obj.FrontSpeed
            fan_info_table[fan_obj.FanId * 2] = fan_obj.RearSpeed
            fan_speed_ratio_table[fan_obj.FanId * 2 - 1] = fan_obj.FrontMaxSpeed == 0 and 0 or
                fan_obj.FrontSpeed * 100 // fan_obj.FrontMaxSpeed
            fan_speed_ratio_table[fan_obj.FanId * 2] = fan_obj.RearMaxSpeed == 0 and 0 or
                fan_obj.RearSpeed * 100 // fan_obj.RearMaxSpeed
        else
            fan_info_table[fan_obj.FanId * 2] = fan_obj.RearSpeed
            fan_speed_ratio_table[fan_obj.FanId * 2] = fan_obj.RearMaxSpeed == 0 and 0 or
                fan_obj.RearSpeed * 100 // fan_obj.RearMaxSpeed
        end
        max_fan_id = fan_obj.FanId > max_fan_id and fan_obj.FanId or max_fan_id
    end
    local cjson_rsp_size = 0
    for index = 1, max_fan_id * 2, 1 do
        if fan_info_table[index] then
            cjson_rsp_size = cjson_rsp_size + 1
            cjson_rsp[cjson_rsp_size] = cjson.json_object_new_object()
            cjson_rsp[cjson_rsp_size]["@odata.id"] = "/redfish/v1/Chassis/1/Thermal#/Fans/" .. cjson_rsp_size - 1
            cjson_rsp[cjson_rsp_size]["Reading"] = fan_info_table[index]
            cjson_rsp[cjson_rsp_size]["SpeedRatio"] = fan_speed_ratio_table[index]
        end
    end
    return cjson_rsp
end

function performance_collection.get_thermal_sensor(sensor_list)
    local res = cjson.json_object_new_array()
    local temperature_sensor_num = 0
    for _, obj in pairs(sensor_list) do
        -- 温度传感器
        if obj.SensorType == SENSOR_TYPE_TEMPERATURE and obj.SensorIdentifier == "Outlet Temp" then
            res[#res + 1] = get_temperatures(obj, temperature_sensor_num)
            temperature_sensor_num = temperature_sensor_num + 1
        end
    end
    return res
end

return performance_collection
