-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at: http://license.coscl.org.cn/MulanPSL2
--
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

local m = {}
local EVENTT_REGISTRY_PREFIX = 'Events'
local cjson = require 'cjson'

function m.get_members(count, skip, top, source_type)
    local log_format = source_type == 'Managers' and 'EventLog' or 'Log1'
    if not count then
        return
    end
    skip = skip or 0
    top = top or (count > 32 and 32 or count)
    if count <= tonumber(skip) then
        return
    end
    local list = {}
    for i = 1, top, 1 do
        if (count - skip - i + 1) > 0 then
            list[i] = string.format('/redfish/v1/%s/1/LogServices/%s/Entries/%s',
                source_type, log_format, count - skip - i + 1)
        end
    end
    return list
end

function m.get_timestamp(event)
    if not event then
        return
    end
    local time
    for _, v in pairs(event) do
        if v.MappingTable[1].Key == 'Timestamp' then
            time = v.MappingTable[1].Value
            break
        end
    end
    return time
end

function m.get_record(event)
    if not event then
        return
    end
    local record
    for _, v in pairs(event) do
        if v.MappingTable[1].Key == 'RecordId' then
            record = v.MappingTable[1].Value
            break
        end
    end
    return record
end

function m.get_code(event)
    if not event then
        return
    end
    local code
    for _, v in pairs(event) do
        if v.MappingTable[1].Key == 'EventCode' then
            code = v.MappingTable[1].Value
            break
        end
    end
    return code
end

function m.get_desc(event)
    if not event then
        return
    end
    local desc
    for _, v in pairs(event) do
        if v.MappingTable[1].Key == 'Description' then
            desc = v.MappingTable[1].Value
            break
        end
    end
    return desc
end

function m.get_severity(event)
    if not event then
        return
    end
    local severity
    for _, v in pairs(event) do
        if v.MappingTable[1].Key == 'Severity' then
            severity = v.MappingTable[1].Value == 'Normal' and 'Informational' or v.MappingTable[1].Value
            break
        end
    end
    return severity
end

function m.get_message_id(s, event, major_ver, minor_ver)
    if not event then
        return
    end
    local eventname, state
    for _, v in pairs(event) do
        if v.MappingTable[1].Key == 'EventName' then
            eventname = v.MappingTable[1].Value
        end
        if v.MappingTable[1].Key == 'State' then
            state = v.MappingTable[1].Value
        end
    end
    local message_id = string.format('%s%s.%s.%s.%s', s, EVENTT_REGISTRY_PREFIX, major_ver, minor_ver, eventname)
    return state == 'Asserted' and message_id or cjson.null
end

function m.get_message_args(event)
    if not event then
        return
    end
    local res = cjson.json_object_new_array()
    local args = {}
    for _, v in pairs(event) do
        if v.MappingTable[1].Key == 'MessageArgs' then
            args = cjson.decode(v.MappingTable[1].Value)
            break
        end
    end
    for _, value in ipairs(args) do
        if value ~= '' then
            res[#res + 1] = value
        end
    end
    return res
end

function m.get_severity_status(event)
    if not event then
        return
    end
    local severity = m.get_severity(event)
    if severity == 'Informational' then
        return 'OK'
    elseif  severity == 'Minor' then
        return 'Warning'
    elseif  severity == 'Major' then
        return 'Warning'
    else
        return severity
    end
end

function m.is_valid_entries_id(list)
    return #list ~= 0
end

function m.get_resolved(event)
    if not event then
        return
    end
    for _, v in pairs(event) do
        if v.MappingTable[1].Key == 'State' then
            return v.MappingTable[1].Value == 'Deasserted'
        end
    end
end

return m
