-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at: http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

local lu = require 'luaunit'
local utils = require 'power_mgmt_utils'
local custom_msg = require 'messages.custom'

TestPowerMgmtUtils = {}

function TestPowerMgmtUtils:test_get_version()
    local tmp = {}
    local version = utils.cfgs.get_version(tmp)
    lu.assertEquals(version, nil)

    tmp = {
        cfgs = {
            Firmware1 = {
                version = 'null'
            }
        }
    }
    version = utils.cfgs.get_version(tmp)
    lu.assertEquals(version, 'null')

    tmp = {
        cfgs = {
            Firmware1 = {
                version = 'DC:104 PFC:104'
            }
        }
    }
    version = utils.cfgs.get_version(tmp)
    lu.assertEquals(version, 'DC:104 PFC:104')
    tmp = {
        cfgs = {
            Firmware1 = {
                version = ''
            }
        }
    }
    version = utils.cfgs.get_version(tmp)
    lu.assertEquals(version, '')
end

function TestPowerMgmtUtils:test_get_power_upgrade_flag_and_slot_number()
    local tmp = {
        power_upgrade_slot_numbers = {},
        is_perform_upgrade = true
    }
    -- 空字符串
    local slot_number = ''
    utils.get_power_upgrade_flag_and_slot_number(tmp, slot_number)
    lu.assertEquals(tmp.is_perform_upgrade, true)
    lu.assertEquals(next(tmp.power_upgrade_slot_numbers), nil)
    -- 单个电源
    slot_number = '1'
    utils.get_power_upgrade_flag_and_slot_number(tmp, slot_number)
    lu.assertEquals(tmp.is_perform_upgrade, true)
    lu.assertEquals(next(tmp.power_upgrade_slot_numbers), 1)
    -- 多个电源
    slot_number = '1,2'
    utils.get_power_upgrade_flag_and_slot_number(tmp, slot_number)
    lu.assertEquals(tmp.is_perform_upgrade, true)
    lu.assertEquals(next(tmp.power_upgrade_slot_numbers), 1)
    -- 逗号结尾
    slot_number = '1,'
    utils.get_power_upgrade_flag_and_slot_number(tmp, slot_number)
    lu.assertEquals(tmp.is_perform_upgrade, false)
    lu.assertEquals(next(tmp.power_upgrade_slot_numbers), nil)
    -- 存在空格
    slot_number = '1, 2'
    utils.get_power_upgrade_flag_and_slot_number(tmp, slot_number)
    lu.assertEquals(tmp.is_perform_upgrade, false)
    lu.assertEquals(next(tmp.power_upgrade_slot_numbers), nil)
    -- 单电源前导0
    slot_number = '01'
    utils.get_power_upgrade_flag_and_slot_number(tmp, slot_number)
    lu.assertEquals(tmp.is_perform_upgrade, false)
    lu.assertEquals(next(tmp.power_upgrade_slot_numbers), nil)
    -- 多个电源前导0
    slot_number = '1,02'
    utils.get_power_upgrade_flag_and_slot_number(tmp, slot_number)
    lu.assertEquals(tmp.is_perform_upgrade, false)
    lu.assertEquals(next(tmp.power_upgrade_slot_numbers), nil)
    -- 电源存在小数点
    slot_number = '1.1'
    utils.get_power_upgrade_flag_and_slot_number(tmp, slot_number)
    lu.assertEquals(tmp.is_perform_upgrade, false)
    lu.assertEquals(next(tmp.power_upgrade_slot_numbers), nil)
    -- 电源存在字符
    slot_number = 'a'
    utils.get_power_upgrade_flag_and_slot_number(tmp, slot_number)
    lu.assertEquals(tmp.is_perform_upgrade, false)
    lu.assertEquals(next(tmp.power_upgrade_slot_numbers), nil)
    -- 重复电源槽位
    slot_number = '1,1'
    utils.get_power_upgrade_flag_and_slot_number(tmp, slot_number)
    lu.assertEquals(tmp.is_perform_upgrade, false)
    lu.assertEquals(next(tmp.power_upgrade_slot_numbers), nil)
end

function TestPowerMgmtUtils:test_get_psu_fan_rpm_1()
    local c_psu_object = {}
    c_psu_object.collection = {}
    local psu_obj = {
        get_psu_fan_rpm = function ()
            return 10
        end
    }
    c_psu_object.collection.find = function ()
        return psu_obj
    end
    local res, rsp = pcall(function()
        return utils.get_psu_fan_rpm(utils, c_psu_object)
    end)
    lu.assertEquals(res, true)
    lu.assertEquals(rsp, 10)
end

function TestPowerMgmtUtils:test_get_psu_fan_rpm_2()
    local c_psu_object = {}
    c_psu_object.collection = {}
    c_psu_object.collection.find = function ()
        return nil
    end
    local res, rsp = pcall(function()
        return utils.get_psu_fan_rpm(utils, c_psu_object)
    end)
    lu.assertEquals(res, false)
    lu.assertEquals(rsp, custom_msg.OperationFailed())
end

function TestPowerMgmtUtils:test_get_psu_register_info_1()
    local c_psu_object = {}
    c_psu_object.collection = {}
    local psu_obj = {
        get_psu_register_info = function (cmd, length)
            return 10
        end
    }
    c_psu_object.collection.find = function ()
        return psu_obj
    end
    local res, rsp = pcall(function()
        return utils.get_psu_register_info(utils, c_psu_object)
    end)
    lu.assertEquals(res, true)
    lu.assertEquals(rsp, 10)
end

function TestPowerMgmtUtils:test_get_psu_register_info_2()
    local c_psu_object = {}
    c_psu_object.collection = {}
    c_psu_object.collection.find = function ()
        return nil
    end
    local res, rsp = pcall(function()
        return utils.get_psu_register_info(utils, c_psu_object)
    end)
    lu.assertEquals(res, false)
    lu.assertEquals(rsp, custom_msg.OperationFailed())
end

function TestPowerMgmtUtils:test_dump_psu_blackbox_1()
    local c_psu_object = {}
    c_psu_object.collection = {}
    c_psu_object.collection.find = function(_, func)
        local mock_obj = {path = '/bmc/kepler/Systems/1/PowerMgmt/OnePower_0_010109'}
        return func(mock_obj)
    end
    local obj = {
        path = '/bmc/kepler/Debug/Chassis/1/PowerMgmt/OnePower/OnePowerDebug_0_010109'
    }
    local power_mgmt_mock = {
        get_instance = function ()
        end,
    }
    local res = pcall(function()
        return utils.dump_psu_blackbox(utils, c_psu_object, obj, power_mgmt_mock)
    end)
    lu.assertEquals(res, false)
end

function TestPowerMgmtUtils:test_dump_psu_blackbox_2()
    local c_psu_object = {}
    c_psu_object.collection = {}
    c_psu_object.collection.find = function(_, func)
        local mock_obj = {path = ''}
        return func(mock_obj)
    end
    local obj = {
        path = '/bmc/kepler/Debug/Chassis/1/PowerMgmt/OnePower/OnePowerDebug_0_010109'
    }
    local power_mgmt_mock = {
        get_instance = function ()
        end,
    }
    local res = pcall(function()
        return utils.dump_psu_blackbox(utils, c_psu_object, obj, power_mgmt_mock)
    end)
    lu.assertEquals(res, false)
end


function TestPowerMgmtUtils:test_dump_psu_blackbox_3()
    local c_psu_object = {}
    c_psu_object.collection = {}
    c_psu_object.collection.find = function(_, func)
        local mock_obj = {path = '/bmc/kepler/Systems/1/PowerMgmt/OnePower_0_010109'}
        return func(mock_obj)
    end
    local obj = {
        path = ''
    }
    local power_mgmt_mock = {
        get_instance = function ()
        end,
    }
    local res = pcall(function()
        return utils.dump_psu_blackbox(utils, c_psu_object, obj, power_mgmt_mock)
    end)
    lu.assertEquals(res, false)
end

function TestPowerMgmtUtils:test_dump_psu_blackbox_4()
    local c_psu_object = {}
    c_psu_object.collection = {}
    c_psu_object.collection.find = function(_, func)
        local psu_obj = {ps_id = 1}
        return psu_obj
    end
    local obj = {
        path = ''
    }
    local power_mgmt_mock = {
        get_instance = function ()
            return {dump_psu_blackbox = function (id)
                return 1
            end}
        end,
    }
    local res = pcall(function()
        return utils.dump_psu_blackbox(utils, c_psu_object, obj, power_mgmt_mock)
    end)
    lu.assertEquals(res, true)
end

function TestPowerMgmtUtils:test_sum()
    local t = {1, 2, 3, 4}
    local res = utils.sum(utils, t)
    lu.assertEquals(res, 10)
    t = {1, 's', 2}
    res = utils.sum(utils, t)
    lu.assertEquals(res, false)
end