-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at: http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

local lu = require 'luaunit'

TestCanbusTpsu = {}

function TestCanbusTpsu:test_reset()
    local canbus_tpsu = require 'device_tree.adapters.power_mgmt.protocol.canbus_tpsu'
    local canbus_tpsu_mock = {
        set_signal_data = function ()
            return 0
        end,
        cmd = {
            CANBUS_CMD_RESET = 0x136
        }
    }
    local ok, _ = pcall(function()
        return canbus_tpsu.reset(canbus_tpsu_mock, 'ForceOn')
    end)
    lu.assertEquals(ok, false)

    ok, _ = pcall(function()
        return canbus_tpsu.reset(canbus_tpsu_mock, 'ForceOff')
    end)

    lu.assertTrue(ok)

    ok, _ = pcall(function()
        return canbus_tpsu.reset(canbus_tpsu_mock, 'On')
    end)
    lu.assertTrue(ok)
    lu.assertEquals(canbus_tpsu.cmd.CANBUS_CMD_PSU_SET_TIME, 0x117)
end

-- 场景：CAN 总线返回 A 路电压类型并被正确解析
function TestCanbusTpsu:test_get_main_circuit_input_voltage_type_success()
    local canbus_tpsu = require 'device_tree.adapters.power_mgmt.protocol.canbus_tpsu'
    local mock = {
        get_signal_data = function ()
            return string.char(0) .. string.char(3)
        end,
        tpsu_cmd = canbus_tpsu.tpsu_cmd
    }
    local value = canbus_tpsu.get_main_circuit_input_voltage_type(mock)
    lu.assertEquals(value, 3)
end

-- 场景：CAN 总线未返回 A 路电压类型，函数应抛错
function TestCanbusTpsu:test_get_main_circuit_input_voltage_type_failed()
    local canbus_tpsu = require 'device_tree.adapters.power_mgmt.protocol.canbus_tpsu'
    local mock = {
        get_signal_data = function ()
            return nil
        end,
        tpsu_cmd = canbus_tpsu.tpsu_cmd
    }
    local ok = pcall(function ()
        return canbus_tpsu.get_main_circuit_input_voltage_type(mock)
    end)
    lu.assertFalse(ok)
end

-- 场景：CAN 总线返回 B 路电压类型并被正确解析
function TestCanbusTpsu:test_get_backup_circuit_input_voltage_type_success()
    local canbus_tpsu = require 'device_tree.adapters.power_mgmt.protocol.canbus_tpsu'
    local mock = {
        get_signal_data = function ()
            return string.char(0) .. string.char(1)
        end,
        tpsu_cmd = canbus_tpsu.tpsu_cmd
    }
    local value = canbus_tpsu.get_backup_circuit_input_voltage_type(mock)
    lu.assertEquals(value, 1)
end

-- 场景：CAN 总线未返回 B 路电压类型，函数应抛错
function TestCanbusTpsu:test_get_backup_circuit_input_voltage_type_failed()
    local canbus_tpsu = require 'device_tree.adapters.power_mgmt.protocol.canbus_tpsu'
    local mock = {
        get_signal_data = function ()
            return nil
        end,
        tpsu_cmd = canbus_tpsu.tpsu_cmd
    }
    local ok = pcall(function ()
        return canbus_tpsu.get_backup_circuit_input_voltage_type(mock)
    end)
    lu.assertFalse(ok)
end

-- 场景：CAN 总线返回 A 路电压值，能够换算为整数电压
function TestCanbusTpsu:test_get_main_circuit_input_voltage_success()
    local canbus_tpsu = require 'device_tree.adapters.power_mgmt.protocol.canbus_tpsu'
    local packed_voltage = string.rep(string.char(0), 2) .. string.pack('>I4', 230 * 1024)
    local mock = {
        get_signal_data = function ()
            return packed_voltage
        end,
        tpsu_cmd = canbus_tpsu.tpsu_cmd
    }
    local value = canbus_tpsu.get_main_circuit_input_voltage(mock)
    lu.assertEquals(value, 230)
end

-- 场景：CAN 总线未返回 A 路电压值，函数应抛错
function TestCanbusTpsu:test_get_main_circuit_input_voltage_failed()
    local canbus_tpsu = require 'device_tree.adapters.power_mgmt.protocol.canbus_tpsu'
    local mock = {
        get_signal_data = function ()
            return nil
        end,
        tpsu_cmd = canbus_tpsu.tpsu_cmd
    }
    local ok = pcall(function ()
        return canbus_tpsu.get_main_circuit_input_voltage(mock)
    end)
    lu.assertFalse(ok)
end

-- 场景：CAN 总线返回 B 路电压值，能够换算为整数电压
function TestCanbusTpsu:test_get_backup_circuit_input_voltage_success()
    local canbus_tpsu = require 'device_tree.adapters.power_mgmt.protocol.canbus_tpsu'
    local packed_voltage = string.rep(string.char(0), 2) .. string.pack('>I4', 255 * 1024)
    local mock = {
        get_signal_data = function ()
            return packed_voltage
        end,
        tpsu_cmd = canbus_tpsu.tpsu_cmd
    }
    local value = canbus_tpsu.get_backup_circuit_input_voltage(mock)
    lu.assertEquals(value, 255)
end

-- 场景：CAN 总线未返回 B 路电压值，函数应抛错
function TestCanbusTpsu:test_get_backup_circuit_input_voltage_failed()
    local canbus_tpsu = require 'device_tree.adapters.power_mgmt.protocol.canbus_tpsu'
    local mock = {
        get_signal_data = function ()
            return nil
        end,
        tpsu_cmd = canbus_tpsu.tpsu_cmd
    }
    local ok = pcall(function ()
        return canbus_tpsu.get_backup_circuit_input_voltage(mock)
    end)
    lu.assertFalse(ok)
end