-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at: http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

local device_object = require 'mc.mdb.device_tree.device_object'
local dev_object_manage = require 'mc.mdb.device_tree.dev_object_manage'
local utils = require 'device_tree.adapters.power_mgmt.utils'
local utils_core = require 'utils.core'
local skynet = require 'skynet'
local ctx = require 'mc.context'
local log = require 'mc.logging'
local org_freedesktop_dbus = require 'sd_bus.org_freedesktop_dbus'
local protocol = require 'device_tree.adapters.power_mgmt.protocol'
local match_rule = org_freedesktop_dbus.MatchRule
local debounce = require 'mc.debounce.debounce'
local base_msg = require 'messages.base'
local custom_msg = require 'messages.custom'

local REMOTE_POWER_MONITOR_SIG_PATH<const> = '/bmc/kepler/power_mgmt/psu'
local is_utf8 = utils_core.utf8_validate
local HEALTH_UPDATE_FAIL <const> = 0
local POWER_WATTS_UPDATE_FAIL <const> = 1
local PLUGIN_DEBOUNCE <const> = 2

local psu_slot = device_object('PsuSlot')

local FRU_SOURCE<const> = {
    NONE = 0,   -- 0 -- none
    EEPROM = 1, -- 1 -- 读取FRU的源为EEPROM
    PSU = 2     -- 2 -- 读取FRU的源为PSU
}

local PS_MONITOR<const> = {
    VOUT_STATUS   = 0, -- 输出电压状态
    IOUT_STATUS   = 1, -- 输出电流状态
    VIN_STATUS    = 2, -- 输入电压状态
    TEMPER_STATUS = 3, -- 电源温度传感器的状态
    FAN_STATUS    = 4, -- 电源风扇的状态
    CML_STATUS    = 5, -- 命令的状态
    MFR_STATUS    = 6, -- 厂商自定义的状态

    PS_MONITOR_END = 7
}

local PS_PRESENT = {
    PRESENT = 1,
    NOT_PRESENT = 0,
    DEFAULT_PRESENT = 255
}

local IS_NOT_HEALTH<const> = 0xff
local READ_FAIL_DEBOUNCE_COUNT <const> = 3
local E_OK = nil -- 函数执行成功返回nil
local E_FAILED = '' -- 空错误信息
-- 电源通讯异常初始值
local COMMUNICATION_STATUS_DEFAULT <const> = -1

local function get_pbit(val)
    return 1 << val
end

function psu_slot:ctor()
    self.protocol = false
    self.onepower_obj = false
    self.last_present = PS_PRESENT.DEFAULT_PRESENT
    self.prop_cb = {}
    self.prop_value = {}
    self.scan_status = 0
    self.ps_id = 0
    self.remote_power_monitor_task = false
    self.read_fail_count = 0
    self.remote_powe_signal = false
    self.last_read_interval = 10
    self.read_interval = 100
    self.is_upgrading = false
    self.monitor = false
    self.init_ok = false
    self.monitor_tasks = {}
    self.comm_status = {
        [HEALTH_UPDATE_FAIL] = COMMUNICATION_STATUS_DEFAULT,
        [POWER_WATTS_UPDATE_FAIL] = COMMUNICATION_STATUS_DEFAULT,
        [PLUGIN_DEBOUNCE] = debounce[debounce.DEBOUNCED_CONT_BIN].new(3, 1, 0)
    }
end

function psu_slot:init()
    self.ps_id = self.SlotNumber
    self:declare_method('PowerSupply', 'bmc.dev.power_supply', 'GetReg', function (prop, ...)
        return self.protocol['get_' .. prop](self.protocol, ...)
    end)
    self:declare_method('PowerSupply', 'bmc.dev.power_supply', 'SetReg', function (prop, ...)
        return self.protocol['set_' .. prop](self.protocol, ...)
    end)
end

function psu_slot:is_present()
    return self.Presence == 1
end

function psu_slot:sleep_time(upgrade_interval_time, time)
    if self.is_upgrading then
        self:sleep_ms(upgrade_interval_time)
    else
        self:sleep_ms(time)
    end
end

function psu_slot:set_psu_lockout_protection(lock_status)
    if not self.protocol or not self.protocol.set_psu_lockout_protection then
        return base_msg.ActionNotSupported('set psu lockout protection')
    end
    if not self.monitor or not self.monitor.check_psu_lock_status or not self.monitor:check_psu_lock_status() then
        return custom_msg.OperationFailed()
    end
    local ok, _ = pcall(function ()
        self.protocol:set_psu_lockout_protection(lock_status)
    end)
    if not ok then
        return base_msg.InternalError()
    end
    return E_OK
end

function psu_slot:reset(reset_type)
    if reset_type == "ForceRestart" then
        local ok, err = pcall(function ()
            self:set_psu_lockout_protection(1)
        end)
        if not ok then
            return base_msg.InternalError()
        end
        if err and err.name == base_msg.ActionNotSupportedMessage.Name then
            return base_msg.ActionNotSupported('%ResetType')
        elseif err and err.name == custom_msg.OperationFailedMessage.Name then
            return custom_msg.ResetOperationNotAllowed('%ResetType')
        elseif err then
            return base_msg.InternalError()
        else
            return E_OK
        end
    end
    if not self.protocol or not self.protocol.reset then
        return base_msg.ActionNotSupported('%ResetType')
    end
    local ok, _ = pcall(function ()
        self.protocol:reset(reset_type)
    end)
    if not ok then
        return base_msg.InternalError()
    end
    return E_OK
end

function psu_slot:set_sleep_mode(sleep_mode)
    self.protocol:set_sleep_mode(sleep_mode)
    self:next_tick(function ()
        self:get_dynamic_data('sleep_mode')
    end)
end

function psu_slot:set_work_mode(mode)
    self.protocol:set_power_mode(mode)
    self:next_tick(function ()
        self:get_dynamic_data('work_mode')
    end)
end

function psu_slot:set_retransfer_delay_seconds(seconds_number)
    if not self.protocol or not self.protocol.set_retransfer_delay_seconds then
        return base_msg.ActionNotSupported('%RetransferDelaySeconds')
    end
    local ok, _ = pcall(function ()
        self.protocol:set_retransfer_delay_seconds(seconds_number)
    end)
    if not ok then
        return base_msg.InternalError()
    end
    return E_OK
end

function psu_slot:get_retransfer_delay_seconds()
    if not self.protocol or not self.protocol.get_retransfer_delay_seconds then
        return base_msg.ActionNotSupported('%RetransferDelaySeconds')
    end
    local ok, rsp = pcall(function ()
        return self.protocol:get_retransfer_delay_seconds()
    end)
    if not ok then
        return base_msg.InternalError()
    end
    return rsp
end

function psu_slot:set_psu_fan_rpm(mode)
    self.protocol:set_psu_fan_speed_rpm(mode)
end

function psu_slot:get_psu_fan_rpm()
    return self.protocol:get_psu_fan_speed_rpm()
end

function psu_slot:get_psu_register_info(cmd, length)
    return self.protocol:get_psu_register_info(cmd, length)
end

function psu_slot:get_upgrade_channel()
    if not self.protocol.get_upgrade_channel then
        log:notice('ps%s not support get upgrade channel', self.ps_id)
        return E_FAILED
    end
    return self.protocol:get_upgrade_channel()
end

function psu_slot:set_output_limit_point_watts(output_power_limit_watts)
    if not self.protocol or not self.protocol.set_output_limit_point_watts then
        return base_msg.ActionNotSupported('output limit point watts')
    end
    local ok, _ = pcall(function ()
        self.protocol:set_output_limit_point_watts(output_power_limit_watts)
    end)
    if not ok then
        return base_msg.InternalError()
    end
    self:next_tick(function ()
        self:get_dynamic_data('output_power_limit_watts')
    end)
    return E_OK
end

function psu_slot:change_polling_channel(upgrade_channel)
    if not self.protocol.change_polling_channel then
        log:notice('ps%s not support change polling channel', self.ps_id)
        return E_FAILED
    end
    self.protocol:change_polling_channel(upgrade_channel)
    return E_OK
end

function psu_slot:recover_polling_channel()
    if not self.protocol.recover_polling_channel then
        log:notice('ps%s not support recover polling channel', self.ps_id)
        return E_FAILED
    end
    self.protocol:recover_polling_channel()
    return E_OK
end

function psu_slot:set_power_latch()
    if not self.protocol.set_power_latch then
        log:notice('ps%s not support PowerLatch', self.ps_id)
        return E_FAILED
    end
    self.protocol:set_power_latch()
    return E_OK
end

function psu_slot:set_alarm_latch(switch_status)
    if not self.protocol or not self.protocol.set_alarm_latch then
        return base_msg.ActionNotSupported('%SwitchStatus')
    end
    local ok, _ = pcall(function ()
        self.protocol:set_alarm_latch(switch_status)
    end)
    if not ok then
        return base_msg.InternalError()
    end
    return E_OK
end

function psu_slot:set_vout_debounce_milliseconds(debounce_milliseconds)
    if not self.protocol or not self.protocol.set_vout_debounce_milliseconds then
        return base_msg.ActionNotSupported('VOUTDebounceMilliseconds')
    end
    local ok, _ = pcall(function ()
        self.protocol:set_vout_debounce_milliseconds(debounce_milliseconds)
    end)
    if not ok then
        return base_msg.InternalError()
    end
    self:next_tick(function ()
        self:get_dynamic_data('vout_debounce_milliseconds')
    end)
    return E_OK
end

function psu_slot:set_depth_of_discharge_volts(debounce_milliseconds)
    if not self.protocol or not self.protocol.set_depth_of_discharge_volts then
        return base_msg.ActionNotSupported('DepthOfDischargeVolts')
    end
    local ok, _ = pcall(function ()
        self.protocol:set_depth_of_discharge_volts(debounce_milliseconds)
    end)
    if not ok then
        return base_msg.InternalError()
    end
    self:next_tick(function ()
        self:get_dynamic_data('depth_of_discharge_volts')
    end)
    return E_OK
end

function psu_slot:set_output_voltage(voltage)
    if not self.protocol or not self.protocol.set_output_voltage then
        return base_msg.ActionNotSupported('voltage regulation')
    end
    local ok, _ = pcall(function ()
        self.protocol:set_output_voltage(voltage)
    end)
    if not ok then
        return base_msg.InternalError()
    end
    return E_OK
end

function psu_slot:set_power_supply_circuit(circuit)
    if not self.protocol or not self.protocol.set_power_supply_circuit then
        return base_msg.ActionNotSupported('%PowerSupplyChannel')
    end
    local ok, _ = pcall(function ()
        self.protocol:set_power_supply_circuit(circuit)
    end)
    if not ok then
        return base_msg.InternalError()
    end
    return E_OK
end

function psu_slot:set_power_capacitor_calibrate()
    if not self.protocol or not self.protocol.set_power_capacitor_calibrate then
        return base_msg.ActionNotSupported('power capatitor calibrate')
    end
    local ok, _ = pcall(function ()
        self.protocol:set_power_capacitor_calibrate()
    end)
    if not ok then
        return base_msg.InternalError()
    end
    return E_OK
end

function psu_slot:get_power_capacitor_calibrate_time()
    if not self.protocol or not self.protocol.get_power_capacitor_calibrate_time then
        return base_msg.ActionNotSupported('power capatitor calibrate time')
    end
    local ok, rsp = pcall(function ()
        return self.protocol:get_power_capacitor_calibrate_time()
    end)
    if not ok then
        return base_msg.InternalError()
    end
    return rsp
end

function psu_slot:get_power_capacitor_calibrate()
    if not self.protocol or not self.protocol.get_power_capacitor_calibrate then
        return base_msg.ActionNotSupported('power capatitor calibrate')
    end
    local ok, _ = pcall(function ()
        self.protocol:get_power_capacitor_calibrate()
    end)
    if not ok then
        return base_msg.InternalError()
    end
    return E_OK
end

--turbo能力使用设置接口
function psu_slot:set_power_capacitor_enable(enable)
    if not self.protocol or not self.protocol.set_power_capacitor_enable then
        return base_msg.ActionNotSupported('%SetPowerCapacitorEnable')
    end
    local ok, _ = pcall(function ()
        self.protocol:set_power_capacitor_enable(enable)
    end)
    if not ok then
        return base_msg.InternalError()
    end
    return E_OK
end

function psu_slot:get_main_circuit_input_voltage_type()
    if not self.protocol or not self.protocol.get_main_circuit_input_voltage_type then
        return base_msg.ActionNotSupported('get main circuit input voltage type')
    end
    local ok, rsp = pcall(function ()
        return self.protocol:get_main_circuit_input_voltage_type()
    end)
    if not ok then
        return base_msg.InternalError()
    end
    return E_OK, rsp
end

function psu_slot:get_backup_circuit_input_voltage_type()
    if not self.protocol or not self.protocol.get_backup_circuit_input_voltage_type then
        return base_msg.ActionNotSupported('get backup circuit input voltage type')
    end
    local ok, rsp = pcall(function ()
        return self.protocol:get_backup_circuit_input_voltage_type()
    end)
    if not ok then
        return base_msg.InternalError()
    end
    return E_OK, rsp
end

function psu_slot:get_main_circuit_input_voltage()
    if not self.protocol or not self.protocol.get_main_circuit_input_voltage then
        return base_msg.ActionNotSupported('get main circuit input voltage')
    end
    local ok, rsp = pcall(function ()
        return self.protocol:get_main_circuit_input_voltage()
    end)
    if not ok then
        return base_msg.InternalError()
    end
    return E_OK, rsp
end

function psu_slot:get_backup_circuit_input_voltage()
    if not self.protocol or not self.protocol.get_backup_circuit_input_voltage then
        return base_msg.ActionNotSupported('get backup circuit input voltage')
    end
    local ok, rsp = pcall(function ()
        return self.protocol:get_backup_circuit_input_voltage()
    end)
    if not ok then
        return base_msg.InternalError()
    end
    return E_OK, rsp
end

function psu_slot:get_life_prediction_level()
    if not self.protocol or not self.protocol.get_life_prediction_level then
        return base_msg.ActionNotSupported('life prediction')
    end
    local ok, rsp = pcall(function ()
        return self.protocol:get_life_prediction_level()
    end)
    if not ok then
        return base_msg.InternalError()
    end
    return E_OK, rsp
end

function psu_slot:get_life_prediction_count()
    if not self.protocol or not self.protocol.get_life_prediction_count then
        return base_msg.ActionNotSupported('life prediction')
    end
    local ok, index, count = pcall(function ()
        return self.protocol:get_life_prediction_count()
    end)
    if not ok then
        return base_msg.InternalError()
    end
    return E_OK, index, count
end

function psu_slot:set_life_prediction_count_index(dod_level, vout_level, current_level)
    if not self.protocol or not self.protocol.set_life_prediction_count_index then
        return base_msg.ActionNotSupported('life prediction')
    end
    local ok, _ = pcall(function ()
        self.protocol:set_life_prediction_count_index(dod_level, vout_level, current_level)
    end)
    if not ok then
        return base_msg.InternalError()
    end
    return E_OK
end

function psu_slot:stop_monitor_tasks()
    local ts = self.monitor_tasks
    self.monitor_tasks = {}
    for _, task in pairs(ts) do
        task:stop()
    end
end

function psu_slot:psm_monitor_stop()
    log:notice('ps%d power monitor stop', self.ps_id)
    if self.monitor then
        self.monitor:stop_monitor_tasks()
    end
end

function psu_slot:power_upgrade(upgrade_path, upgrade_process, para_tab)
    self:psm_monitor_stop()
    self:dal_refresh_property('Health', IS_NOT_HEALTH)
    local ok, resp = pcall(function()
        return self.protocol:upgrade(upgrade_path, upgrade_process, para_tab)
    end)
    self:sleep_ms(self.read_interval * 10)
    self.monitor:fetch_power_supply_info()
    self.monitor:update_health_event()
    self:power_monitor_start()
    self:sleep_ms(self.read_interval * 20)
    if not ok then
        error(resp)
    end
    return resp
end

function psu_slot:get_fru_data()
    local fru_data = self.protocol:get_fru_data()
    self:dal_refresh_property('fru_data', fru_data)
    return fru_data
end

function psu_slot:dal_refresh_property(prop, value)
    self.prop_value[prop] = value
    if not self.prop_cb[prop] then
        return
    end
    for _, cb in pairs(self.prop_cb[prop]) do
        if self.init_ok then
            self:next_tick(cb, value)
        else
            -- init阶段需要把属性刷新完才允许分发到app侧
            cb(value)
        end
    end
end

function psu_slot:get_protocol(...)
    local protocol_name, req_protocol, monitor = protocol(...)
    if req_protocol then
        self.protocol = req_protocol.new(self, protocol_name)
        self:dal_refresh_property('protocol', protocol_name)
    end
    if monitor then
        self.monitor = monitor(self)
    else
        log:warn('ps%d cannot find monitor, protocol_name = %s', self.ps_id, protocol_name)
    end
end

local PS_TRY_CNT = 3 -- 重试次数
local PS_TRY_DELAY = 100 -- 重试时间间隔 ms
function psu_slot:get_dynamic_data(prop, retry_times)
    if not self.protocol or not self.protocol['get_' .. prop] then
        return
    end
    retry_times = retry_times or READ_FAIL_DEBOUNCE_COUNT
    local ok, resp
    for _ = 1, retry_times do
        ok, resp = pcall(self.protocol['get_' .. prop], self.protocol)
        if ok then
            self:dal_refresh_property(prop, resp)
            return E_OK, resp
        end
        self:sleep_ms(PS_TRY_DELAY)
    end
    return E_FAILED, resp
end

function psu_slot:get_power_supply_info(prop)
    local ret, data = self:get_dynamic_data(prop, PS_TRY_CNT)
    if ret == E_OK then
        return data
    end
    log:error('ps%s read data(%s) failed. resp: %s', self.ps_id, prop, data)
end

function psu_slot:fetch_power_supply_info(physical_interface)
    local ok, resp = pcall(function ()
        self:get_protocol(physical_interface)
    end)
    if not ok then
        self.protocol = false
        self.monitor = false
        log:error('ps%s no protocol(%s). resp: %s', self.ps_id, physical_interface, resp)
        return
    end
    if self.monitor then
        self.monitor:fetch_power_supply_info()
    end
    local manufacturer = self.prop_value['manufacturer'] or self.prop_value['Manufacturer']
    local model = self.prop_value['model'] or self.prop_value['Model']
    if manufacturer and model then
        -- 支持厂商自定义实现命令字
        ok = pcall(function ()
            self:get_protocol(physical_interface, manufacturer, model)
        end)
        if ok and self.monitor then
            self.monitor:fetch_power_supply_info()
        end
    end
end

function psu_slot:clear_property_fetch_task()
    self.prop_cb = {}
    self.prop_value = {}
end

function psu_slot:fetch_property(prop, cb)
    if not self.prop_cb[prop] then
        self.prop_cb[prop] = {}
    end
    table.insert(self.prop_cb[prop], cb)
end

function psu_slot:power_monitor_start()
    if self.monitor then
        log:notice('ps%d power monitor start', self.ps_id)
        self.monitor:power_monitor_start()
    else
        log:warn('ps%d power monitor start failed', self.ps_id)
    end
end

function psu_slot:get_black_box_data()
    if self.monitor and self.monitor.get_event_log then
        return self.monitor:get_event_log()
    end
    return self.protocol:get_event_log()
end

function psu_slot:presence_detect()
    self:next_tick(function ()
        while true do
            if self.Presence == PS_PRESENT.DEFAULT_PRESENT then
                goto continue
            end
            if self.last_present == PS_PRESENT.DEFAULT_PRESENT then
                self.last_present = self.Presence
                goto continue
            end
            if self:is_present() and self.last_present == PS_PRESENT.NOT_PRESENT then
                utils.operation_log(nil, 'Plug in power supply %s by user', self.ps_id)
            elseif not self:is_present() and self.last_present == PS_PRESENT.PRESENT then
                utils.operation_log(nil, 'Plug out power supply %s by user', self.ps_id)
            end
            self.last_present = self.Presence
            ::continue::
            self:sleep_ms(1000)
        end
    end)
end

function psu_slot:start()
    self:presence_detect()
end

return psu_slot