-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at: http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

local device_object = require 'mc.mdb.device_tree.device_object'
local dev_object_manage = require 'mc.mdb.device_tree.dev_object_manage'
local utils = require 'device_tree.adapters.power_mgmt.utils'
local utils_core = require 'utils.core'
local skynet = require 'skynet'
local ctx = require 'mc.context'
local log = require 'mc.logging'
local org_freedesktop_dbus = require 'sd_bus.org_freedesktop_dbus'
local protocol = require 'device_tree.adapters.power_mgmt.protocol'
local match_rule = org_freedesktop_dbus.MatchRule
local base_msg = require 'messages.base'
local enums = require 'macros.power_mgmt_enums'

local is_utf8 = utils_core.utf8_validate

local psu_slot = device_object('PsuSlot')

local FRU_SOURCE<const> = {
    NONE = 0,   -- 0 -- none
    EEPROM = 1, -- 1 -- 读取FRU的源为EEPROM
    PSU = 2     -- 2 -- 读取FRU的源为PSU
}

local PS_MONITOR<const> = {
    VOUT_STATUS   = 0, -- 输出电压状态
    IOUT_STATUS   = 1, -- 输出电流状态
    VIN_STATUS    = 2, -- 输入电压状态
    TEMPER_STATUS = 3, -- 电源温度传感器的状态
    FAN_STATUS    = 4, -- 电源风扇的状态
    CML_STATUS    = 5, -- 命令的状态
    MFR_STATUS    = 6, -- 厂商自定义的状态

    PS_MONITOR_END = 7
}

local PS_PRESENT = {
    PRESENT = 1,
    NOT_PRESENT = 0,
    DEFAULT_PRESENT = 255
}

local TIME_SERVICE_INTERVAL<const> = 1000 * 60 * 5
local IS_NOT_HEALTH<const> = 0xff
local READ_FAIL_DEBOUNCE_COUNT <const> = 3
local E_OK = nil -- 函数执行成功返回nil
local E_FAILED = '' -- 空错误信息

local function get_pbit(val)
    return 1 << val
end

-- HPC canbus
function psu_slot:ctor()
    self.protocol = false
    self.onepower_obj = false
    self.last_present = PS_PRESENT.DEFAULT_PRESENT
    self.prop_cb = {}
    self.prop_value = {}
    self.scan_status = 0
    self.ps_id = 0
    self.read_fail_count = 0
    self.last_read_interval = 10
    self.read_interval = 1000
    self.monitor = false
    self.init_ok = false
    self.is_upgrading = false
    self.monitor_tasks = {}
end

function psu_slot:init()
    self.ps_id = self.SlotNumber
    self:declare_method('PowerSupply', 'bmc.dev.power_supply', 'GetReg', function (prop, ...)
        return self.protocol['get_' .. prop](self.protocol, ...)
    end)
    self:declare_method('PowerSupply', 'bmc.dev.power_supply', 'SetReg', function (prop, ...)
        return self.protocol['set_' .. prop](self.protocol, ...)
    end)
end

function psu_slot:is_present()
    return self.Presence == 1
end

function psu_slot:sleep_time(upgrade_interval_time, time)
    if self.is_upgrading then
        self:sleep_ms(upgrade_interval_time)
    else
        self:sleep_ms(time)
    end
end

function psu_slot:set_psu_fan_rpm(mode)
    -- 不支持风扇调速，直接返回
    return
end

function psu_slot:set_sleep_mode(sleep_mode)
    -- 不支持休眠模式，直接返回
    error(base_msg.ActionNotSupported('%SetSleepMode'))
end

function psu_slot:set_work_mode(mode)
    -- 不支持主备切换，直接返回
    return
end

function psu_slot:stop_monitor_tasks()
    local ts = self.monitor_tasks
    self.monitor_tasks = {}
    for _, task in pairs(ts) do
        task:stop()
    end
end

function psu_slot:psm_monitor_stop()
    log:notice('ps%d power monitor stop', self.ps_id)
    self:stop_monitor_tasks()
end

function psu_slot:power_upgrade(upgrade_path, upgrade_process, para_tab)
    self:psm_monitor_stop()
    self:dal_refresh_property('Health', IS_NOT_HEALTH)
    local ok, resp = pcall(function()
        return self.protocol:upgrade(upgrade_path, upgrade_process, para_tab)
    end)
    self:sleep_ms(self.read_interval * 10)
    self:get_firmware_version()
    self:refresh_batch_query_info()
    self:power_monitor_start()
    self:sleep_ms(self.read_interval * 20)
    if not ok then
        error(resp)
    end
    return resp
end

function psu_slot:get_fru_data(value)
    local fru_data = self.protocol:get_fru_data(value)
    self:dal_refresh_property('fru_data', fru_data)
    return fru_data
end

function psu_slot:dal_refresh_property(prop, value)
    self.prop_value[prop] = value
    if not self.prop_cb[prop] then
        return
    end
    for _, cb in pairs(self.prop_cb[prop]) do
        if self.init_ok then
            self:next_tick(cb, value)
        else
            -- init阶段需要把属性刷新完才允许分发到app侧
            cb(value)
        end
    end
end

function psu_slot:get_protocol(...)
    local protocol_name, req_protocol = protocol(...)
    if req_protocol then
        self.protocol = req_protocol.new(self, protocol_name)
        self:dal_refresh_property('protocol', protocol_name)
    end
end

function psu_slot:get_power_supply_info()
    local ok, resp = pcall(function ()
        return self.protocol:update_elabel_info()
    end)

    if not ok then
        log:info('ps%d update_elabel_info failed, response = %s', self.ps_id, resp)
        return resp
    end
    for key, value in pairs(resp) do
        self:dal_refresh_property(key, value)
    end
    log:notice('ps%s read elabel_info successfully.', self.ps_id)
end

function psu_slot:get_firmware_version()
    local ok, resp = pcall(function ()
        return self.protocol:refresh_basic_info()
    end)

    if not ok then
        log:info('ps%d refresh_basic_info failed, response = %s', self.ps_id, resp)
        return resp
    end
    self:dal_refresh_property('firmware_version', resp)
    log:notice('ps%s read firmware_version successfully.', self.ps_id)
end

function psu_slot:fetch_power_supply_info(physical_interface)
    local ok, resp = pcall(function ()
        self:get_protocol(physical_interface)
    end)
    if not ok then
        self.protocol = nil
        log:error('ps%s no protocol(%s). resp: %s', self.ps_id, physical_interface, resp)
        return
    end
    resp = self:get_power_supply_info()
    if resp ~= E_OK then
        log:warn('ps%d update_elabel_info failed, response = %s', self.ps_id, resp)
        self:next_tick(function ()
            while self:get_power_supply_info() ~= E_OK do
                self:sleep_time(10000, 1000)
            end
        end)
    end
    resp = self:get_firmware_version()
    if resp ~= E_OK then
        log:warn('ps%d refresh_basic_info failed, response = %s', self.ps_id, resp)
        self:next_tick(function ()
            while self:get_firmware_version() ~= E_OK do
                self:sleep_time(10000, 1000)
            end
        end)
    end
end

function psu_slot:clear_property_fetch_task()
    self.prop_cb = {}
    self.prop_value = {}
end

function psu_slot:fetch_property(prop, cb)
    if not self.prop_cb[prop] then
        self.prop_cb[prop] = {}
    end
    table.insert(self.prop_cb[prop], cb)
end

function psu_slot:update_health_event(alarm_status)
    if not alarm_status then
        return
    end
    local event = 0
    -- 输出过压
    local output_over_voltage = alarm_status & get_pbit(8)
    if output_over_voltage > 0 then
        -- 根据canbus协议AlarmStatus属性bit8为1，代表输出过压。但是根据电源告警，若输出过压，OutputVoltageFault bit7需要置为1
        output_over_voltage = get_pbit(7)
    end

    -- 输入欠压
    local input_under_voltage = alarm_status & get_pbit(1)
    if input_under_voltage > 0 then
        -- 根据canbus协议AlarmStatus属性bit1为1，代表输入欠压。但是根据电源告警，若输入欠压，InputVoltageFault bit4需要置为1
        input_under_voltage = get_pbit(4)
        event = event | get_pbit(4)
    end
    -- 输入过压
    local input_over_voltage = alarm_status & get_pbit(0)
    if input_over_voltage > 0 then
        -- 根据canbus协议AlarmStatus属性bit0为1，代表输入过压。但是根据电源告警，若输入过压，InputVoltageFault bit7需要置为1
        input_over_voltage = get_pbit(7)
        event = event | get_pbit(4)
    end
    local input_voltage_fault = input_under_voltage | input_over_voltage

    -- 过温
    local over_temper_fault = alarm_status & get_pbit(16)
    if over_temper_fault > 0 then
        -- 根据canbus协议AlarmStatus属性bit16为1，代表过温。但是根据电源告警，若过温，OverTemperature bit7需要置为1
        over_temper_fault = get_pbit(7)
        event = event | get_pbit(3)
    end

    -- 输出过流
    local output_over_current = alarm_status & get_pbit(9)
    if output_over_current > 0 then
        -- 根据canbus协议AlarmStatus属性bit16为1，代表输出过流。但是根据电源告警，若输出过流，OutputCurrentFault bit7需要置为1
        output_over_current = get_pbit(7)
    end
    self:dal_refresh_property('health_event', {
        event = event,
        output_voltage_fault = output_over_voltage,
        output_current_fault = output_over_current,
        temper_fault = over_temper_fault,
        input_voltage_fault = input_voltage_fault
    })
    self:dal_refresh_property('health', event)
    return E_OK
end

function psu_slot:health_check(alarm_status)
    self:update_health_event(alarm_status)
    self.scan_status = self.scan_status | get_pbit(0)
    self:dal_refresh_property('scan_status', self.scan_status)
end

function psu_slot:refresh_batch_query_info()
    -- 0x40 批量查询信息
    local batch_query_info = self.protocol['refresh_batch_query_info']
    local batch_query_info_ok, batch_query_info_resp = pcall(batch_query_info, self.protocol)
    if batch_query_info_ok then
        log:info('[power_mgmt]ps%d refresh_batch_query_info ok', self.ps_id)
        for prop, value in pairs(batch_query_info_resp) do
            self:dal_refresh_property(prop, value)
        end
        self:health_check(batch_query_info_resp.AlarmStatus)
    else
        log:info('[power_mgmt]ps%s refresh_batch_query_info failed, response = %s', self.ps_id,
                    batch_query_info_resp)
    end
end

local PS_TRY_CNT = 30 -- 重试次数
local PS_TRY_DELAY = 100 -- 重试时间间隔 ms
function psu_slot:get_dynamic_data(prop, retry_times)
    if not self.protocol or not self.protocol['get_' .. prop] then
        return
    end
    retry_times = retry_times or READ_FAIL_DEBOUNCE_COUNT
    local ok, resp
    for _ = 1, retry_times do
        ok, resp = pcall(self.protocol['get_' .. prop], self.protocol)
        if ok then
            self:dal_refresh_property(prop, resp)
            return E_OK, resp
        end
        self:sleep_ms(PS_TRY_DELAY)
    end
    return E_FAILED, resp
end

function psu_slot:time_service_task()
    while true do
        self.protocol:time_service()
        self:sleep_ms(TIME_SERVICE_INTERVAL)
    end
end

function psu_slot:update_communication_status()
    local read_fail_count = 0
    local ok, _
    while true do
        if not self:is_present() then
            goto continue
        end
        ok, _ = pcall(function()
            return self.protocol:can_get_canbus_basic_info()
        end)
        if not ok then
            if read_fail_count < 4 then
                read_fail_count = read_fail_count + 1
            else
                self:dal_refresh_property('communication_status', 1)
            end
        else
            self:dal_refresh_property('communication_status', 0)
        end
        ::continue::
        self:sleep_time(15000, 15000)
    end
end

function psu_slot:power_monitor_entry()
    table.insert(self.monitor_tasks, self:next_tick(function()
        while true do
            self:refresh_batch_query_info()
            self.scan_status = self.scan_status | get_pbit(2) | get_pbit(1)
            self:dal_refresh_property('scan_status', self.scan_status)
            self:sleep_time(1000, 1000)
        end
    end))
    table.insert(self.monitor_tasks, self:next_tick(
        function()
            self:time_service_task()
        end
    ))
    table.insert(self.monitor_tasks, self:next_tick(function()
        self:update_communication_status()
    end))
    self.scan_status = self.scan_status | get_pbit(3)
    self:dal_refresh_property('scan_status', self.scan_status)
    self:get_dynamic_data('rate', PS_TRY_CNT)
end

function psu_slot:power_monitor_start()
    local ok, uptime = pcall(utils_core.get_bmc_uptime)
    if not ok then
        log:error('Get bmc uptime failed')
    else
        log:notice('Current bmc uptime is %s', uptime)
        local wait_time = enums.UPTIME - uptime
        if wait_time > 0 then
            skynet.sleep(wait_time * 100) -- BMC启动前UPTIME时间内不进行电源数据采集
        end
    end
    log:notice('ps%d power monitor start', self.ps_id)
    self:power_monitor_entry()
end

function psu_slot:get_black_box_data()
    return self.protocol:get_event_log()
end

function psu_slot:presence_detect()
    self:next_tick(function ()
        while true do
            if self.Presence == PS_PRESENT.DEFAULT_PRESENT then
                goto continue
            end
            if self.last_present == PS_PRESENT.DEFAULT_PRESENT then
                self.last_present = self.Presence
                goto continue
            end
            if self:is_present() and self.last_present == PS_PRESENT.NOT_PRESENT then
                utils.operation_log(nil, 'Plug in power supply %s by user', self.ps_id)
            elseif not self:is_present() and self.last_present == PS_PRESENT.PRESENT then
                utils.operation_log(nil, 'Plug out power supply %s by user', self.ps_id)
            end
            self.last_present = self.Presence
            ::continue::
            self:sleep_ms(1000)
        end
    end)
end

function psu_slot:start()
    self:presence_detect()
end

return psu_slot