-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at: http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

local luaunit = require 'luaunit'

-- Mock skynet module
local mock_skynet = {
    register_protocol = function(params) end,
    pack = function(...) return {...} end,
    unpack = function(data) return table.unpack(data) end,
    send = function(service, protocol, data) end
}

-- Mock log module
local mock_log = {
    notice = function(self, fmt, ...) end,
    info = function(self, fmt, ...) end,
    error = function(self, fmt, ...) end
}

-- Mock ncsi_cmd module
local mock_ncsi_cmd = {}

-- 存储原始模块
local original_modules = {}

-- 测试套件
TestNCSICore = {}

-- 常量定义
local NCSI_FALSE = 1
local NCSI_TRUE = 0

function TestNCSICore:setUp()
    -- 保存原始模块
    if package.loaded['skynet'] then
        original_modules.skynet = package.loaded['skynet']
    end
    if package.loaded['mc.logging'] then
        original_modules.logging = package.loaded['mc.logging']
    end
    if package.loaded['ncsi.ncsi_protocol.ncsi_cmd'] then
        original_modules.ncsi_cmd = package.loaded['ncsi.ncsi_protocol.ncsi_cmd']
    end

    -- 替换为mock模块
    package.loaded['skynet'] = mock_skynet
    package.loaded['mc.logging'] = mock_log
    package.loaded['ncsi.ncsi_protocol.ncsi_cmd'] = mock_ncsi_cmd

    -- 重新加载ncsi_core模块
    package.loaded['ncsi.ncsi_core'] = nil
    self.ncsi_core = require 'ncsi.ncsi_core'

    -- 重置mock函数
    self:resetMocks()

    -- 记录调用信息
    self.call_log = {}
end

function TestNCSICore:tearDown()
    -- 恢复原始模块
    for name, module in pairs(original_modules) do
        if name == 'skynet' then
            package.loaded['skynet'] = module
        elseif name == 'logging' then
            package.loaded['mc.logging'] = module
        elseif name == 'ncsi_cmd' then
            package.loaded['ncsi.ncsi_protocol.ncsi_cmd'] = module
        end
    end

    -- 清理ncsi_core模块
    package.loaded['ncsi.ncsi_core'] = nil
end

function TestNCSICore:resetMocks()
    -- 重置所有mock函数
    mock_ncsi_cmd.ncsi_clear_initial_state = function(package_id, channel_id, eth_name)
        table.insert(self.call_log, {'ncsi_clear_initial_state', package_id, channel_id, eth_name})
        return NCSI_TRUE
    end

    mock_ncsi_cmd.ncsi_select_package = function(package_id, eth_name)
        table.insert(self.call_log, {'ncsi_select_package', package_id, eth_name})
        return NCSI_TRUE
    end

    mock_ncsi_cmd.ncsi_disable_channel = function(package_id, channel_id, eth_name)
        table.insert(self.call_log, {'ncsi_disable_channel', package_id, channel_id, eth_name})
        return NCSI_TRUE
    end

    mock_ncsi_cmd.ncsi_enable_channel = function(package_id, channel_id, eth_name)
        table.insert(self.call_log, {'ncsi_enable_channel', package_id, channel_id, eth_name})
        return NCSI_TRUE
    end

    mock_ncsi_cmd.ncsi_enable_brdcast_filter = function(package_id, channel_id, eth_name)
        table.insert(self.call_log, {'ncsi_enable_brdcast_filter', package_id, channel_id, eth_name})
        return NCSI_TRUE
    end

    mock_ncsi_cmd.ncsi_disable_brdcast_filter = function(package_id, channel_id, eth_name)
        table.insert(self.call_log, {'ncsi_disable_brdcast_filter', package_id, channel_id, eth_name})
        return NCSI_TRUE
    end

    mock_ncsi_cmd.ncsi_disable_multi_filter = function(package_id, channel_id, eth_name)
        table.insert(self.call_log, {'ncsi_disable_multi_filter', package_id, channel_id, eth_name})
        return NCSI_TRUE
    end

    mock_ncsi_cmd.ncsi_enable_multi_filter = function(package_id, channel_id, eth_name)
        table.insert(self.call_log, {'ncsi_enable_multi_filter', package_id, channel_id, eth_name})
        return NCSI_TRUE
    end

    mock_ncsi_cmd.ncsi_enable_channelTX = function(package_id, channel_id, eth_name)
        table.insert(self.call_log, {'ncsi_enable_channelTX', package_id, channel_id, eth_name})
        return NCSI_TRUE
    end

    mock_ncsi_cmd.ncsi_disable_channelTX = function(package_id, channel_id, eth_name)
        table.insert(self.call_log, {'ncsi_disable_channelTX', package_id, channel_id, eth_name})
        return NCSI_TRUE
    end

    mock_ncsi_cmd.ncsi_set_vlan_filter = function(package_id, channel_id, vlan_filter, vlan_state, vlan_id, eth_name)
        table.insert(self.call_log, {'ncsi_set_vlan_filter', package_id, channel_id,
            vlan_filter, vlan_state, vlan_id, eth_name})
        return NCSI_TRUE
    end

    mock_ncsi_cmd.ncsi_get_link_status = function(package_id, channel_id, eth_name, eth_id)
        table.insert(self.call_log, {'ncsi_get_link_status', package_id, channel_id, eth_name, eth_id})
        return NCSI_TRUE, {link_up = true, speed = 1000}
    end

    mock_ncsi_cmd.ncsi_common_init = function()
        table.insert(self.call_log, {'ncsi_common_init'})
    end

    mock_ncsi_cmd.ncsi_paramter_init = function(channel_id, eth_name, state)
        table.insert(self.call_log, {'ncsi_paramter_init', channel_id, eth_name, state})
        return NCSI_TRUE, 0
    end

    mock_ncsi_cmd.ncsi_receive_packet = function(fd, eth_name)
        table.insert(self.call_log, {'ncsi_receive_packet', fd, eth_name})
        return NCSI_TRUE, 0, "test_data"
    end

    mock_ncsi_cmd.ncsi_eth_mac_init = function(mac_str)
        table.insert(self.call_log, {'ncsi_eth_mac_init', mac_str})
        return NCSI_TRUE
    end

    mock_ncsi_cmd.ncsi_set_lldp_status = function(eth_name, package_id, channel_id, status)
        table.insert(self.call_log, {'ncsi_set_lldp_status', eth_name, package_id, channel_id, status})
        return NCSI_TRUE
    end

    mock_ncsi_cmd.ncsi_enable_vlan_req = function(package_id, channel_id, eth_name, vlan_mode)
        table.insert(self.call_log, {'ncsi_enable_vlan_req', package_id, channel_id, eth_name, vlan_mode})
        return NCSI_TRUE
    end

    mock_ncsi_cmd.ncsi_disable_vlan_req = function(package_id, channel_id, eth_name)
        table.insert(self.call_log, {'ncsi_disable_vlan_req', package_id, channel_id, eth_name})
        return NCSI_TRUE
    end

    mock_ncsi_cmd.ncsi_get_log = function(package_id, eth_name, log_dir, time)
        table.insert(self.call_log, {'ncsi_get_log', package_id, eth_name, log_dir, time})
        return NCSI_TRUE
    end

    mock_ncsi_cmd.ncsi_get_log_new = function(package_id, eth_name, log_dir, time)
        table.insert(self.call_log, {'ncsi_get_log_new', package_id, eth_name, log_dir, time})
        return NCSI_TRUE
    end

    mock_ncsi_cmd.ncsi_get_blackbox = function(package_id, eth_name, log_name)
        table.insert(self.call_log, {'ncsi_get_blackbox', package_id, eth_name, log_name})
        return NCSI_TRUE
    end

    mock_ncsi_cmd.get_ncsi_channel_cnt = function()
        table.insert(self.call_log, {'get_ncsi_channel_cnt'})
        return 4
    end
end

-- 测试 ncsi_clear_initial_state 函数
function TestNCSICore:test_ncsi_clear_initial_state_success()
    local result = self.ncsi_core.ncsi_clear_initial_state(0, 1, "eth0")

    luaunit.assertTrue(result)
    luaunit.assertEquals(#self.call_log, 1)
    luaunit.assertEquals(self.call_log[1], {'ncsi_clear_initial_state', 0, 1, "eth0"})
end

function TestNCSICore:test_ncsi_clear_initial_state_failure()
    mock_ncsi_cmd.ncsi_clear_initial_state = function(package_id, channel_id, eth_name)
        return NCSI_FALSE
    end

    local result = self.ncsi_core.ncsi_clear_initial_state(0, 1, "eth0")

    luaunit.assertFalse(result)
end

-- 测试 ncsi_select_package 函数
function TestNCSICore:test_ncsi_select_package_success()
    local result = self.ncsi_core.ncsi_select_package(0, "eth0")

    luaunit.assertTrue(result)
    luaunit.assertEquals(#self.call_log, 1)
    luaunit.assertEquals(self.call_log[1], {'ncsi_select_package', 0, "eth0"})
end

function TestNCSICore:test_ncsi_select_package_failure()
    mock_ncsi_cmd.ncsi_select_package = function(package_id, eth_name)
        return NCSI_FALSE
    end

    local result = self.ncsi_core.ncsi_select_package(0, "eth0")

    luaunit.assertFalse(result)
end

-- 测试 ncsi_disable_channel 函数
function TestNCSICore:test_ncsi_disable_channel_success()
    local result = self.ncsi_core.ncsi_disable_channel(0, 1, "eth0")

    luaunit.assertTrue(result)
    luaunit.assertEquals(#self.call_log, 1)
    luaunit.assertEquals(self.call_log[1], {'ncsi_disable_channel', 0, 1, "eth0"})
end

function TestNCSICore:test_ncsi_disable_channel_failure()
    mock_ncsi_cmd.ncsi_disable_channel = function(package_id, channel_id, eth_name)
        return NCSI_FALSE
    end

    local result = self.ncsi_core.ncsi_disable_channel(0, 1, "eth0")

    luaunit.assertFalse(result)
end

-- 测试 ncsi_enable_channel 函数
function TestNCSICore:test_ncsi_enable_channel_success()
    local result = self.ncsi_core.ncsi_enable_channel(0, 1, "eth0")

    luaunit.assertTrue(result)
    luaunit.assertEquals(#self.call_log, 1)
    luaunit.assertEquals(self.call_log[1], {'ncsi_enable_channel', 0, 1, "eth0"})
end

function TestNCSICore:test_ncsi_enable_channel_failure()
    mock_ncsi_cmd.ncsi_enable_channel = function(package_id, channel_id, eth_name)
        return NCSI_FALSE
    end

    local result = self.ncsi_core.ncsi_enable_channel(0, 1, "eth0")

    luaunit.assertFalse(result)
end

-- 测试 ncsi_enable_brdcast_filter 函数
function TestNCSICore:test_ncsi_enable_brdcast_filter_success()
    local result = self.ncsi_core.ncsi_enable_brdcast_filter(0, 1, "eth0")

    luaunit.assertTrue(result)
    luaunit.assertEquals(#self.call_log, 1)
    luaunit.assertEquals(self.call_log[1], {'ncsi_enable_brdcast_filter', 0, 1, "eth0"})
end

function TestNCSICore:test_ncsi_enable_brdcast_filter_failure()
    mock_ncsi_cmd.ncsi_enable_brdcast_filter = function(package_id, channel_id, eth_name)
        return NCSI_FALSE
    end

    local result = self.ncsi_core.ncsi_enable_brdcast_filter(0, 1, "eth0")

    luaunit.assertFalse(result)
end

-- 测试 ncsi_disable_brdcast_filter 函数
function TestNCSICore:test_ncsi_disable_brdcast_filter_success()
    local result = self.ncsi_core.ncsi_disable_brdcast_filter(0, 1, "eth0")

    luaunit.assertTrue(result)
    luaunit.assertEquals(#self.call_log, 1)
    luaunit.assertEquals(self.call_log[1], {'ncsi_disable_brdcast_filter', 0, 1, "eth0"})
end

function TestNCSICore:test_ncsi_disable_brdcast_filter_failure()
    mock_ncsi_cmd.ncsi_disable_brdcast_filter = function(package_id, channel_id, eth_name)
        return NCSI_FALSE
    end

    local result = self.ncsi_core.ncsi_disable_brdcast_filter(0, 1, "eth0")

    luaunit.assertFalse(result)
end

-- 测试 ncsi_disable_multi_filter 函数
function TestNCSICore:test_ncsi_disable_multi_filter_success()
    local result = self.ncsi_core.ncsi_disable_multi_filter(0, 1, "eth0")

    luaunit.assertTrue(result)
    luaunit.assertEquals(#self.call_log, 1)
    luaunit.assertEquals(self.call_log[1], {'ncsi_disable_multi_filter', 0, 1, "eth0"})
end

function TestNCSICore:test_ncsi_disable_multi_filter_failure()
    mock_ncsi_cmd.ncsi_disable_multi_filter = function(package_id, channel_id, eth_name)
        return NCSI_FALSE
    end

    local result = self.ncsi_core.ncsi_disable_multi_filter(0, 1, "eth0")

    luaunit.assertFalse(result)
end

-- 测试 ncsi_enable_multi_filter 函数
function TestNCSICore:test_ncsi_enable_multi_filter_success()
    local result = self.ncsi_core.ncsi_enable_multi_filter(0, 1, "eth0")

    luaunit.assertTrue(result)
    luaunit.assertEquals(#self.call_log, 1)
    luaunit.assertEquals(self.call_log[1], {'ncsi_enable_multi_filter', 0, 1, "eth0"})
end

function TestNCSICore:test_ncsi_enable_multi_filter_failure()
    mock_ncsi_cmd.ncsi_enable_multi_filter = function(package_id, channel_id, eth_name)
        return NCSI_FALSE
    end

    local result = self.ncsi_core.ncsi_enable_multi_filter(0, 1, "eth0")

    luaunit.assertFalse(result)
end

-- 测试 ncsi_enable_channelTX 函数
function TestNCSICore:test_ncsi_enable_channelTX_success()
    local result = self.ncsi_core.ncsi_enable_channelTX(0, 1, "eth0")

    luaunit.assertTrue(result)
    luaunit.assertEquals(#self.call_log, 1)
    luaunit.assertEquals(self.call_log[1], {'ncsi_enable_channelTX', 0, 1, "eth0"})
end

function TestNCSICore:test_ncsi_enable_channelTX_failure()
    mock_ncsi_cmd.ncsi_enable_channelTX = function(package_id, channel_id, eth_name)
        return NCSI_FALSE
    end

    local result = self.ncsi_core.ncsi_enable_channelTX(0, 1, "eth0")

    luaunit.assertFalse(result)
end

-- 测试 ncsi_disable_channelTX 函数
function TestNCSICore:test_ncsi_disable_channelTX_success()
    local result = self.ncsi_core.ncsi_disable_channelTX(0, 1, "eth0")

    luaunit.assertTrue(result)
    luaunit.assertEquals(#self.call_log, 1)
    luaunit.assertEquals(self.call_log[1], {'ncsi_disable_channelTX', 0, 1, "eth0"})
end

function TestNCSICore:test_ncsi_disable_channelTX_failure()
    mock_ncsi_cmd.ncsi_disable_channelTX = function(package_id, channel_id, eth_name)
        return NCSI_FALSE
    end

    local result = self.ncsi_core.ncsi_disable_channelTX(0, 1, "eth0")

    luaunit.assertFalse(result)
end

-- 测试 ncsi_set_vlan_filter 函数
function TestNCSICore:test_ncsi_set_vlan_filter_success()
    local result = self.ncsi_core.ncsi_set_vlan_filter(0, 1, 1, 1, 100, "eth0")

    luaunit.assertTrue(result)
    luaunit.assertEquals(#self.call_log, 1)
    luaunit.assertEquals(self.call_log[1], {'ncsi_set_vlan_filter', 0, 1, 1, 1, 100, "eth0"})
end

function TestNCSICore:test_ncsi_set_vlan_filter_failure()
    mock_ncsi_cmd.ncsi_set_vlan_filter = function(package_id, channel_id, vlan_filter, vlan_state, vlan_id, eth_name)
        return NCSI_FALSE
    end

    local result = self.ncsi_core.ncsi_set_vlan_filter(0, 1, 1, 1, 100, "eth0")

    luaunit.assertFalse(result)
end

-- 测试 ncsi_get_link_status 函数
function TestNCSICore:test_ncsi_get_link_status_success()
    local result, link_status = self.ncsi_core.ncsi_get_link_status(0, 1, "eth0", 0)

    luaunit.assertTrue(result)
    luaunit.assertEquals(link_status, {link_up = true, speed = 1000})
    luaunit.assertEquals(#self.call_log, 1)
    luaunit.assertEquals(self.call_log[1], {'ncsi_get_link_status', 0, 1, "eth0", 0})
end

function TestNCSICore:test_ncsi_get_link_status_failure()
    mock_ncsi_cmd.ncsi_get_link_status = function(package_id, channel_id, eth_name, eth_id)
        return NCSI_FALSE, nil
    end

    local result, link_status = self.ncsi_core.ncsi_get_link_status(0, 1, "eth0", 0)

    luaunit.assertFalse(result)
    luaunit.assertNil(link_status)
end

-- 测试 ncsi_common_init 函数
function TestNCSICore:test_ncsi_common_init()
    self.ncsi_core.ncsi_common_init()

    luaunit.assertEquals(#self.call_log, 1)
    luaunit.assertEquals(self.call_log[1], {'ncsi_common_init'})
end

-- 测试 ncsi_paramter_init 函数
function TestNCSICore:test_ncsi_paramter_init_success()
    local result, package_id = self.ncsi_core.ncsi_paramter_init(1, "eth0", 1)

    luaunit.assertTrue(result)
    luaunit.assertEquals(package_id, 0)
    luaunit.assertEquals(#self.call_log, 1)
    luaunit.assertEquals(self.call_log[1], {'ncsi_paramter_init', 1, "eth0", 1})
end

function TestNCSICore:test_ncsi_paramter_init_failure()
    mock_ncsi_cmd.ncsi_paramter_init = function(channel_id, eth_name, state)
        return NCSI_FALSE, nil
    end

    local result, package_id = self.ncsi_core.ncsi_paramter_init(1, "eth0", 1)

    luaunit.assertFalse(result)
    luaunit.assertNil(package_id)
end

-- 测试 ncsi_receive_packet 函数
function TestNCSICore:test_ncsi_receive_packet_success()
    local data = self.ncsi_core.ncsi_receive_packet(3, "eth0")

    luaunit.assertEquals(data, "test_data")
    luaunit.assertEquals(#self.call_log, 1)
    luaunit.assertEquals(self.call_log[1], {'ncsi_receive_packet', 3, "eth0"})
end

function TestNCSICore:test_ncsi_receive_packet_aen_oem_status()
    -- Mock receive_packet to return aen_oem_status = 1
    mock_ncsi_cmd.ncsi_receive_packet = function(fd, eth_name)
        return NCSI_TRUE, 1, "test_data"  -- aen_oem_status = 1
    end

    -- Mock skynet.send to track calls
    local skynet_send_called = false
    mock_skynet.send = function(service, protocol, data)
        skynet_send_called = true
        luaunit.assertEquals(service, 'network_adapter')
        luaunit.assertEquals(protocol, 'network_adapter_ncsi_reinit')
        luaunit.assertEquals(data, '')
    end

    local data = self.ncsi_core.ncsi_receive_packet(3, "eth0")

    luaunit.assertNil(data)  -- 当aen_oem_status=1时，函数应该返回nil
    luaunit.assertTrue(skynet_send_called)
end

function TestNCSICore:test_ncsi_receive_packet_failure()
    mock_ncsi_cmd.ncsi_receive_packet = function(fd, eth_name)
        return NCSI_FALSE, 0, nil
    end

    local data = self.ncsi_core.ncsi_receive_packet(3, "eth0")

    luaunit.assertNil(data)
end

-- 测试 ncsi_eth_mac_init 函数
function TestNCSICore:test_ncsi_eth_mac_init_success()
    local result = self.ncsi_core.ncsi_eth_mac_init("00:18:82:00:00:01")

    luaunit.assertTrue(result)
    luaunit.assertEquals(#self.call_log, 1)
    luaunit.assertEquals(self.call_log[1], {'ncsi_eth_mac_init', "00:18:82:00:00:01"})
end

function TestNCSICore:test_ncsi_eth_mac_init_failure()
    mock_ncsi_cmd.ncsi_eth_mac_init = function(mac_str)
        return NCSI_FALSE
    end

    local result = self.ncsi_core.ncsi_eth_mac_init("00:18:82:00:00:01")

    luaunit.assertFalse(result)
end

-- 测试 ncsi_set_lldp_status 函数
function TestNCSICore:test_ncsi_set_lldp_status_success()
    local result = self.ncsi_core.ncsi_set_lldp_status("eth0", 0, 1, 1)

    luaunit.assertTrue(result)
    luaunit.assertEquals(#self.call_log, 1)
    luaunit.assertEquals(self.call_log[1], {'ncsi_set_lldp_status', "eth0", 0, 1, 1})
end

function TestNCSICore:test_ncsi_set_lldp_status_failure()
    mock_ncsi_cmd.ncsi_set_lldp_status = function(eth_name, package_id, channel_id, status)
        return NCSI_FALSE
    end

    local result = self.ncsi_core.ncsi_set_lldp_status("eth0", 0, 1, 1)

    luaunit.assertFalse(result)
end

-- 测试 ncsi_enable_vlan_req 函数
function TestNCSICore:test_ncsi_enable_vlan_req_success()
    local result = self.ncsi_core.ncsi_enable_vlan_req(0, 1, "eth0", 1)

    luaunit.assertTrue(result)
    luaunit.assertEquals(#self.call_log, 1)
    luaunit.assertEquals(self.call_log[1], {'ncsi_enable_vlan_req', 0, 1, "eth0", 1})
end

function TestNCSICore:test_ncsi_enable_vlan_req_failure()
    mock_ncsi_cmd.ncsi_enable_vlan_req = function(package_id, channel_id, eth_name, vlan_mode)
        return NCSI_FALSE
    end

    local result = self.ncsi_core.ncsi_enable_vlan_req(0, 1, "eth0", 1)

    luaunit.assertFalse(result)
end

-- 测试 ncsi_disable_vlan_req 函数
function TestNCSICore:test_ncsi_disable_vlan_req_success()
    local result = self.ncsi_core.ncsi_disable_vlan_req(0, 1, "eth0")

    luaunit.assertTrue(result)
    luaunit.assertEquals(#self.call_log, 1)
    luaunit.assertEquals(self.call_log[1], {'ncsi_disable_vlan_req', 0, 1, "eth0"})
end

function TestNCSICore:test_ncsi_disable_vlan_req_failure()
    mock_ncsi_cmd.ncsi_disable_vlan_req = function(package_id, channel_id, eth_name)
        return NCSI_FALSE
    end

    local result = self.ncsi_core.ncsi_disable_vlan_req(0, 1, "eth0")

    luaunit.assertFalse(result)
end

-- 测试 ncsi_get_log 函数
function TestNCSICore:test_ncsi_get_log_success()
    local result = self.ncsi_core.ncsi_get_log(0, "eth0", "/tmp/logs", 1234567890)

    luaunit.assertTrue(result)
    luaunit.assertEquals(#self.call_log, 1)
    luaunit.assertEquals(self.call_log[1], {'ncsi_get_log', 0, "eth0", "/tmp/logs", 1234567890})
end

function TestNCSICore:test_ncsi_get_log_failure()
    mock_ncsi_cmd.ncsi_get_log = function(package_id, eth_name, log_dir, time)
        return NCSI_FALSE
    end

    local result = self.ncsi_core.ncsi_get_log(0, "eth0", "/tmp/logs", 1234567890)

    luaunit.assertFalse(result)
end

-- 测试 ncsi_get_log_new 函数
function TestNCSICore:test_ncsi_get_log_new_success()
    local result = self.ncsi_core.ncsi_get_log_new(0, "eth0", "/tmp/logs", 1234567890)

    luaunit.assertTrue(result)
    luaunit.assertEquals(#self.call_log, 1)
    luaunit.assertEquals(self.call_log[1], {'ncsi_get_log_new', 0, "eth0", "/tmp/logs", 1234567890})
end

function TestNCSICore:test_ncsi_get_log_new_failure()
    mock_ncsi_cmd.ncsi_get_log_new = function(package_id, eth_name, log_dir, time)
        return NCSI_FALSE
    end

    local result = self.ncsi_core.ncsi_get_log_new(0, "eth0", "/tmp/logs", 1234567890)

    luaunit.assertFalse(result)
end

-- 测试 ncsi_get_blackbox 函数
function TestNCSICore:test_ncsi_get_blackbox_success()
    local result = self.ncsi_core.ncsi_get_blackbox(0, "eth0", "blackbox.log")

    luaunit.assertTrue(result)
    luaunit.assertEquals(#self.call_log, 1)
    luaunit.assertEquals(self.call_log[1], {'ncsi_get_blackbox', 0, "eth0", "blackbox.log"})
end

function TestNCSICore:test_ncsi_get_blackbox_failure()
    mock_ncsi_cmd.ncsi_get_blackbox = function(package_id, eth_name, log_name)
        return NCSI_FALSE
    end

    local result = self.ncsi_core.ncsi_get_blackbox(0, "eth0", "blackbox.log")

    luaunit.assertFalse(result)
end

-- 测试 get_ncsi_channel_cnt 函数
function TestNCSICore:test_get_ncsi_channel_cnt_valid()
    local result = self.ncsi_core.get_ncsi_channel_cnt()

    luaunit.assertEquals(result, 4)
    luaunit.assertEquals(#self.call_log, 1)
    luaunit.assertEquals(self.call_log[1], {'get_ncsi_channel_cnt'})
end

function TestNCSICore:test_get_ncsi_channel_cnt_invalid_zero()
    mock_ncsi_cmd.get_ncsi_channel_cnt = function()
        return 0  -- 无效值
    end

    local result = self.ncsi_core.get_ncsi_channel_cnt()

    luaunit.assertNil(result)
end

function TestNCSICore:test_get_ncsi_channel_cnt_invalid_too_large()
    mock_ncsi_cmd.get_ncsi_channel_cnt = function()
        return 5  -- 无效值，超过最大值4
    end

    local result = self.ncsi_core.get_ncsi_channel_cnt()

    luaunit.assertNil(result)
end

function TestNCSICore:test_get_ncsi_channel_cnt_valid_boundary()
    -- 测试边界值1和4
    mock_ncsi_cmd.get_ncsi_channel_cnt = function()
        return 1
    end

    local result = self.ncsi_core.get_ncsi_channel_cnt()
    luaunit.assertEquals(result, 1)

    mock_ncsi_cmd.get_ncsi_channel_cnt = function()
        return 4
    end

    result = self.ncsi_core.get_ncsi_channel_cnt()
    luaunit.assertEquals(result, 4)
end

return TestNCSICore