-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at: http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

local bs = require 'mc.bitstring'
local log = require 'mc.logging'
local core = require 'network.core'
local ncsi_protocol_intf = require 'ncsi_protocol_intf'
local ncsi_utils = require 'ncsi.ncsi_protocol.ncsi_utils'
local ncsi_def = require 'ncsi.ncsi_protocol.ncsi_def'
local ncsi_packet = require 'ncsi.ncsi_protocol.ncsi_packet'
local ncsi_parameter = require 'ncsi.ncsi_protocol.ncsi_parameter'

local ncsi_multicast_filter = {}

-- 命令类型常量定义
local ENABLE_MULTICAST_FILTER = 0x12
local ENABLE_MULTICAST_FILTER_RSP = 0x92
local DISABLE_MULTICAST_FILTER = 0x13
local DISABLE_MULTICAST_FILTER_RSP = 0x93

-- 请求长度
local ENABLE_MULTIFILTER_REQ_LEN = 4
local DISABLE_MULTICAST_REQ_LEN = 0

-- 报文大小
local ENABLE_MULTIFILTER_PKT_SIZE = 34
local DISABLE_MULTICAST_PKT_SIZE = 34

-- 多播过滤请求报文结构
local enable_multicast_filter_req_bs = bs.new([[<<
    multicast_filter:32,
    check_sum:32,
    data:22/string,
    fcs:32
>>]])

-- 禁用全局多播过滤请求报文结构
local disable_multicast_req_bs = bs.new([[<<
    check_sum:32,
    pad_data:26/string,
    fcs:32
>>]])

-- 填充多播过滤请求的payload
local function fill_enable_multicast_filter_payload(req_packet, multicast_filter)
    -- 定义公共的payload数据结构
    local payload_data = {
        multicast_filter = multicast_filter or 0,
        check_sum = 0, -- 初始值
        data = string.rep('\0', ENABLE_MULTIFILTER_REQ_LEN),
        fcs = 0        -- 初始值
    }

    -- 首先创建一个check_sum为0的初始payload
    req_packet.payload = enable_multicast_filter_req_bs:pack(payload_data)

    -- 计算校验和和CRC32
    local check_sum = ncsi_utils.get_checksum(req_packet, ncsi_def.PACKET_HEAD_LEN + ENABLE_MULTIFILTER_REQ_LEN)
    local crc32 = ncsi_utils.get_crc32(req_packet, ncsi_def.PACKET_HEAD_LEN + ENABLE_MULTIFILTER_PKT_SIZE - 4)

    -- 更新checksum和FCS字段
    payload_data.check_sum = core.htonl(check_sum)
    payload_data.fcs = core.htonl(crc32)

    -- 返回最终的payload
    return enable_multicast_filter_req_bs:pack(payload_data)
end

-- 写入多播过滤请求
local function write_enable_multicast_filter_req(req_packet, eth_name, multicast_filter)
    ncsi_utils.ncsi_cmd_common_config(req_packet)
    req_packet.packet_head.payload_len_hi = (ENABLE_MULTIFILTER_REQ_LEN >> 8) & 0x0f
    req_packet.packet_head.payload_len_lo = ENABLE_MULTIFILTER_REQ_LEN & 0xff
    req_packet.payload = fill_enable_multicast_filter_payload(req_packet, multicast_filter)

    local req_data = ncsi_utils.ncsi_packet_bs:pack(req_packet)
    return ncsi_protocol_intf.send_ncsi_cmd(req_data,
        ENABLE_MULTIFILTER_PKT_SIZE + ncsi_def.ETHERNET_HEAD_LEN + ncsi_def.PACKET_HEAD_LEN, eth_name)
end

-- 填充禁用全局多播过滤请求的payload
local function fill_disable_multicast_payload(req_packet)
    local check_sum = ncsi_utils.get_checksum(req_packet, ncsi_def.PACKET_HEAD_LEN + DISABLE_MULTICAST_REQ_LEN)
    local crc32 = ncsi_utils.get_crc32(req_packet, ncsi_def.PACKET_HEAD_LEN + DISABLE_MULTICAST_PKT_SIZE - 4)

    local payload_data = disable_multicast_req_bs:pack({
        check_sum = core.htonl(check_sum),
        pad_data = '',
        fcs = core.htonl(crc32)
    })

    return payload_data
end

-- 写入禁用全局多播过滤请求
local function write_disable_multicast_req(req_packet, eth_name)
    ncsi_utils.ncsi_cmd_common_config(req_packet)
    req_packet.packet_head.payload_len_hi = (DISABLE_MULTICAST_REQ_LEN >> 8) & 0x0f
    req_packet.packet_head.payload_len_lo = DISABLE_MULTICAST_REQ_LEN & 0xff
    req_packet.payload = fill_disable_multicast_payload(req_packet)

    local req_data = ncsi_utils.ncsi_packet_bs:pack(req_packet)
    return ncsi_protocol_intf.send_ncsi_cmd(req_data,
        DISABLE_MULTICAST_PKT_SIZE + ncsi_def.ETHERNET_HEAD_LEN + ncsi_def.PACKET_HEAD_LEN, eth_name)
end

-- 读取多播过滤响应
local function read_multicast_filter_rsp(rsp)
    return ncsi_packet.read_common_rsp(rsp, 'multicast filter')
end

-- 命令处理表
local multicast_filter_table = {
    [ENABLE_MULTICAST_FILTER] = write_enable_multicast_filter_req,
    [ENABLE_MULTICAST_FILTER_RSP] = read_multicast_filter_rsp,
    [DISABLE_MULTICAST_FILTER] = write_disable_multicast_req,
    [DISABLE_MULTICAST_FILTER_RSP] = read_multicast_filter_rsp
}

-- 启用多播过滤
function ncsi_multicast_filter.ncsi_enable_multicast_filter(package_id, channel_id, eth_name, multicast_filter_cap)
    ncsi_parameter.get_instance():get_ncsi_parameter().current_channel = channel_id

    local req_packet = ncsi_packet.create_request_packet(package_id, channel_id, ENABLE_MULTICAST_FILTER)

    local custom_cmd_table = ncsi_utils.create_custom_cmd_table(
        multicast_filter_table, ENABLE_MULTICAST_FILTER, write_enable_multicast_filter_req, multicast_filter_cap
    )

    local ret = ncsi_utils.ncsi_cmd_ctrl(package_id, channel_id, req_packet, eth_name, custom_cmd_table)
    if ret ~= ncsi_def.NCSI_SUCCESS then
        log:error('ncsi cmd ctrl enable multicast filter failed, package_id = %s, channel_id = %s, eth_name = %s',
            package_id, channel_id, eth_name)
    end
    return ret
end

-- 禁用多播过滤
function ncsi_multicast_filter.ncsi_disable_multicast_filter(package_id, channel_id, eth_name)
    local req_packet = ncsi_packet.create_request_packet(package_id, channel_id, DISABLE_MULTICAST_FILTER)
    local ret = ncsi_utils.ncsi_cmd_ctrl(package_id, channel_id, req_packet, eth_name, multicast_filter_table)
    if ret ~= ncsi_def.NCSI_SUCCESS then
        log:error('ncsi cmd ctrl disable multicast filter failed, package_id = %s, channel_id = %s, eth_name = %s',
            package_id, channel_id, eth_name)
    end

    return ret
end

return ncsi_multicast_filter