-- Copyright (c) Huawei Technologies Co., Ltd. 2025-2025. All rights reserved.
--
-- this file licensed under the Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at: http://license.coscl.org.cn/MulanPSL2
--
-- THIS SOFTWARE IS PROVIDED ON AN \"AS IS\" BASIS, WITHOUT WARRANTIES OF ANY KIND, EITHER EXPRESS OR
-- IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT, MERCHANTABILITY OR FIT FOR A PARTICULAR
-- PURPOSE.
-- See the Mulan PSL v2 for more details.
--

local class = require 'mc.class'
local log = require 'mc.logging'
local protocol = require 'protocol_open.protocol.protocol'
local ctx = require 'mc.context'

local ncsi_standard = class(protocol)

local request_params_template <const> = {
    packet_type = true,
    channel_id = true,
    package_id = true,
    data = true,
    expect_rsp_packet_type = true
}

-- 只检查是否含有request_params_template以外的参数
function ncsi_standard:validate_request_params(req)
    for key in pairs(req) do
        if not self.request_params_template[key] then
            return false
        end
    end
    return true
end

function ncsi_standard:construct_request_data(_, request)
    local req_ctx = {
        PacketType = tostring(request.packet_type),
        ChannelId = tostring(request.channel_id or 0),
        PackageId = tostring(request.package_id or 0)
    }
    local rsp_ctx = {
        PacketType = tostring(request.expect_rsp_packet_type)
    }
    return req_ctx, rsp_ctx, request.data or ''
end

function ncsi_standard:unpack_response_data(_, rsp_bin)
    return rsp_bin
end

function ncsi_standard:send_request(request)
    local para_ctx = {}
    local req_ctx, rsp_ctx, request_bin = self:construct_request_data(para_ctx, request)
    local ok, rsp_data_bin = pcall(self.endpoint.Request, self.endpoint, ctx.get_context_or_default(),
        request_bin, 0, req_ctx, rsp_ctx)
    if not ok or not rsp_data_bin then
        log:debug('unable to send ncsi request to endpoint, error: %s', rsp_data_bin)
        return nil
    end

    return self:unpack_response_data(para_ctx, rsp_data_bin)
end

function ncsi_standard:ctor(params)
    if not params or not params.endpoint then
        log:raise('unable to create ncsi protocol with invalid params')
    end
    self.name = 'ncsi_standard'
    self.endpoint = params.endpoint
    self.request_params_template = request_params_template
end

return ncsi_standard
