-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at:
--         http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

local Class = require 'mc.class'
local c_lsw_service = require 'lsw.service'
local rpc = require 'interface.rpc'
local mgt = require 'domain.device.switch_mgt'
local reboot_manage = require 'mc.mdb.micro_component.reboot'
local log = require 'mc.logging'
local object_manage = require 'mc.mdb.object_manage'
local ipmi_service = require 'service.ipmi_service'
local ipmi_req = require 'lsw.ipmi.ipmi'

local lsw_app = Class(c_lsw_service)

local SYSTEM_ID<const> = 1
local LAN_SWITCH_ID<const> = 1

function lsw_app:on_reboot_prepare()
    log:info('[lsw_app] reboot prepare')
end

function lsw_app:on_reboot_cancel()
    log:info('[lsw_app] reboot cancel')
end

function lsw_app:on_reboot_action()
    log:info('[lsw_app] reboot action')
end

function lsw_app:register_reboot_methods()
    -- 注册平滑重启回调函数
    -- Prepare准备重启回调
    reboot_manage.on_prepare(function()
        self:on_reboot_prepare()
        return 0
    end)
    -- Action执行重启回调
    reboot_manage.on_action(function()
        self:on_reboot_action()
        return 0
    end)
    -- Cancel取消重启回调
    reboot_manage.on_cancel(function()
        self:on_reboot_cancel()
    end)
end

function lsw_app:register_object_callback()
    object_manage.on_add_object(self.bus, function(class_name, object, position)
        self.mgt:on_add_object(class_name, object, position)
    end)
    object_manage.on_delete_object(self.bus, function(class_name, object, position)
    end)
    object_manage.on_add_object_complete(self.bus, function(position)
        self.mgt:on_add_object_complete(position)
    end)
    object_manage.on_delete_object_complete(self.bus, function(position)
    end)
end

function lsw_app:register_rpc_methods()
    rpc.new(self, self.mgt)
end

function lsw_app:init_domain()
    self.mgt = mgt.new(self.db)
end

function lsw_app:register_ipmi()
    self:register_ipmi_cmd(ipmi_req.GetPortLinkStatus, function(...)
        return self:get_port_link_status(...)
    end)
    self:register_ipmi_cmd(ipmi_req.GetPortSpeed, function(...)
        return self:get_port_speed(...)
    end)
end


function lsw_app:init_service()
    self.ipmi_service = ipmi_service.new(self.mgt)
    self:CreateLanSwitch(SYSTEM_ID, LAN_SWITCH_ID, function(obj)
        obj.ObjectName = 'LanSwitch'
    end)
    self:CreateLanSwitchMock()
    log:notice("lsw_app:init_service, %s",self.ipmi_service)
end

function lsw_app:init()
    lsw_app.super.init(self)
    self:init_domain()
    self:register_rpc_methods()
    self:register_object_callback()
    self:init_service()
    self:register_ipmi()
end

function lsw_app:get_port_link_status(req, ctx)
    local ipmi_ser = self.ipmi_service
    if not ipmi_ser then
        log:error('ipmi_service is nil')
        return
    end

    return ipmi_ser:get_port_link_status(req, ctx)
end

function lsw_app:get_port_speed(req, ctx)
    local ipmi_ser = self.ipmi_service
    if not ipmi_ser then
        log:error('ipmi_service is nil')
        return
    end

    return ipmi_ser:get_port_speed(req, ctx)
end

return lsw_app