-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at:
--         http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.


local skynet = require 'skynet'
local class = require 'mc.class'
local c_service = require 'syssdm.service'
local m_messages = require 'syssdm.types.messages'
local m_router = require 'cmd_service.router'
local c_realtimecmd = require 'cmd_service.realtime_command'

local csyssdm = class(c_service)

function csyssdm:ctor()
end

function csyssdm:init()
    csyssdm.super.init(self)
end

-- system power control initiator call this method to pass the command
---@param SystemId integer
---@param FrudId integer
---@param PowerCtrlType SyssdmEnums.PowerCtrlType
---@param RestartCause SyssdmEnums.RestartCause
---@return SyssdmMsg.PowerCtrlRsp
function csyssdm:PowerCtrl(SystemId, FrudId, PowerCtrlType, RestartCause)
    -- new command sender and send it
    local cmd_strategy = c_realtimecmd.new()
    local sender = m_router.csender.new(self.bus, self.app_path, self.app_interface, cmd_strategy)
    local ret = sender:power_ctrl(SystemId, FrudId, PowerCtrlType:value(), RestartCause:value())
    if ret ~= 0 then
        return m_messages.PowerCtrlRsp.new(-1)
    end

    return m_messages.PowerCtrlRsp.new(0)
end

-- upgrade prepare initiator call this method to pass the command
---@param SystemId integer
---@param FirmwareType string
---@return SyssdmMsg.UpgradeCtrlRsp
function csyssdm:UpgradePrepare(SystemId, FirmwareType)
    -- new command sender and send it
    local cmd_strategy = c_realtimecmd.new()
    local sender = m_router.csender.new(self.bus, self.app_path, self.app_interface, cmd_strategy)
    local ret = sender:upgrade_prepare(SystemId, FirmwareType)
    if ret ~= 0 then
        return m_messages.UpgradeCtrlRsp.new(-1)
    end

    return m_messages.UpgradeCtrlRsp.new(0)
end

-- upgrade prepare executor call this method to pass the command reply
---@param SystemId integer
---@param FirmwareType string
---@param PrepareResult integer
---@return SyssdmMsg.UpgradeCtrlRsp
function csyssdm:UpgradePrepareReply(SystemId, FirmwareType, PrepareResult)
    -- new command replier and send it
    local cmd_strategy = c_realtimecmd.new()
    local replier = m_router.creplier.new(self.bus, self.app_path, self.app_interface, cmd_strategy)
    local ret = replier:upgrade_prepare_reply(SystemId, FirmwareType, PrepareResult)
    if ret ~= 0 then
        return m_messages.UpgradeCtrlRsp.new(-1)
    end

    return m_messages.UpgradeCtrlRsp.new(0)
end

-- upgrade process initiator call this method to pass the command
---@param SystemId integer
---@param FirmwareType string
---@param FilePath string
---@return SyssdmMsg.UpgradeCtrlRsp
function csyssdm:UpgradeProcess(SystemId, FirmwareType, FilePath)
    -- new command sender and send it
    local cmd_strategy = c_realtimecmd.new()
    local sender = m_router.csender.new(self.bus, self.app_path, self.app_interface, cmd_strategy)
    local ret = sender:upgrade_process(SystemId, FirmwareType, FilePath)
    if ret ~= 0 then
        return m_messages.UpgradeCtrlRsp.new(-1)
    end

    return m_messages.UpgradeCtrlRsp.new(0)
end

-- upgrade process executor call this method to pass the command reply
---@param SystemId integer
---@param FirmwareType string
---@param ProcessResult integer
---@return SyssdmMsg.UpgradeCtrlRsp
function csyssdm:UpgradeProcessReply(SystemId, FirmwareType, ProcessResult)
    -- new command replier and send it
    local cmd_strategy = c_realtimecmd.new()
    local replier = m_router.creplier.new(self.bus, self.app_path, self.app_interface, cmd_strategy)
    local ret = replier:upgrade_process_reply(SystemId, FirmwareType, ProcessResult)
    if ret ~= 0 then
        return m_messages.UpgradeCtrlRsp.new(-1)
    end

    return m_messages.UpgradeCtrlRsp.new(0)
end

-- upgrade executor call this method to pass the upgrade status reply
---@param SystemId integer
---@param FirmwareType string
---@param Status integer
---@param Progress integer
---@param ResultDescription string
---@return SyssdmMsg.UpgradeCtrlRsp
function csyssdm:UpgradeStatusReply(SystemId, FirmwareType, Status, Progress, ResultDescription)
    -- new command replier and send it
    local cmd_strategy = c_realtimecmd.new()
    local replier = m_router.creplier.new(self.bus, self.app_path, self.app_interface, cmd_strategy)
    local ret = replier:upgrade_status_reply(SystemId, FirmwareType, Status, Progress, ResultDescription)
    if ret ~= 0 then
        return m_messages.UpgradeCtrlRsp.new(-1)
    end

    return m_messages.UpgradeCtrlRsp.new(0)
end

-- upgrade finish initiator call this method to pass the command
---@param SystemId integer
---@param FirmwareType string
---@return SyssdmMsg.UpgradeCtrlRsp
function csyssdm:UpgradeFinish(SystemId, FirmwareType)
    -- new command sender and send it
    local cmd_strategy = c_realtimecmd.new()
    local sender = m_router.csender.new(self.bus, self.app_path, self.app_interface, cmd_strategy)
    local ret = sender:upgrade_finish(SystemId, FirmwareType)
    if ret ~= 0 then
        return m_messages.UpgradeCtrlRsp.new(-1)
    end

    return m_messages.UpgradeCtrlRsp.new(0)
end

-- upgrade finish executor call this method to pass the command reply
---@param SystemId integer
---@param FirmwareType string
---@param FinishResult integer
---@return SyssdmMsg.UpgradeCtrlRsp
function csyssdm:UpgradeFinishReply(SystemId, FirmwareType, FinishResult)
    -- new command replier and send it
    local cmd_strategy = c_realtimecmd.new()
    local replier = m_router.creplier.new(self.bus, self.app_path, self.app_interface, cmd_strategy)
    local ret = replier:upgrade_finish_reply(SystemId, FirmwareType, FinishResult)
    if ret ~= 0 then
        return m_messages.UpgradeCtrlRsp.new(-1)
    end

    return m_messages.UpgradeCtrlRsp.new(0)
end

return csyssdm
