-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at: http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

local lu = require 'luaunit'
local defs = require 'unit_manager.class.logic_fw.comm_defs'
local fw_mgmt_client = require 'general_hardware.client'
local context = require 'mc.context'
local log = require 'mc.logging'

-- ==================== 模拟模块 ====================

-- 模拟signal模块 - 存储固件列表和升级配置
local signal = {}
signal.fw_list = {}         -- 固件对象列表
signal.upg_cfg_list = {}    -- 升级配置列表
signal.db = {}              -- 数据库对象
signal.bus = {}             -- 总线对象

-- 模拟fructl模块 - 判断是否为多主机类型
local fructl = {}
function fructl.is_multihost_type(bus)
    return bus.is_multihost or false
end

-- ==================== 测试类 ====================

TestSignalActive = {}

-- 测试类初始化 - 在所有测试用例执行前调用一次
function TestSignalActive:setupClass()
    -- 保存原始函数，用于测试完成后恢复
    original_register_action = fw_mgmt_client.FirmwareActiveFirmwareActiveRegisterActiveAction
    
    -- 打桩fw_mgmt_client.FirmwareActiveFirmwareActiveRegisterActiveAction函数
    -- 用于捕获注册生效动作的调用和参数
    self.register_action_called = false   -- 标记是否被调用
    self.register_action_params = nil     -- 保存调用参数
    fw_mgmt_client.FirmwareActiveFirmwareActiveRegisterActiveAction = function(ctx, params)
        self.register_action_called = true
        self.register_action_params = params
    end
end

-- 测试类清理 - 在所有测试用例执行完成后调用一次
function TestSignalActive:teardownClass()
    -- 恢复原始函数，避免影响其他测试
    if original_register_action then
        fw_mgmt_client.FirmwareActiveFirmwareActiveRegisterActiveAction = original_register_action
    end
end

-- 每个测试用例前的准备工作 - 在每个测试用例执行前调用
function TestSignalActive:setUp()
    -- 重置所有测试数据，保证测试用例之间相互独立
    signal.fw_list = {}           -- 清空固件列表
    signal.upg_cfg_list = {}      -- 清空升级配置列表
    signal.db = {}                -- 清空数据库对象
    signal.bus = {}               -- 清空总线对象
    self.register_action_called = false    -- 重置调用标记
    self.register_action_params = nil      -- 清空参数记录
end

-- ==================== VALID_CONDITION_MAP 测试 ====================
-- 测试生效条件枚举值到字符串的映射表
-- 映射表将数字枚举(0,1,2,3)映射为对应的字符串('PowerOff', 'PowerCycle', 'ChassisPowerOff', 'ResetBMC')

-- 测试用例1：验证PowerOff(下电)条件的映射
-- 目的：确保枚举值0正确映射为字符串'PowerOff'
function TestSignalActive:test_valid_condition_map_power_off()
    -- 构造生效条件映射表
    local VALID_CONDITION_MAP = {
        [defs.VALID_CONDITION.POWER_OFF] = defs.VALID_CONDITION_STR.POWER_OFF,
        [defs.VALID_CONDITION.POWER_CYCLE] = defs.VALID_CONDITION_STR.POWER_CYCLE,
        [defs.VALID_CONDITION.CHASSIS_POWER_OFF] = defs.VALID_CONDITION_STR.CHASSIS_POWER_OFF,
        [defs.VALID_CONDITION.RESET_BMC] = defs.VALID_CONDITION_STR.RESET_BMC
    }
    
    -- 验证：数字0映射为'PowerOff'
    lu.assertEquals(VALID_CONDITION_MAP[0], 'PowerOff')
    -- 验证：枚举常量POWER_OFF映射为常量字符串POWER_OFF
    lu.assertEquals(VALID_CONDITION_MAP[defs.VALID_CONDITION.POWER_OFF], defs.VALID_CONDITION_STR.POWER_OFF)
end

-- 测试用例2：验证PowerCycle(电源循环)条件的映射
-- 目的：确保枚举值1正确映射为字符串'PowerCycle'
function TestSignalActive:test_valid_condition_map_power_cycle()
    local VALID_CONDITION_MAP = {
        [defs.VALID_CONDITION.POWER_OFF] = defs.VALID_CONDITION_STR.POWER_OFF,
        [defs.VALID_CONDITION.POWER_CYCLE] = defs.VALID_CONDITION_STR.POWER_CYCLE,
        [defs.VALID_CONDITION.CHASSIS_POWER_OFF] = defs.VALID_CONDITION_STR.CHASSIS_POWER_OFF,
        [defs.VALID_CONDITION.RESET_BMC] = defs.VALID_CONDITION_STR.RESET_BMC
    }
    
    -- 验证：数字1映射为'PowerCycle'
    lu.assertEquals(VALID_CONDITION_MAP[1], 'PowerCycle')
    -- 验证：枚举常量POWER_CYCLE映射为常量字符串POWER_CYCLE
    lu.assertEquals(VALID_CONDITION_MAP[defs.VALID_CONDITION.POWER_CYCLE], defs.VALID_CONDITION_STR.POWER_CYCLE)
end

-- 测试用例3：验证ChassisPowerOff(机箱下电)条件的映射
-- 目的：确保枚举值2正确映射为字符串'ChassisPowerOff'
function TestSignalActive:test_valid_condition_map_chassis_power_off()
    local VALID_CONDITION_MAP = {
        [defs.VALID_CONDITION.POWER_OFF] = defs.VALID_CONDITION_STR.POWER_OFF,
        [defs.VALID_CONDITION.POWER_CYCLE] = defs.VALID_CONDITION_STR.POWER_CYCLE,
        [defs.VALID_CONDITION.CHASSIS_POWER_OFF] = defs.VALID_CONDITION_STR.CHASSIS_POWER_OFF,
        [defs.VALID_CONDITION.RESET_BMC] = defs.VALID_CONDITION_STR.RESET_BMC
    }
    
    -- 验证：数字2映射为'ChassisPowerOff'
    lu.assertEquals(VALID_CONDITION_MAP[2], 'ChassisPowerOff')
    -- 验证：枚举常量CHASSIS_POWER_OFF映射为常量字符串CHASSIS_POWER_OFF
    lu.assertEquals(VALID_CONDITION_MAP[defs.VALID_CONDITION.CHASSIS_POWER_OFF], 
        defs.VALID_CONDITION_STR.CHASSIS_POWER_OFF)
end

-- 测试用例4：验证ResetBMC(复位BMC)条件的映射
-- 目的：确保枚举值3正确映射为字符串'ResetBMC'
function TestSignalActive:test_valid_condition_map_reset_bmc()
    local VALID_CONDITION_MAP = {
        [defs.VALID_CONDITION.POWER_OFF] = defs.VALID_CONDITION_STR.POWER_OFF,
        [defs.VALID_CONDITION.POWER_CYCLE] = defs.VALID_CONDITION_STR.POWER_CYCLE,
        [defs.VALID_CONDITION.CHASSIS_POWER_OFF] = defs.VALID_CONDITION_STR.CHASSIS_POWER_OFF,
        [defs.VALID_CONDITION.RESET_BMC] = defs.VALID_CONDITION_STR.RESET_BMC
    }
    
    -- 验证：数字3映射为'ResetBMC'
    lu.assertEquals(VALID_CONDITION_MAP[3], 'ResetBMC')
    -- 验证：枚举常量RESET_BMC映射为常量字符串RESET_BMC
    lu.assertEquals(VALID_CONDITION_MAP[defs.VALID_CONDITION.RESET_BMC], defs.VALID_CONDITION_STR.RESET_BMC)
end

-- ==================== get_update_component_fw 测试 ====================
-- 测试根据system_id获取待升级的固件组件对象
-- 该函数通过匹配固件名称和UID来查找对应的固件对象

-- 测试用例5：成功获取固件对象
-- 场景：固件名称匹配、UID匹配、system_id匹配
-- 预期：成功返回对应的固件对象
function TestSignalActive:test_get_update_component_fw_success()
    -- 配置升级配置列表：包含一个名为"TestFirmware"的固件配置
    signal.upg_cfg_list = {
        {
            name = "TestFirmware",
            uid = "test_uid_001",
            check_fw_uid_exist = function(fw)
                return fw.uid == "test_uid_001"
            end
        }
    }
    
    -- 配置固件列表：包含一个system_id=1, uid为"test_uid_001"的CPLD固件
    signal.fw_list = {
        {
            system_id = 1,
            uid = "test_uid_001",
            csr = {
                Name = "TestCPLD"
            }
        }
    }
    
    -- 模拟get_update_component_fw函数：根据system_id和fw_name查找匹配的固件
    local function get_update_component_fw(system_id, fw_name)
        for _, cfg in ipairs(signal.upg_cfg_list) do
            if cfg.name ~= fw_name or not cfg.uid then
                goto continue
            end
            
            for _, fw in ipairs(signal.fw_list) do
                if (system_id == defs.ALL_SYSTEM_ID or system_id == fw.system_id) and
                    cfg:check_fw_uid_exist(fw) then
                    return fw
                end
            end
            ::continue::
        end
    end
    
    -- 执行测试：调用函数获取固件对象，传入system_id和固件名称
    local result = get_update_component_fw(1, "TestFirmware")
    -- 验证：返回结果不为空
    lu.assertNotNil(result)
    -- 验证：返回的固件UID正确
    lu.assertEquals(result.uid, "test_uid_001")
    -- 验证：返回的固件名称正确
    lu.assertEquals(result.csr.Name, "TestCPLD")
end

-- 测试用例6：无匹配固件
-- 场景：升级配置列表中没有匹配当前固件名称的配置
-- 预期：返回nil
function TestSignalActive:test_get_update_component_fw_no_matching_firmware()
    signal.upg_cfg_list = {
        {
            name = "TestFirmware",
            uid = "test_uid_001"
        }
    }
    
    signal.fw_list = {}
    
    local function get_update_component_fw(system_id, fw_name)
        for _, cfg in ipairs(signal.upg_cfg_list) do
            if cfg.name ~= fw_name or not cfg.uid then
                goto continue
            end
            
            for _, fw in ipairs(signal.fw_list) do
                if (system_id == defs.ALL_SYSTEM_ID or system_id == fw.system_id) and
                    cfg:check_fw_uid_exist(fw) then
                    return fw
                end
            end
            ::continue::
        end
    end
    
    -- 执行测试：传入不存在的固件名称
    local result = get_update_component_fw(1, "NonExistFirmware")
    -- 验证：返回nil
    lu.assertNil(result)
end

-- 测试用例7：使用ALL_SYSTEM_ID获取固件
-- 场景：system_id为ALL_SYSTEM_ID，可以匹配任意system_id的固件
-- 预期：成功返回固件对象
function TestSignalActive:test_get_update_component_fw_system_id_all()
    signal.upg_cfg_list = {
        {
            name = "TestFirmware",
            uid = "test_uid_001",
            check_fw_uid_exist = function(fw)
                return fw.uid == "test_uid_001"
            end
        }
    }
    
    signal.fw_list = {
        {
            system_id = 5,
            uid = "test_uid_001",
            csr = {
                Name = "TestCPLD"
            }
        }
    }
    
    local function get_update_component_fw(system_id, fw_name)
        for _, cfg in ipairs(signal.upg_cfg_list) do
            if cfg.name ~= fw_name or not cfg.uid then
                goto continue
            end
            
            for _, fw in ipairs(signal.fw_list) do
                if (system_id == defs.ALL_SYSTEM_ID or system_id == fw.system_id) and
                    cfg:check_fw_uid_exist(fw) then
                    return fw
                end
            end
            ::continue::
        end
    end
    
    -- 执行测试：使用ALL_SYSTEM_ID
    local result = get_update_component_fw(defs.ALL_SYSTEM_ID, "TestFirmware")
    -- 验证：成功返回固件对象
    lu.assertNotNil(result)
    -- 验证：返回的system_id正确
    lu.assertEquals(result.system_id, 5)
end

-- ==================== get_active_condition 测试 ====================
-- 测试获取CPLD生效条件的逻辑
-- 生效条件优先级：多主机场景 > CSR配置(ChassisPowerOff/ResetBMC) > 装备定制化 > CSR配置(其他) > 默认值

-- 测试用例9：多主机场景强制返回ChassisPowerOff
-- 场景：系统为多主机类型
-- 预期：无论其他配置如何，都返回'ChassisPowerOff'
function TestSignalActive:test_get_active_condition_multihost()
    -- 设置为多主机类型
    signal.bus = {is_multihost = true}
    
    local VALID_CONDITION_MAP = {
        [defs.VALID_CONDITION.POWER_OFF] = defs.VALID_CONDITION_STR.POWER_OFF,
        [defs.VALID_CONDITION.POWER_CYCLE] = defs.VALID_CONDITION_STR.POWER_CYCLE,
        [defs.VALID_CONDITION.CHASSIS_POWER_OFF] = defs.VALID_CONDITION_STR.CHASSIS_POWER_OFF,
        [defs.VALID_CONDITION.RESET_BMC] = defs.VALID_CONDITION_STR.RESET_BMC
    }
    
    local function get_active_condition(db, fw)
        if fructl.is_multihost_type(signal.bus) then
            return defs.VALID_CONDITION_STR.CHASSIS_POWER_OFF
        end
        
        if fw and (fw.valid_condition == defs.VALID_CONDITION.CHASSIS_POWER_OFF or
            fw.valid_condition == defs.VALID_CONDITION.RESET_BMC) then
            return VALID_CONDITION_MAP[fw.valid_condition]
        end
        
        local customize_sign = db.CustomizeSign and db.CustomizeSign({Id = 'customsize_sign'})
        if customize_sign then
            return customize_sign.ActiveCondition
        end
        
        return fw and fw.valid_condition and VALID_CONDITION_MAP[fw.valid_condition] or 
            defs.VALID_CONDITION_STR.POWER_OFF
    end
    
    -- 执行测试：调用get_active_condition函数
    local result = get_active_condition({}, nil)
    -- 验证：返回ChassisPowerOff
    lu.assertEquals(result, defs.VALID_CONDITION_STR.CHASSIS_POWER_OFF)
end

-- 测试用例10：固件配置为ResetBMC条件
-- 场景：非多主机，固件valid_condition配置为ResetBMC(3)
-- 预期：返回'ResetBMC'
function TestSignalActive:test_get_active_condition_reset_bmc()
    signal.bus = {is_multihost = false}
    
    local VALID_CONDITION_MAP = {
        [defs.VALID_CONDITION.POWER_OFF] = defs.VALID_CONDITION_STR.POWER_OFF,
        [defs.VALID_CONDITION.POWER_CYCLE] = defs.VALID_CONDITION_STR.POWER_CYCLE,
        [defs.VALID_CONDITION.CHASSIS_POWER_OFF] = defs.VALID_CONDITION_STR.CHASSIS_POWER_OFF,
        [defs.VALID_CONDITION.RESET_BMC] = defs.VALID_CONDITION_STR.RESET_BMC
    }
    
    local function get_active_condition(db, fw)
        if fructl.is_multihost_type(signal.bus) then
            return defs.VALID_CONDITION_STR.CHASSIS_POWER_OFF
        end
        
        if fw and (fw.valid_condition == defs.VALID_CONDITION.CHASSIS_POWER_OFF or
            fw.valid_condition == defs.VALID_CONDITION.RESET_BMC) then
            return VALID_CONDITION_MAP[fw.valid_condition]
        end
        
        local customize_sign = db.CustomizeSign and db.CustomizeSign({Id = 'customsize_sign'})
        if customize_sign then
            return customize_sign.ActiveCondition
        end
        
        return fw and fw.valid_condition and VALID_CONDITION_MAP[fw.valid_condition] or 
            defs.VALID_CONDITION_STR.POWER_OFF
    end
    
    -- 配置固件对象：valid_condition为ResetBMC
    local fw = {valid_condition = defs.VALID_CONDITION.RESET_BMC}
    -- 执行测试
    local result = get_active_condition({}, fw)
    -- 验证：返回ResetBMC
    lu.assertEquals(result, defs.VALID_CONDITION_STR.RESET_BMC)
end

-- 测试用例11：装备定制化覆盖生效条件为PowerCycle
-- 场景：固件配置为PowerOff，但装备定制化配置覆盖为PowerCycle
-- 预期：返回装备定制化配置的'PowerCycle'
function TestSignalActive:test_get_active_condition_customize_power_cycle()
    signal.bus = {is_multihost = false}
    
    local VALID_CONDITION_MAP = {
        [defs.VALID_CONDITION.POWER_OFF] = defs.VALID_CONDITION_STR.POWER_OFF,
        [defs.VALID_CONDITION.POWER_CYCLE] = defs.VALID_CONDITION_STR.POWER_CYCLE,
        [defs.VALID_CONDITION.CHASSIS_POWER_OFF] = defs.VALID_CONDITION_STR.CHASSIS_POWER_OFF,
        [defs.VALID_CONDITION.RESET_BMC] = defs.VALID_CONDITION_STR.RESET_BMC
    }
    
    local function get_active_condition(db, fw)
        if fructl.is_multihost_type(signal.bus) then
            return defs.VALID_CONDITION_STR.CHASSIS_POWER_OFF
        end
        
        if fw and (fw.valid_condition == defs.VALID_CONDITION.CHASSIS_POWER_OFF or
            fw.valid_condition == defs.VALID_CONDITION.RESET_BMC) then
            return VALID_CONDITION_MAP[fw.valid_condition]
        end
        
        local customize_sign = db.CustomizeSign and db.CustomizeSign({Id = 'customsize_sign'})
        if customize_sign then
            return customize_sign.ActiveCondition
        end
        
        return fw and fw.valid_condition and VALID_CONDITION_MAP[fw.valid_condition] or 
            defs.VALID_CONDITION_STR.POWER_OFF
    end
    
    local db = {
        CustomizeSign = function(query)
            return {ActiveCondition = 'PowerCycle'}
        end
    }
    local fw = {valid_condition = defs.VALID_CONDITION.POWER_OFF}
    
    local result = get_active_condition(db, fw)
    lu.assertEquals(result, 'PowerCycle')
end

function TestSignalActive:test_get_active_condition_default_power_off()
    signal.bus = {is_multihost = false}
    
    local VALID_CONDITION_MAP = {
        [defs.VALID_CONDITION.POWER_OFF] = defs.VALID_CONDITION_STR.POWER_OFF,
        [defs.VALID_CONDITION.POWER_CYCLE] = defs.VALID_CONDITION_STR.POWER_CYCLE,
        [defs.VALID_CONDITION.CHASSIS_POWER_OFF] = defs.VALID_CONDITION_STR.CHASSIS_POWER_OFF,
        [defs.VALID_CONDITION.RESET_BMC] = defs.VALID_CONDITION_STR.RESET_BMC
    }
    
    local function get_active_condition(db, fw)
        if fructl.is_multihost_type(signal.bus) then
            return defs.VALID_CONDITION_STR.CHASSIS_POWER_OFF
        end
        
        if fw and (fw.valid_condition == defs.VALID_CONDITION.CHASSIS_POWER_OFF or
            fw.valid_condition == defs.VALID_CONDITION.RESET_BMC) then
            return VALID_CONDITION_MAP[fw.valid_condition]
        end
        
        local customize_sign = db.CustomizeSign and db.CustomizeSign({Id = 'customsize_sign'})
        if customize_sign then
            return customize_sign.ActiveCondition
        end
        
        return fw and fw.valid_condition and VALID_CONDITION_MAP[fw.valid_condition] or 
            defs.VALID_CONDITION_STR.POWER_OFF
    end
    
    local result = get_active_condition({}, nil)
    lu.assertEquals(result, defs.VALID_CONDITION_STR.POWER_OFF)
end

-- ==================== get_active_mode 测试 ====================
-- 测试获取CPLD生效动作的逻辑
-- 根据固件的valid_mode属性返回对应的生效动作字符串

-- 测试用例13：BMC重启模式
-- 场景：固件valid_mode配置为SUPPORT_BMC_RESET_ACTIVE(2)
-- 预期：返回'ResetBMC'
function TestSignalActive:test_get_active_mode_reset_bmc()
    -- 模拟get_active_mode函数
    local function get_active_mode(fw)
        if fw and fw.valid_mode == defs.VALID_ACTION.SUPPORT_BMC_RESET_ACTIVE then
            return defs.VALID_ACTION_STR.RESET_BMC
        end
        if fw and fw.valid_mode == defs.VALID_ACTION.SUPPORT_NO_RESET_ACTIVE then
            return defs.VALID_ACTION_STR.RESET_NONE
        end
        return defs.VALID_ACTION_STR.RESET_AC
    end
    
    local fw = {valid_mode = defs.VALID_ACTION.SUPPORT_BMC_RESET_ACTIVE}
    local result = get_active_mode(fw)
    lu.assertEquals(result, defs.VALID_ACTION_STR.RESET_BMC)
end

function TestSignalActive:test_get_active_mode_no_reset()
    local function get_active_mode(fw)
        if fw and fw.valid_mode == defs.VALID_ACTION.SUPPORT_BMC_RESET_ACTIVE then
            return defs.VALID_ACTION_STR.RESET_BMC
        end
        if fw and fw.valid_mode == defs.VALID_ACTION.SUPPORT_NO_RESET_ACTIVE then
            return defs.VALID_ACTION_STR.RESET_NONE
        end
        return defs.VALID_ACTION_STR.RESET_AC
    end
    
    local fw = {valid_mode = defs.VALID_ACTION.SUPPORT_NO_RESET_ACTIVE}
    local result = get_active_mode(fw)
    lu.assertEquals(result, defs.VALID_ACTION_STR.RESET_NONE)
end

function TestSignalActive:test_get_active_mode_default_reset_ac()
    local function get_active_mode(fw)
        if fw and fw.valid_mode == defs.VALID_ACTION.SUPPORT_BMC_RESET_ACTIVE then
            return defs.VALID_ACTION_STR.RESET_BMC
        end
        if fw and fw.valid_mode == defs.VALID_ACTION.SUPPORT_NO_RESET_ACTIVE then
            return defs.VALID_ACTION_STR.RESET_NONE
        end
        return defs.VALID_ACTION_STR.RESET_AC
    end
    
    local fw = {valid_mode = defs.VALID_ACTION.VALID_ACTION_DEFAULT}
    local result = get_active_mode(fw)
    lu.assertEquals(result, defs.VALID_ACTION_STR.RESET_AC)
end

function TestSignalActive:test_get_active_mode_nil_fw()
    local function get_active_mode(fw)
        if fw and fw.valid_mode == defs.VALID_ACTION.SUPPORT_BMC_RESET_ACTIVE then
            return defs.VALID_ACTION_STR.RESET_BMC
        end
        if fw and fw.valid_mode == defs.VALID_ACTION.SUPPORT_NO_RESET_ACTIVE then
            return defs.VALID_ACTION_STR.RESET_NONE
        end
        return defs.VALID_ACTION_STR.RESET_AC
    end
    
    local result = get_active_mode(nil)
    lu.assertEquals(result, defs.VALID_ACTION_STR.RESET_AC)
end

-- ==================== send_register_active_action 测试 ====================
-- 测试向固件管理服务注册CPLD生效动作
-- 该函数将生效条件、生效动作等信息发送给固件管理服务

-- 测试用例17：成功注册生效动作
-- 场景：调用send_register_active_action注册PowerOff条件、ResetAC动作
-- 预期：成功调用fw_mgmt_client接口，参数格式正确，FirmwareId固定为'CPLD'
function TestSignalActive:test_send_register_active_action_success()
    local function send_register_active_action(active_condition, active_mode, active_status)
        local param = {}
        param[#param+1] = {Key = 'FirmwareId', Value = 'CPLD'}
        param[#param+1] = {Key = 'FirmwareType', Value = 'CPLD'}
        param[#param+1] = {Key = 'ActiveCondition', Value = active_condition}
        param[#param+1] = {Key = 'ActiveMode', Value = active_mode}
        param[#param+1] = {Key = 'ActiveStatus', Value = active_status}
        
        local ok, _ = pcall(function()
            fw_mgmt_client.FirmwareActiveFirmwareActiveRegisterActiveAction(context.new(), param)
        end)
        return ok
    end
    
    -- 执行测试：注册PowerOff条件、ResetAC动作
    local ok = send_register_active_action('PowerOff', 'ResetAC', 'Ready')
    -- 验证：调用成功
    lu.assertTrue(ok)
    lu.assertTrue(self.register_action_called)
    lu.assertNotNil(self.register_action_params)
    
    -- 验证参数
    local params = self.register_action_params
    lu.assertEquals(params[1].Key, 'FirmwareId')
    lu.assertEquals(params[1].Value, 'CPLD')  -- FirmwareId固定为'CPLD'
    lu.assertEquals(params[2].Key, 'FirmwareType')
    lu.assertEquals(params[2].Value, 'CPLD')
    lu.assertEquals(params[3].Key, 'ActiveCondition')
    lu.assertEquals(params[3].Value, 'PowerOff')
    lu.assertEquals(params[4].Key, 'ActiveMode')
    lu.assertEquals(params[4].Value, 'ResetAC')
    lu.assertEquals(params[5].Key, 'ActiveStatus')
    lu.assertEquals(params[5].Value, 'Ready')
end

-- 测试用例18：注册ResetBMC条件和动作
-- 场景：调用send_register_active_action注册ResetBMC条件、ResetBMC动作
-- 预期：成功调用fw_mgmt_client接口，FirmwareId为'CPLD'
function TestSignalActive:test_send_register_active_action_reset_bmc()
    local function send_register_active_action(active_condition, active_mode, active_status)
        local param = {}
        param[#param+1] = {Key = 'FirmwareId', Value = 'CPLD'}
        param[#param+1] = {Key = 'FirmwareType', Value = 'CPLD'}
        param[#param+1] = {Key = 'ActiveCondition', Value = active_condition}
        param[#param+1] = {Key = 'ActiveMode', Value = active_mode}
        param[#param+1] = {Key = 'ActiveStatus', Value = active_status}
        
        fw_mgmt_client.FirmwareActiveFirmwareActiveRegisterActiveAction(context.new(), param)
    end
    
    -- 执行测试：注册ResetBMC条件和动作
    send_register_active_action('ResetBMC', 'ResetBMC', 'Idle')
    -- 验证：调用成功
    lu.assertTrue(self.register_action_called)
    
    -- 验证参数
    local params = self.register_action_params
    lu.assertEquals(params[1].Value, 'CPLD')  -- FirmwareId固定为'CPLD'
    lu.assertEquals(params[3].Value, 'ResetBMC')  -- ActiveCondition为ResetBMC
    lu.assertEquals(params[4].Value, 'ResetBMC')  -- ActiveMode为ResetBMC
end

-- ==================== is_immediately_active 测试 ====================
-- 测试判断CPLD是否应立即生效
-- 判断逻辑：系统下电 + 生效条件不是ResetBMC = 立即生效

-- 测试用例19：系统上电状态不立即生效
-- 场景：power_state为'ON'
-- 预期：返回false，不立即生效
function TestSignalActive:test_is_immediately_active_power_on()
    signal.bus = {is_multihost = false}
    
    local VALID_CONDITION_MAP = {
        [defs.VALID_CONDITION.POWER_OFF] = defs.VALID_CONDITION_STR.POWER_OFF,
        [defs.VALID_CONDITION.RESET_BMC] = defs.VALID_CONDITION_STR.RESET_BMC
    }
    
    local function is_immediately_active(system_id, db, power_state)
        if power_state ~= 'OFF' then
            return false
        end
        
        -- 简化版get_update_component_fw
        local component_fw = {valid_condition = defs.VALID_CONDITION.POWER_OFF}
        
        -- 简化版get_active_condition
        local active_condition = VALID_CONDITION_MAP[component_fw.valid_condition]
        
        if active_condition == defs.VALID_CONDITION_STR.RESET_BMC then
            return false
        end
        return true
    end
    
    -- 执行测试：系统上电状态
    local result = is_immediately_active(1, {}, 'ON')
    -- 验证：不立即生效
    lu.assertFalse(result)
end

-- 测试用例20：系统下电但生效条件为ResetBMC不立即生效
-- 场景：power_state为'OFF'，但生效条件为ResetBMC
-- 预期：返回false，需要等待BMC重启时生效
function TestSignalActive:test_is_immediately_active_power_off_reset_bmc()
    signal.bus = {is_multihost = false}
    
    local VALID_CONDITION_MAP = {
        [defs.VALID_CONDITION.RESET_BMC] = defs.VALID_CONDITION_STR.RESET_BMC
    }
    
    local function is_immediately_active(system_id, db, power_state)
        if power_state ~= 'OFF' then
            return false
        end
        
        local component_fw = {valid_condition = defs.VALID_CONDITION.RESET_BMC}
        local active_condition = VALID_CONDITION_MAP[component_fw.valid_condition]
        
        if active_condition == defs.VALID_CONDITION_STR.RESET_BMC then
            return false
        end
        return true
    end
    
    -- 执行测试：系统下电+ResetBMC条件
    local result = is_immediately_active(1, {}, 'OFF')
    -- 验证：不立即生效（需要等待BMC重启）
    lu.assertFalse(result)
end

-- 测试用例21：系统下电且生效条件为PowerOff立即生效
-- 场景：power_state为'OFF'，生效条件为PowerOff
-- 预期：返回true，立即生效
function TestSignalActive:test_is_immediately_active_power_off_power_off_condition()
    signal.bus = {is_multihost = false}
    
    local VALID_CONDITION_MAP = {
        [defs.VALID_CONDITION.POWER_OFF] = defs.VALID_CONDITION_STR.POWER_OFF
    }
    
    local function is_immediately_active(system_id, db, power_state)
        if power_state ~= 'OFF' then
            return false
        end
        
        local component_fw = {valid_condition = defs.VALID_CONDITION.POWER_OFF}
        local active_condition = VALID_CONDITION_MAP[component_fw.valid_condition]
        
        if active_condition == defs.VALID_CONDITION_STR.RESET_BMC then
            return false
        end
        return true
    end
    
    -- 执行测试：系统下电+PowerOff条件
    local result = is_immediately_active(1, {}, 'OFF')
    -- 验证：立即生效
    lu.assertTrue(result)
end

-- 导出测试类
return TestSignalActive

