-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at: http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

-- Description: retimer升级
local singleton = require 'mc.singleton'
local log = require 'mc.logging'
local cmn = require 'common'
local fw_mgmt = require 'general_hardware.client'
local ctx = require 'mc.context'
local parser_cfg = require 'retimer.upgrade.parser_cfg'
local RETIMER_CONSTANTS = require 'retimer.retimer_constants'
local utils = require 'mc.utils'
local upgrade_subject = require 'upgrade_subject.upgrade_subject'

local retimer_upg_service = {}
retimer_upg_service.__index = retimer_upg_service

local RET_OK<const> = 0
local RET_ERR<const> = -1

-- 升级主任务
--- @param system_id number 系统id
--- @param firmware_type string 升级类型
--- @param file_path string 升级文件目录/dev/shm/upgrade/FirmwareX
function retimer_upg_service:upgrade_task(system_id, firmware_type, file_path, parameters)
    -- 解析出升级目录
    if string.match(file_path, "Firmware1$") == nil then
        fw_mgmt:UpdateServiceUpdateServiceProcessReply(ctx.new(), system_id, firmware_type, RET_ERR, parameters)
        return
    end

    local upgrade_status = RET_OK
    -- 没有可升级的资源树对象标志位
    local no_upgrade_target = true
    local dir = parser_cfg.parse_dir(file_path)
    -- 遍历解析hpm包update.cfg得到的固件配置self.cfgs,示例：firmware_name：Firmware1到Firmware5
    for firmware_name, cfg in pairs(self.cfgs) do
        -- 判断固件属于哪种芯片
        local firmware_chip_type = cfg.component_id_ex 

        -- 解析出文件路径对应的cfg配置文件，并获取升级列表
        self.retimer_upgrade_list = self:get_upgrade_list(firmware_chip_type)
        if #self.retimer_upgrade_list == 0 then
            log:notice('[RetimerUpgrade] Can not find retimer with chip type %s.', firmware_chip_type)
            goto continue
        end
        no_upgrade_target = false

        local chip_callback = self.register_chips_callback[firmware_chip_type]
        if chip_callback == nil then
            log:error('[RetimerUpgrade] Has no register chiptype %s.', firmware_chip_type)
            upgrade_status = RET_ERR
            goto continue
        end
    
        -- 解压固件
        utils.secure_tar_unzip( dir .. firmware_name, dir,
            0x6400000, -- 最大解压限制100M
            1024
        )
        
        -- 固件升级准备回调
        local ok = chip_callback.upgrade_prepare_func(self, dir, firmware_chip_type)
        if not ok then
            log:error('[RetimerUpgrade] Prepare failed.')
            upgrade_status = RET_ERR
            goto continue
        end
        
        -- 遍历升级列表，匹配二进制文件
        for _, retimer in pairs(self.retimer_upgrade_list) do
            local ok = chip_callback.upgrade_single_retimer_func(retimer, dir, firmware_chip_type)
            if not ok then
                log:error('[RetimerUpgrade] upgrade retimer %s failed.', retimer.name)
                upgrade_status = RET_ERR
            end
            chip_callback.upgrade_single_retimer_finish_func(self, retimer)
        end

        -- 固件升级结束回调
        chip_callback.upgrade_finish_func(self)

        ::continue::
    end

    if no_upgrade_target then
        upgrade_status = RET_ERR
    end
    log:notice("[RetimerUpgrade] final upgrade status is %s", upgrade_status)
    fw_mgmt:UpdateServiceUpdateServiceProcessReply(ctx.new(), system_id, firmware_type, upgrade_status, parameters)
end

-- 通过升级文件路径获取升级固件列表
function retimer_upg_service:get_upgrade_list(firmware_chip_type)
    local retimer_upgrade_list = {}
    for _, retimer in pairs(self.retimer_collection) do
        if firmware_chip_type == retimer:get_chip_type() then
            table.insert(retimer_upgrade_list, retimer)
        end
    end
    return retimer_upgrade_list
end

function retimer_upg_service:on_upgrade_prepare(context, system_id, firmware_type, cfg_path, _, parameters)
    log:notice("[retimer_upg_service] prepare_upgrade, firmware_type:%s", firmware_type)
    local ok, rsp = fw_mgmt:PFileFileChown(ctx.new(), nil, cfg_path, 104, 104)  --设置配置文件属主为secbox
    if not ok then
        log:error('[RetimerUpgrade] chown cfg file failed, error %s', rsp)
        fw_mgmt:UpdateServiceUpdateServicePrepareReply(ctx.new(), system_id, firmware_type, '', RET_ERR, parameters)
        return
    end
    self.cfgs = parser_cfg.get_cfgs(cfg_path)
    log:notice('[RetimerUpgrade] Prepare retimer upgrade, firmware_type= %s', firmware_type)

    local version = ''
    fw_mgmt:UpdateServiceUpdateServicePrepareReply(ctx.new(), system_id, firmware_type, version, RET_OK, parameters)
end

function retimer_upg_service:on_upgrade_process(context, system_id, firmware_type, file_path, parameters)
    if not next(self.cfgs) then
        return
    end
    log:notice('[retimer_upg_service] Process retimer upgrade, firmware_type= %s', firmware_type)

    local ok, rsp = fw_mgmt:PFileFileChown(ctx.new(), nil, file_path, 104, 104)
    if not ok then
        log:error('[RetimerUpgrade] chown file failed, error %s', rsp)
        fw_mgmt:UpdateServiceUpdateServiceProcessReply(ctx.new(), system_id, firmware_type, RET_ERR, parameters)
        return
    end

    cmn.skynet.fork(function ()
        self:upgrade_task(system_id, firmware_type, file_path, parameters)
    end)
end

function retimer_upg_service:on_upgrade_finish(context, system_id, firmware_type, parameters)
    if not next(self.cfgs) then
        return
    end
    log:notice('[retimer_upg_service] Finish retimer upgrade, firmware_type= %s', firmware_type)

    fw_mgmt:UpdateServiceUpdateServiceFinishReply(ctx.new(), system_id, firmware_type, RET_OK, parameters)
end

function retimer_upg_service.new()
    return setmetatable({}, retimer_upg_service)
end

function retimer_upg_service:init(bus, retimer_collection)
    log:info("[RetimerUpgrade] init")
    self.bus = bus
    self.retimer_collection = retimer_collection
    self.retimer_upgrade_list = {}

    self.cfgs = {}

    self.register_chips_callback = {
        [RETIMER_CONSTANTS.FW_TYPE.FW_5902L] = require 'retimer.upgrade.retimer_upg_5902l',
        [RETIMER_CONSTANTS.FW_TYPE.FW_M88RT51632_X16] = require 'retimer.upgrade.retimer_upg_M88RT51632',
        [RETIMER_CONSTANTS.FW_TYPE.FW_M88RT51632_X8X8] = require 'retimer.upgrade.retimer_upg_M88RT51632',
        [RETIMER_CONSTANTS.FW_TYPE.FW_M88RT51632_X8X4X4] = require 'retimer.upgrade.retimer_upg_M88RT51632',
        [RETIMER_CONSTANTS.FW_TYPE.FW_M88RT51632_X4X4X8] = require 'retimer.upgrade.retimer_upg_M88RT51632',
        [RETIMER_CONSTANTS.FW_TYPE.FW_M88RT51632_X4X4X4X4] = require 'retimer.upgrade.retimer_upg_M88RT51632',
        [RETIMER_CONSTANTS.FW_TYPE.FW_CS81532_X16] = require 'retimer.upgrade.retimer_upg_CS81532',
        [RETIMER_CONSTANTS.FW_TYPE.FW_CS81532_X8X8] = require 'retimer.upgrade.retimer_upg_CS81532',
        [RETIMER_CONSTANTS.FW_TYPE.FW_CS81532_X4X4X4X4] = require 'retimer.upgrade.retimer_upg_CS81532'
    }

    -- 注册固件升级监测
    upgrade_subject.get_instance():register_upgrade_observer(self, 'Retimer')
end

return singleton(retimer_upg_service)