-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at:
--          http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.
local ipmi = require 'ipmi'
local mdb = require 'mc.mdb'
local class = require 'mc.class'
local context = require 'mc.context'
local open_db = require 'general_hardware.db'
local app_base = require 'mc.service_app_base'
local object_manage = require 'mc.mdb.object_manage'
local persist_client = require 'persistence.persist_client_lib'
local orm_classes = require 'general_hardware.orm_classes'
local ok, datas = pcall(require, 'general_hardware.datas')
if not ok then
    datas = nil -- 如果没有datas配置，证明当前组件不需要datas，仅打开数据库
end

local ProcessorMetricsTypes = require 'general_hardware.json_types.ProcessorMetrics'
local PropertiesTypes = require 'mdb.bmc.kepler.Object.PropertiesInterface'
local BoardTypes = require 'general_hardware.json_types.Board'
local UnitTypes = require 'general_hardware.json_types.Unit'
local CpuBoardTypes = require 'general_hardware.json_types.CpuBoard'
local NpuBoardTypes = require 'general_hardware.json_types.NpuBoard'
local MetricTypes = require 'general_hardware.json_types.Metric'
local NpuBoardsTypes = require 'general_hardware.json_types.NpuBoards'
local HddBackplaneTypes = require 'general_hardware.json_types.HddBackplane'
local SimulateTypes = require 'general_hardware.json_types.Simulate'
local M2Types = require 'general_hardware.json_types.M2'
local NetworkAdapterTypes = require 'general_hardware.json_types.NetworkAdapter'
local NetworkAdapterMetricsTypes = require 'general_hardware.json_types.NetworkAdapterMetrics'
local VrdMgmtTypes = require 'general_hardware.json_types.VrdMgmt'
local VrdChipTypes = require 'general_hardware.json_types.VrdChip'
local VrdPowerTypes = require 'general_hardware.json_types.VrdPower'
local RetimerTypes = require 'general_hardware.json_types.Retimer'
local SecurityModuleTypes = require 'general_hardware.json_types.SecurityModule'
local DPUCardTypes = require 'general_hardware.json_types.DPUCard'
local PCIeCardTypes = require 'general_hardware.json_types.PCIeCard'
local FaultTypes = require 'general_hardware.json_types.Fault'
local MetricsTypes = require 'general_hardware.json_types.Metrics'
local DPUCardProcessorTypes = require 'general_hardware.json_types.DPUCardProcessor'
local ManagerTypes = require 'general_hardware.json_types.Manager'
local SOLTypes = require 'general_hardware.json_types.SOL'
local SystemTypes = require 'general_hardware.json_types.System'
local HardwareTypes = require 'general_hardware.json_types.Hardware'
local ManufactureTypes = require 'general_hardware.json_types.Manufacture'
local ExtensionTypes = require 'general_hardware.json_types.Extension'
local SystemsProcessorTypes = require 'general_hardware.json_types.SystemsProcessor'
local GPUTypes = require 'general_hardware.json_types.GPU'
local PowerTypes = require 'general_hardware.json_types.Power'
local StatusTypes = require 'general_hardware.json_types.Status'
local MaintenanceTypes = require 'general_hardware.json_types.Maintenance'
local SmcFirmwareStatusTypes = require 'general_hardware.json_types.SmcFirmwareStatus'

local GPUs = mdb.register_object('/bmc/kepler/Systems/:SystemId/Processors/GPU', {
    {name = 'bmc.kepler.Systems.Processor.ProcessorMetrics', interface = ProcessorMetricsTypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function GPUs:ctor(SystemId)
    self.path = '/bmc/kepler/Systems/' .. SystemId .. '/Processors/GPU'
end

local ExpBoard = mdb.register_object('/bmc/kepler/Systems/:SystemId/Boards/ExpBoard/:Id', {
    {name = 'bmc.kepler.Systems.Board', interface = BoardTypes.interface},
    {name = 'bmc.kepler.Systems.Board.Unit', interface = UnitTypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function ExpBoard:ctor(SystemId, Id)
    self.path = '/bmc/kepler/Systems/' .. SystemId .. '/Boards/ExpBoard/' .. Id
end

local CpuBoard = mdb.register_object('/bmc/kepler/Systems/:SystemId/Boards/CpuBoard/:Id', {
    {name = 'bmc.kepler.Systems.Board', interface = BoardTypes.interface},
    {name = 'bmc.kepler.Systems.Board.Unit', interface = UnitTypes.interface},
    {name = 'bmc.kepler.Systems.Board.CpuBoard', interface = CpuBoardTypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function CpuBoard:ctor(SystemId, Id)
    self.path = '/bmc/kepler/Systems/' .. SystemId .. '/Boards/CpuBoard/' .. Id
end

local NpuBoard = mdb.register_object('/bmc/kepler/Systems/:SystemId/Boards/NpuBoard/:Id', {
    {name = 'bmc.kepler.Systems.Board', interface = BoardTypes.interface},
    {name = 'bmc.kepler.Systems.Board.Unit', interface = UnitTypes.interface},
    {name = 'bmc.kepler.Systems.Board.NpuBoard', interface = NpuBoardTypes.interface},
    {name = 'bmc.kepler.Metric', interface = MetricTypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function NpuBoard:ctor(SystemId, Id)
    self.path = '/bmc/kepler/Systems/' .. SystemId .. '/Boards/NpuBoard/' .. Id
end

local NpuBoards = mdb.register_object('/bmc/kepler/Systems/:SystemId/Boards/NpuBoard', {
    {name = 'bmc.kepler.Systems.Board.NpuBoards', interface = NpuBoardsTypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function NpuBoards:ctor(SystemId)
    self.path = '/bmc/kepler/Systems/' .. SystemId .. '/Boards/NpuBoard'
end

local PsuBoard = mdb.register_object('/bmc/kepler/Systems/:SystemId/Boards/PsuBoard/:Id', {
    {name = 'bmc.kepler.Systems.Board', interface = BoardTypes.interface},
    {name = 'bmc.kepler.Systems.Board.Unit', interface = UnitTypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function PsuBoard:ctor(SystemId, Id)
    self.path = '/bmc/kepler/Systems/' .. SystemId .. '/Boards/PsuBoard/' .. Id
end

local PeuBoard = mdb.register_object('/bmc/kepler/Systems/:SystemId/Boards/PeuBoard/:Id', {
    {name = 'bmc.kepler.Systems.Board', interface = BoardTypes.interface},
    {name = 'bmc.kepler.Systems.Board.Unit', interface = UnitTypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function PeuBoard:ctor(SystemId, Id)
    self.path = '/bmc/kepler/Systems/' .. SystemId .. '/Boards/PeuBoard/' .. Id
end

local HddBackplane = mdb.register_object('/bmc/kepler/Systems/:SystemId/Boards/HddBackplane/:Id', {
    {name = 'bmc.kepler.Systems.Board', interface = BoardTypes.interface},
    {name = 'bmc.kepler.Systems.Board.Unit', interface = UnitTypes.interface},
    {name = 'bmc.kepler.Systems.HddBackplane', interface = HddBackplaneTypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function HddBackplane:ctor(SystemId, Id)
    self.path = '/bmc/kepler/Systems/' .. SystemId .. '/Boards/HddBackplane/' .. Id
end

local FanBoard = mdb.register_object('/bmc/kepler/Systems/:SystemId/Boards/FanBoard/:Id', {
    {name = 'bmc.kepler.Systems.Board', interface = BoardTypes.interface},
    {name = 'bmc.kepler.Systems.Board.Unit', interface = UnitTypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function FanBoard:ctor(SystemId, Id)
    self.path = '/bmc/kepler/Systems/' .. SystemId .. '/Boards/FanBoard/' .. Id
end

local Boards = mdb.register_object('/bmc/kepler/Systems/:SystemId/Boards', {
    {name = 'bmc.kepler.Manufacture.Simulate', interface = SimulateTypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function Boards:ctor(SystemId)
    self.path = '/bmc/kepler/Systems/' .. SystemId .. '/Boards'
end

local RiserCard = mdb.register_object('/bmc/kepler/Systems/:SystemId/Boards/RiserCard/:Id', {
    {name = 'bmc.kepler.Systems.Board', interface = BoardTypes.interface},
    {name = 'bmc.kepler.Systems.Board.Unit', interface = UnitTypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function RiserCard:ctor(SystemId, Id)
    self.path = '/bmc/kepler/Systems/' .. SystemId .. '/Boards/RiserCard/' .. Id
end

local M2TransferCard = mdb.register_object('/bmc/kepler/Systems/:SystemId/Boards/M2TransferCard/:Id', {
    {name = 'bmc.kepler.Systems.Board', interface = BoardTypes.interface},
    {name = 'bmc.kepler.Systems.Board.Unit', interface = UnitTypes.interface},
    {name = 'bmc.kepler.Systems.HddBackplane', interface = HddBackplaneTypes.interface},
    {name = 'bmc.kepler.Systems.Board.M2', interface = M2Types.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function M2TransferCard:ctor(SystemId, Id)
    self.path = '/bmc/kepler/Systems/' .. SystemId .. '/Boards/M2TransferCard/' .. Id
end

local BoardNICCard = mdb.register_object('/bmc/kepler/Systems/:SystemId/Boards/NICCard/:Id', {
    {name = 'bmc.kepler.Systems.Board', interface = BoardTypes.interface},
    {name = 'bmc.kepler.Systems.Board.Unit', interface = UnitTypes.interface},
    {name = 'bmc.kepler.Systems.NetworkAdapter', interface = NetworkAdapterTypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function BoardNICCard:ctor(SystemId, Id)
    self.path = '/bmc/kepler/Systems/' .. SystemId .. '/Boards/NICCard/' .. Id
end

local BoardPCIeCard = mdb.register_object('/bmc/kepler/Systems/:SystemId/Boards/PCIeCard/:Id', {
    {name = 'bmc.kepler.Systems.Board', interface = BoardTypes.interface},
    {name = 'bmc.kepler.Systems.Board.Unit', interface = UnitTypes.interface},
    {name = 'bmc.kepler.Systems.NetworkAdapter', interface = NetworkAdapterTypes.interface},
    {name = 'bmc.kepler.Systems.NetworkAdapter.NetworkAdapterMetrics', interface = NetworkAdapterMetricsTypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function BoardPCIeCard:ctor(SystemId, Id)
    self.path = '/bmc/kepler/Systems/' .. SystemId .. '/Boards/PCIeCard/' .. Id
end

local ChassisBMC = mdb.register_object('/bmc/kepler/Systems/:SystemId/Boards/ChassisBMC/:Id', {
    {name = 'bmc.kepler.Systems.Board', interface = BoardTypes.interface},
    {name = 'bmc.kepler.Systems.Board.Unit', interface = UnitTypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function ChassisBMC:ctor(SystemId, Id)
    self.path = '/bmc/kepler/Systems/' .. SystemId .. '/Boards/ChassisBMC/' .. Id
end

local VrdMgmt = mdb.register_object('/bmc/kepler/Systems/:SystemId/VrdMgmt/:Id', {
    {name = 'bmc.kepler.Systems.VrdMgmt', interface = VrdMgmtTypes.interface},
    {name = 'bmc.kepler.Metric', interface = MetricTypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function VrdMgmt:ctor(SystemId, Id)
    self.path = '/bmc/kepler/Systems/' .. SystemId .. '/VrdMgmt/' .. Id
end

local VrdChip = mdb.register_object('/bmc/kepler/Systems/:SystemId/VrdMgmt/:Id1/VrdChips/:Id2', {
    {name = 'bmc.kepler.Systems.VrdChip', interface = VrdChipTypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function VrdChip:ctor(SystemId, Id1, Id2)
    self.path = '/bmc/kepler/Systems/' .. SystemId .. '/VrdMgmt/' .. Id1 .. '/VrdChips/' .. Id2
end

local VrdPower = mdb.register_object('/bmc/kepler/Systems/:SystemId/VrdMgmt/:Id1/VrdChips/:Id2/VrdPowers/:Id3', {
    {name = 'bmc.kepler.Systems.VrdPower', interface = VrdPowerTypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function VrdPower:ctor(SystemId, Id1, Id2, Id3)
    self.path = '/bmc/kepler/Systems/' .. SystemId .. '/VrdMgmt/' .. Id1 .. '/VrdChips/' .. Id2 .. '/VrdPowers/' .. Id3
end

local Retimer = mdb.register_object('/bmc/kepler/Systems/:SystemId/Retimers/:Id', {
    {name = 'bmc.kepler.Systems.Retimer', interface = RetimerTypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function Retimer:ctor(SystemId, Id)
    self.path = '/bmc/kepler/Systems/' .. SystemId .. '/Retimers/' .. Id
end

local SecurityModule = mdb.register_object('/bmc/kepler/Systems/:SystemId/SecurityModule/:Id', {
    {name = 'bmc.kepler.Systems.SecurityModule', interface = SecurityModuleTypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function SecurityModule:ctor(SystemId, Id)
    self.path = '/bmc/kepler/Systems/' .. SystemId .. '/SecurityModule/' .. Id
end

local DPUCard = mdb.register_object('/bmc/kepler/Systems/:SystemId/PCIeDevices/PCIeCards/DPUCards/:Id', {
    {name = 'bmc.kepler.Systems.DPUCard', interface = DPUCardTypes.interface},
    {name = 'bmc.kepler.Systems.PCIeDevices.PCIeCard', interface = PCIeCardTypes.interface},
    {name = 'bmc.kepler.Systems.DPUCard.Fault', interface = FaultTypes.interface},
    {name = 'bmc.kepler.Systems.DPUCard.Metrics', interface = MetricsTypes.interface},
    {name = 'bmc.kepler.Systems.DPUCard.Processor', interface = DPUCardProcessorTypes.interface},
    {name = 'bmc.kepler.Systems.DPUCard.Manager', interface = ManagerTypes.interface},
    {name = 'bmc.kepler.Systems.DPUCard.Manager.SOL', interface = SOLTypes.interface},
    {name = 'bmc.kepler.Systems.DPUCard.System', interface = SystemTypes.interface},
    {name = 'bmc.kepler.Inventory.Hardware', interface = HardwareTypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function DPUCard:ctor(SystemId, Id)
    self.path = '/bmc/kepler/Systems/' .. SystemId .. '/PCIeDevices/PCIeCards/DPUCards/' .. Id
end

local DftMemoryVolt = mdb.register_object('/bmc/kepler/Manufacture/SelfTest/DftMemoryVolt/:Id', {
    {name = 'bmc.kepler.Manufacture', interface = ManufactureTypes.interface},
    {name = 'bmc.kepler.Manufacture.Extension', interface = ExtensionTypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function DftMemoryVolt:ctor(Id)
    self.path = '/bmc/kepler/Manufacture/SelfTest/DftMemoryVolt/' .. Id
end

local Dft12V0Vlot = mdb.register_object('/bmc/kepler/Manufacture/SelfTest/Dft12V0Vlot/:Id', {
    {name = 'bmc.kepler.Manufacture', interface = ManufactureTypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function Dft12V0Vlot:ctor(Id)
    self.path = '/bmc/kepler/Manufacture/SelfTest/Dft12V0Vlot/' .. Id
end

local Dft5V0Vlot = mdb.register_object('/bmc/kepler/Manufacture/SelfTest/Dft5V0Vlot/:Id', {
    {name = 'bmc.kepler.Manufacture', interface = ManufactureTypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function Dft5V0Vlot:ctor(Id)
    self.path = '/bmc/kepler/Manufacture/SelfTest/Dft5V0Vlot/' .. Id
end

local Dft3V3Vlot = mdb.register_object('/bmc/kepler/Manufacture/SelfTest/Dft3V3Vlot/:Id', {
    {name = 'bmc.kepler.Manufacture', interface = ManufactureTypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function Dft3V3Vlot:ctor(Id)
    self.path = '/bmc/kepler/Manufacture/SelfTest/Dft3V3Vlot/' .. Id
end

local DftCardPresence = mdb.register_object('/bmc/kepler/Manufacture/SelfTest/DftCardPresence/:Id', {
    {name = 'bmc.kepler.Manufacture', interface = ManufactureTypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function DftCardPresence:ctor(Id)
    self.path = '/bmc/kepler/Manufacture/SelfTest/DftCardPresence/' .. Id
end

local DftButtonCell = mdb.register_object('/bmc/kepler/Manufacture/SelfTest/DftButtonCell/:Id', {
    {name = 'bmc.kepler.Manufacture', interface = ManufactureTypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function DftButtonCell:ctor(Id)
    self.path = '/bmc/kepler/Manufacture/SelfTest/DftButtonCell/' .. Id
end

local DftCpld = mdb.register_object('/bmc/kepler/Manufacture/SelfTest/DftCpld/:Id', {
    {name = 'bmc.kepler.Manufacture', interface = ManufactureTypes.interface},
    {name = 'bmc.kepler.Manufacture.Extension', interface = ExtensionTypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function DftCpld:ctor(Id)
    self.path = '/bmc/kepler/Manufacture/SelfTest/DftCpld/' .. Id
end

local DftHwChannel = mdb.register_object('/bmc/kepler/Manufacture/PrepareTest/DftHwChannel/:Id', {
    {name = 'bmc.kepler.Manufacture', interface = ManufactureTypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function DftHwChannel:ctor(Id)
    self.path = '/bmc/kepler/Manufacture/PrepareTest/DftHwChannel/' .. Id
end

local DftFlash = mdb.register_object('/bmc/kepler/Manufacture/SelfTest/DftFlash/:Id', {
    {name = 'bmc.kepler.Manufacture', interface = ManufactureTypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function DftFlash:ctor(Id)
    self.path = '/bmc/kepler/Manufacture/SelfTest/DftFlash/' .. Id
end

local DftJTAG = mdb.register_object('/bmc/kepler/Manufacture/PrepareTest/DftJTAG/:Id', {
    {name = 'bmc.kepler.Manufacture', interface = ManufactureTypes.interface},
    {name = 'bmc.kepler.Manufacture.Extension', interface = ExtensionTypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function DftJTAG:ctor(Id)
    self.path = '/bmc/kepler/Manufacture/PrepareTest/DftJTAG/' .. Id
end

local DftIOTest = mdb.register_object('/bmc/kepler/Manufacture/PrepareTest/DftIOTest/:Id', {
    {name = 'bmc.kepler.Manufacture', interface = ManufactureTypes.interface},
    {name = 'bmc.kepler.Manufacture.Extension', interface = ExtensionTypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function DftIOTest:ctor(Id)
    self.path = '/bmc/kepler/Manufacture/PrepareTest/DftIOTest/' .. Id
end

local GPU = mdb.register_object('/bmc/kepler/Systems/:SystemId/Processors/GPU/:Id', {
    {name = 'bmc.kepler.Systems.Processor', interface = SystemsProcessorTypes.interface},
    {name = 'bmc.kepler.Systems.Processor.GPU', interface = GPUTypes.interface},
    {name = 'bmc.kepler.Systems.Processor.GPU.Power', interface = PowerTypes.interface},
    {name = 'bmc.kepler.Systems.Processor.GPU.Status', interface = StatusTypes.interface},
    {name = 'bmc.kepler.Inventory.Hardware', interface = HardwareTypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function GPU:ctor(SystemId, Id)
    self.path = '/bmc/kepler/Systems/' .. SystemId .. '/Processors/GPU/' .. Id
end

local MaintenanceCsr = mdb.register_object('/bmc/kepler/UpdateService/Maintenance/Csr', {
    {name = 'bmc.kepler.UpdateService.Maintenance', interface = MaintenanceTypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function MaintenanceCsr:ctor()
    self.path = '/bmc/kepler/UpdateService/Maintenance/Csr'
end

local SmcFirmwareStatus = mdb.register_object('/bmc/kepler/Release/Systems/:SystemId/Smc', {
    {name = 'bmc.kepler.Release.Systems.SmcFirmwareStatus', interface = SmcFirmwareStatusTypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function SmcFirmwareStatus:ctor(SystemId)
    self.path = '/bmc/kepler/Release/Systems/' .. SystemId .. '/Smc'
end

local model = require 'class.model'

local general_hardware_service = class(app_base.Service)

general_hardware_service.package = 'general_hardware'

function general_hardware_service:CreateGPUs(SystemId, prop_setting_cb)
    local path = '/bmc/kepler/Systems/' .. SystemId .. '/Processors/GPU'
    return object_manage.create_object('GPUs', path, path, prop_setting_cb)
end

function general_hardware_service:CreateExpBoard(SystemId, Id, prop_setting_cb)
    local path = '/bmc/kepler/Systems/' .. SystemId .. '/Boards/ExpBoard/' .. Id
    return object_manage.create_object('ExpBoard', path, path, prop_setting_cb)
end

function general_hardware_service:CreateCpuBoard(SystemId, Id, prop_setting_cb)
    local path = '/bmc/kepler/Systems/' .. SystemId .. '/Boards/CpuBoard/' .. Id
    return object_manage.create_object('CpuBoard', path, path, prop_setting_cb)
end

function general_hardware_service:CreateNpuBoard(SystemId, Id, prop_setting_cb)
    local path = '/bmc/kepler/Systems/' .. SystemId .. '/Boards/NpuBoard/' .. Id
    return object_manage.create_object('NpuBoard', path, path, prop_setting_cb)
end

function general_hardware_service:CreateNpuBoards(SystemId, prop_setting_cb)
    local path = '/bmc/kepler/Systems/' .. SystemId .. '/Boards/NpuBoard'
    return object_manage.create_object('NpuBoards', path, path, prop_setting_cb)
end

function general_hardware_service:CreatePsuBoard(SystemId, Id, prop_setting_cb)
    local path = '/bmc/kepler/Systems/' .. SystemId .. '/Boards/PsuBoard/' .. Id
    return object_manage.create_object('PsuBoard', path, path, prop_setting_cb)
end

function general_hardware_service:CreatePeuBoard(SystemId, Id, prop_setting_cb)
    local path = '/bmc/kepler/Systems/' .. SystemId .. '/Boards/PeuBoard/' .. Id
    return object_manage.create_object('PeuBoard', path, path, prop_setting_cb)
end

function general_hardware_service:CreateHddBackplane(SystemId, Id, prop_setting_cb)
    local path = '/bmc/kepler/Systems/' .. SystemId .. '/Boards/HddBackplane/' .. Id
    return object_manage.create_object('HddBackplane', path, path, prop_setting_cb)
end

function general_hardware_service:CreateFanBoard(SystemId, Id, prop_setting_cb)
    local path = '/bmc/kepler/Systems/' .. SystemId .. '/Boards/FanBoard/' .. Id
    return object_manage.create_object('FanBoard', path, path, prop_setting_cb)
end

function general_hardware_service:CreateBoards(SystemId, prop_setting_cb)
    local path = '/bmc/kepler/Systems/' .. SystemId .. '/Boards'
    return object_manage.create_object('Boards', path, path, prop_setting_cb)
end

function general_hardware_service:CreateRiserCard(SystemId, Id, prop_setting_cb)
    local path = '/bmc/kepler/Systems/' .. SystemId .. '/Boards/RiserCard/' .. Id
    return object_manage.create_object('RiserCard', path, path, prop_setting_cb)
end

function general_hardware_service:CreateM2TransferCard(SystemId, Id, prop_setting_cb)
    local path = '/bmc/kepler/Systems/' .. SystemId .. '/Boards/M2TransferCard/' .. Id
    return object_manage.create_object('M2TransferCard', path, path, prop_setting_cb)
end

function general_hardware_service:CreateBoardNICCard(SystemId, Id, prop_setting_cb)
    local path = '/bmc/kepler/Systems/' .. SystemId .. '/Boards/NICCard/' .. Id
    return object_manage.create_object('BoardNICCard', path, path, prop_setting_cb)
end

function general_hardware_service:CreateBoardPCIeCard(SystemId, Id, prop_setting_cb)
    local path = '/bmc/kepler/Systems/' .. SystemId .. '/Boards/PCIeCard/' .. Id
    return object_manage.create_object('BoardPCIeCard', path, path, prop_setting_cb)
end

function general_hardware_service:CreateChassisBMC(SystemId, Id, prop_setting_cb)
    local path = '/bmc/kepler/Systems/' .. SystemId .. '/Boards/ChassisBMC/' .. Id
    return object_manage.create_object('ChassisBMC', path, path, prop_setting_cb)
end

function general_hardware_service:CreateVrdMgmt(SystemId, Id, prop_setting_cb)
    local path = '/bmc/kepler/Systems/' .. SystemId .. '/VrdMgmt/' .. Id
    return object_manage.create_object('VrdMgmt', path, path, prop_setting_cb)
end

function general_hardware_service:CreateVrdChip(SystemId, Id1, Id2, prop_setting_cb)
    local path = '/bmc/kepler/Systems/' .. SystemId .. '/VrdMgmt/' .. Id1 .. '/VrdChips/' .. Id2
    return object_manage.create_object('VrdChip', path, path, prop_setting_cb)
end

function general_hardware_service:CreateVrdPower(SystemId, Id1, Id2, Id3, prop_setting_cb)
    local path = '/bmc/kepler/Systems/' .. SystemId .. '/VrdMgmt/' .. Id1 .. '/VrdChips/' .. Id2 .. '/VrdPowers/' .. Id3
    return object_manage.create_object('VrdPower', path, path, prop_setting_cb)
end

function general_hardware_service:CreateRetimer(SystemId, Id, prop_setting_cb)
    local path = '/bmc/kepler/Systems/' .. SystemId .. '/Retimers/' .. Id
    return object_manage.create_object('Retimer', path, path, prop_setting_cb)
end

function general_hardware_service:CreateSecurityModule(SystemId, Id, prop_setting_cb)
    local path = '/bmc/kepler/Systems/' .. SystemId .. '/SecurityModule/' .. Id
    return object_manage.create_object('SecurityModule', path, path, prop_setting_cb)
end

function general_hardware_service:CreateDPUCard(SystemId, Id, prop_setting_cb)
    local path = '/bmc/kepler/Systems/' .. SystemId .. '/PCIeDevices/PCIeCards/DPUCards/' .. Id
    return object_manage.create_object('DPUCard', path, path, prop_setting_cb)
end

function general_hardware_service:CreateDftMemoryVolt(Id, prop_setting_cb)
    local path = '/bmc/kepler/Manufacture/SelfTest/DftMemoryVolt/' .. Id
    return object_manage.create_object('DftMemoryVolt', path, path, prop_setting_cb)
end

function general_hardware_service:CreateDft12V0Vlot(Id, prop_setting_cb)
    local path = '/bmc/kepler/Manufacture/SelfTest/Dft12V0Vlot/' .. Id
    return object_manage.create_object('Dft12V0Vlot', path, path, prop_setting_cb)
end

function general_hardware_service:CreateDft5V0Vlot(Id, prop_setting_cb)
    local path = '/bmc/kepler/Manufacture/SelfTest/Dft5V0Vlot/' .. Id
    return object_manage.create_object('Dft5V0Vlot', path, path, prop_setting_cb)
end

function general_hardware_service:CreateDft3V3Vlot(Id, prop_setting_cb)
    local path = '/bmc/kepler/Manufacture/SelfTest/Dft3V3Vlot/' .. Id
    return object_manage.create_object('Dft3V3Vlot', path, path, prop_setting_cb)
end

function general_hardware_service:CreateDftCardPresence(Id, prop_setting_cb)
    local path = '/bmc/kepler/Manufacture/SelfTest/DftCardPresence/' .. Id
    return object_manage.create_object('DftCardPresence', path, path, prop_setting_cb)
end

function general_hardware_service:CreateDftButtonCell(Id, prop_setting_cb)
    local path = '/bmc/kepler/Manufacture/SelfTest/DftButtonCell/' .. Id
    return object_manage.create_object('DftButtonCell', path, path, prop_setting_cb)
end

function general_hardware_service:CreateDftCpld(Id, prop_setting_cb)
    local path = '/bmc/kepler/Manufacture/SelfTest/DftCpld/' .. Id
    return object_manage.create_object('DftCpld', path, path, prop_setting_cb)
end

function general_hardware_service:CreateDftHwChannel(Id, prop_setting_cb)
    local path = '/bmc/kepler/Manufacture/PrepareTest/DftHwChannel/' .. Id
    return object_manage.create_object('DftHwChannel', path, path, prop_setting_cb)
end

function general_hardware_service:CreateDftFlash(Id, prop_setting_cb)
    local path = '/bmc/kepler/Manufacture/SelfTest/DftFlash/' .. Id
    return object_manage.create_object('DftFlash', path, path, prop_setting_cb)
end

function general_hardware_service:CreateDftJTAG(Id, prop_setting_cb)
    local path = '/bmc/kepler/Manufacture/PrepareTest/DftJTAG/' .. Id
    return object_manage.create_object('DftJTAG', path, path, prop_setting_cb)
end

function general_hardware_service:CreateDftIOTest(Id, prop_setting_cb)
    local path = '/bmc/kepler/Manufacture/PrepareTest/DftIOTest/' .. Id
    return object_manage.create_object('DftIOTest', path, path, prop_setting_cb)
end

function general_hardware_service:CreateGPU(SystemId, Id, prop_setting_cb)
    local path = '/bmc/kepler/Systems/' .. SystemId .. '/Processors/GPU/' .. Id
    return object_manage.create_object('GPU', path, path, prop_setting_cb)
end

function general_hardware_service:CreateMaintenanceCsr(prop_setting_cb)
    return object_manage.create_object('MaintenanceCsr', 'MaintenanceCsr_0',
        '/bmc/kepler/UpdateService/Maintenance/Csr', prop_setting_cb)
end

function general_hardware_service:CreateSmcFirmwareStatus(SystemId, prop_setting_cb)
    local path = '/bmc/kepler/Release/Systems/' .. SystemId .. '/Smc'
    return object_manage.create_object('SmcFirmwareStatus', path, path, prop_setting_cb)
end

function general_hardware_service:ImplGPUsProcessorMetricsSetBandwidthThreshold(cb)
    model.ImplGPUsProcessorMetricsSetBandwidthThreshold(cb)
end

function general_hardware_service:ImplGPUsProcessorMetricsGetBandwidthHistory(cb)
    model.ImplGPUsProcessorMetricsGetBandwidthHistory(cb)
end

function general_hardware_service:ImplGPUsProcessorMetricsClearBandwidthHistory(cb)
    model.ImplGPUsProcessorMetricsClearBandwidthHistory(cb)
end

function general_hardware_service:ImplNpuBoardNpuBoardReset(cb)
    model.ImplNpuBoardNpuBoardReset(cb)
end

function general_hardware_service:ImplNpuBoardNpuBoardResetDevice(cb)
    model.ImplNpuBoardNpuBoardResetDevice(cb)
end

function general_hardware_service:ImplNpuBoardMetricGetData(cb)
    model.ImplNpuBoardMetricGetData(cb)
end

function general_hardware_service:ImplNpuBoardMetricGetItems(cb)
    model.ImplNpuBoardMetricGetItems(cb)
end

function general_hardware_service:ImplNpuBoardMetricSynDataAcquisitionEnable(cb)
    model.ImplNpuBoardMetricSynDataAcquisitionEnable(cb)
end

function general_hardware_service:ImplNpuBoardsNpuBoardsSetNpuPowerCap(cb)
    model.ImplNpuBoardsNpuBoardsSetNpuPowerCap(cb)
end

function general_hardware_service:ImplBoardNICCardNetworkAdapterStartRemovingDevice(cb)
    model.ImplBoardNICCardNetworkAdapterStartRemovingDevice(cb)
end

function general_hardware_service:ImplBoardNICCardNetworkAdapterDumpBlackBox(cb)
    model.ImplBoardNICCardNetworkAdapterDumpBlackBox(cb)
end

function general_hardware_service:ImplBoardPCIeCardNetworkAdapterStartRemovingDevice(cb)
    model.ImplBoardPCIeCardNetworkAdapterStartRemovingDevice(cb)
end

function general_hardware_service:ImplBoardPCIeCardNetworkAdapterDumpBlackBox(cb)
    model.ImplBoardPCIeCardNetworkAdapterDumpBlackBox(cb)
end

function general_hardware_service:ImplVrdMgmtMetricGetData(cb)
    model.ImplVrdMgmtMetricGetData(cb)
end

function general_hardware_service:ImplVrdMgmtMetricGetItems(cb)
    model.ImplVrdMgmtMetricGetItems(cb)
end

function general_hardware_service:ImplVrdMgmtMetricSynDataAcquisitionEnable(cb)
    model.ImplVrdMgmtMetricSynDataAcquisitionEnable(cb)
end

function general_hardware_service:ImplDPUCardDPUCardSetDpuNmi(cb)
    model.ImplDPUCardDPUCardSetDpuNmi(cb)
end

function general_hardware_service:ImplDPUCardDPUCardReset(cb)
    model.ImplDPUCardDPUCardReset(cb)
end

function general_hardware_service:ImplDPUCardDPUCardSetPowerState(cb)
    model.ImplDPUCardDPUCardSetPowerState(cb)
end

function general_hardware_service:ImplDPUCardDPUCardSetBootOption(cb)
    model.ImplDPUCardDPUCardSetBootOption(cb)
end

function general_hardware_service:ImplDPUCardDPUCardImportPublicKey(cb)
    model.ImplDPUCardDPUCardImportPublicKey(cb)
end

function general_hardware_service:ImplDPUCardDPUCardSetLLDPStatus(cb)
    model.ImplDPUCardDPUCardSetLLDPStatus(cb)
end

function general_hardware_service:ImplDPUCardDPUCardGetLLDPStatus(cb)
    model.ImplDPUCardDPUCardGetLLDPStatus(cb)
end

function general_hardware_service:ImplDPUCardSOLSetSerialDirection(cb)
    model.ImplDPUCardSOLSetSerialDirection(cb)
end

function general_hardware_service:ImplDftMemoryVoltManufactureStart(cb)
    model.ImplDftMemoryVoltManufactureStart(cb)
end

function general_hardware_service:ImplDftMemoryVoltManufactureStop(cb)
    model.ImplDftMemoryVoltManufactureStop(cb)
end

function general_hardware_service:ImplDftMemoryVoltManufactureGetResult(cb)
    model.ImplDftMemoryVoltManufactureGetResult(cb)
end

function general_hardware_service:ImplDft12V0VlotManufactureStart(cb)
    model.ImplDft12V0VlotManufactureStart(cb)
end

function general_hardware_service:ImplDft12V0VlotManufactureStop(cb)
    model.ImplDft12V0VlotManufactureStop(cb)
end

function general_hardware_service:ImplDft12V0VlotManufactureGetResult(cb)
    model.ImplDft12V0VlotManufactureGetResult(cb)
end

function general_hardware_service:ImplDft5V0VlotManufactureStart(cb)
    model.ImplDft5V0VlotManufactureStart(cb)
end

function general_hardware_service:ImplDft5V0VlotManufactureStop(cb)
    model.ImplDft5V0VlotManufactureStop(cb)
end

function general_hardware_service:ImplDft5V0VlotManufactureGetResult(cb)
    model.ImplDft5V0VlotManufactureGetResult(cb)
end

function general_hardware_service:ImplDft3V3VlotManufactureStart(cb)
    model.ImplDft3V3VlotManufactureStart(cb)
end

function general_hardware_service:ImplDft3V3VlotManufactureStop(cb)
    model.ImplDft3V3VlotManufactureStop(cb)
end

function general_hardware_service:ImplDft3V3VlotManufactureGetResult(cb)
    model.ImplDft3V3VlotManufactureGetResult(cb)
end

function general_hardware_service:ImplDftCardPresenceManufactureStart(cb)
    model.ImplDftCardPresenceManufactureStart(cb)
end

function general_hardware_service:ImplDftCardPresenceManufactureStop(cb)
    model.ImplDftCardPresenceManufactureStop(cb)
end

function general_hardware_service:ImplDftCardPresenceManufactureGetResult(cb)
    model.ImplDftCardPresenceManufactureGetResult(cb)
end

function general_hardware_service:ImplDftButtonCellManufactureStart(cb)
    model.ImplDftButtonCellManufactureStart(cb)
end

function general_hardware_service:ImplDftButtonCellManufactureStop(cb)
    model.ImplDftButtonCellManufactureStop(cb)
end

function general_hardware_service:ImplDftButtonCellManufactureGetResult(cb)
    model.ImplDftButtonCellManufactureGetResult(cb)
end

function general_hardware_service:ImplDftCpldManufactureStart(cb)
    model.ImplDftCpldManufactureStart(cb)
end

function general_hardware_service:ImplDftCpldManufactureStop(cb)
    model.ImplDftCpldManufactureStop(cb)
end

function general_hardware_service:ImplDftCpldManufactureGetResult(cb)
    model.ImplDftCpldManufactureGetResult(cb)
end

function general_hardware_service:ImplDftHwChannelManufactureStart(cb)
    model.ImplDftHwChannelManufactureStart(cb)
end

function general_hardware_service:ImplDftHwChannelManufactureStop(cb)
    model.ImplDftHwChannelManufactureStop(cb)
end

function general_hardware_service:ImplDftHwChannelManufactureGetResult(cb)
    model.ImplDftHwChannelManufactureGetResult(cb)
end

function general_hardware_service:ImplDftFlashManufactureStart(cb)
    model.ImplDftFlashManufactureStart(cb)
end

function general_hardware_service:ImplDftFlashManufactureStop(cb)
    model.ImplDftFlashManufactureStop(cb)
end

function general_hardware_service:ImplDftFlashManufactureGetResult(cb)
    model.ImplDftFlashManufactureGetResult(cb)
end

function general_hardware_service:ImplDftJTAGManufactureStart(cb)
    model.ImplDftJTAGManufactureStart(cb)
end

function general_hardware_service:ImplDftJTAGManufactureStop(cb)
    model.ImplDftJTAGManufactureStop(cb)
end

function general_hardware_service:ImplDftJTAGManufactureGetResult(cb)
    model.ImplDftJTAGManufactureGetResult(cb)
end

function general_hardware_service:ImplDftIOTestManufactureStart(cb)
    model.ImplDftIOTestManufactureStart(cb)
end

function general_hardware_service:ImplDftIOTestManufactureStop(cb)
    model.ImplDftIOTestManufactureStop(cb)
end

function general_hardware_service:ImplDftIOTestManufactureGetResult(cb)
    model.ImplDftIOTestManufactureGetResult(cb)
end

function general_hardware_service:ImplMaintenanceCsrMaintenanceImport(cb)
    model.ImplMaintenanceCsrMaintenanceImport(cb)
end

function general_hardware_service:ImplSmcFirmwareStatusSmcFirmwareStatusGetFirmwareObjectDetails(cb)
    model.ImplSmcFirmwareStatusSmcFirmwareStatusGetFirmwareObjectDetails(cb)
end

---@param MetricName string
---@param TagPath string
function general_hardware_service:NpuBoardMetricCollectSignal(MetricName, TagPath)
    self.bus:signal('/bmc/kepler/Systems/${SystemId}/Boards/NpuBoard/${Id}', 'bmc.kepler.Metric', 'CollectSignal',
        'a{ss}ss', context.get_context() or {}, MetricName, TagPath)
end

---@param MetricName string
---@param TagPath string
function general_hardware_service:VrdMgmtMetricCollectSignal(MetricName, TagPath)
    self.bus:signal('/bmc/kepler/Systems/${SystemId}/VrdMgmt/${Id}', 'bmc.kepler.Metric', 'CollectSignal', 'a{ss}ss',
        context.get_context() or {}, MetricName, TagPath)
end

function general_hardware_service:get_bus()
    return self.bus
end

function general_hardware_service:register_ipmi_cmd(ipmi_cmd, cb)
    self.ipmi_cmds[ipmi_cmd.name] = ipmi.register_ipmi_cmd(self.bus, self.service_name, ipmi_cmd,
        cb or self[ipmi_cmd.name])
end

function general_hardware_service:unregister_ipmi_cmd(ipmi_cmd)
    local cmd_obj = self.ipmi_cmds[ipmi_cmd.name]
    if not cmd_obj then
        return
    end

    cmd_obj:unregister()
    self.ipmi_cmds[ipmi_cmd.name] = nil
end

function general_hardware_service:ctor()
    self.ipmi_cmds = {}
    self.signal_slots = {}
    self.name = self.name or general_hardware_service.package
    self.db = open_db(':memory:', datas)

    orm_classes.init(self.db)
    self.bus:request_name(app_base.Service.get_service_name(self.name))
    model.init(self.bus)
    general_hardware_service.bus = self.bus
end

function general_hardware_service:pre_init()
    general_hardware_service.super.pre_init(self)
    self.persist = persist_client.new(self.bus, self.db, self, {
        ['t_hardware_security_module'] = true,
        ['t_dpucard'] = true,
        ['t_cpld_space_test_result'] = true
    })
    object_manage.set_persist_client(self.persist)
end

function general_hardware_service:init()
    general_hardware_service.super.init(self)
end

return general_hardware_service
