-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at:
--         http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

-- lm75 模型（块设备）
-- 支持offsetwidth: 1
-- 数据大小：2
local Lm75 = {
    offset_width = 1,
    size = 2
}

local log = require "log"
local cjson = require "cjson"
local CommonI2c = require "protocol.CommonI2c"
local BinParser = require "data.BinParser"

function Lm75:new(o, offset_width, size)
    o = o or {}
    setmetatable(o, self)
    self.__index = self
    self.offset_width = offset_width
    self.size = size
    return o
end

local Lm75Parser = {
    prot_parser = {},
    data_parser = {},
}

function Lm75Parser:new(o, prot_parser, data_parser)
    o = o or {}
    setmetatable(o, self)
    self.__index = self
    self.prot_parser = prot_parser
    self.data_parser = data_parser
    return o
end

function Lm75Parser:read_data(length)
    local offset = self.prot_parser:parse_offset()
    return self.data_parser:read_data(offset, length)
end

function Lm75Parser:write_data()
    local offset = self.prot_parser:parse_offset()
    local data = self.prot_parser:parse_data()

    self.data_parser:write_data(offset, data)
end

function Lm75:read(tx_buffer, length, filename)
    if type(tx_buffer) ~= "table" or #tx_buffer < 1 then
        return
    end

    if #tx_buffer == 1 then
        tx_buffer[2] = 0
    end

    local common_i2c = CommonI2c:new(nil, tx_buffer, self.offset_width)
    local bin_parser = BinParser:new(nil, filename, self.size)

    local lm75_parser = Lm75Parser:new(nil, common_i2c, bin_parser)
    return lm75_parser:read_data(length)
end

function Lm75:write(tx_buffer, filename)
    if type(tx_buffer) ~= "table" or #tx_buffer < 3 then
        return
    end
    local common_i2c = CommonI2c:new(nil, tx_buffer, self.offset_width)
    local bin_parser = BinParser:new(nil, filename, self.size)

    local lm75_parser = Lm75Parser:new(nil, common_i2c, bin_parser)
    lm75_parser:write_data()
end

return Lm75