-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at:
--         http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

-- Description: fructrl RPC methods

local class = require 'mc.class'
local c_service = require 'fructrl.service'
local log = require 'mc.logging'
local p_ipmi = require 'pwr_powerctrl_ipmi'
local c_fructrl_obj_mgnt = require 'fructrl_obj_mgnt'
local reboot = require 'mc.mdb.micro_component.reboot'
local intf_debug = require 'mc.mdb.micro_component.debug'
local dump = require 'fructrl_dump'
local orm_object_manage = require 'mc.orm.object_manage'
local mdb_config_manage = require 'mc.mdb.micro_component.config_manage'
local im_ex_eng = require 'import_export_engine'
local mc_admin = require 'mc.mc_admin'
local skynet = require 'skynet'

-- class fructrl
local cpayload = class(c_service)

function cpayload:init()
    log:notice('[Fructrl] service init')
    cpayload.super.init(self)
    -- 依赖检查
    cpayload:check_dependencies()
    -- 开启持久化特性
    orm_object_manage.get_instance(self.db, self.bus):start()

    self.fructrl_obj_mgnt = c_fructrl_obj_mgnt.new(self.db, self.bus)
    self.fructrl_obj_mgnt:object_manage_callback()

    -- 初始化平滑重启接口
    self:init_reboot()
    -- 自定义日志导出
    intf_debug.on_dump(function(...)
        return dump.log_dump_cb(...)
    end)
    -- 导入导出注册
    self:import_export_engine_register()
    -- 资源RPC回调接口
    self:register_rpc_methods()
    -- 注册需要的ipmi接口
    skynet.fork_once(function()
        p_ipmi:register_ipmi(function (...)
            self:register_ipmi_cmd(...)
        end)
    end)
end

-- 依赖检查
function cpayload:check_dependencies()
    local admin = mc_admin.new()
    admin:parse_dependency(APP_WORKING_DIRECTORY .. '/mds/service.json')
    admin:check_dependency(self.bus)
end

-- 往导入导出引擎注册信息
function cpayload:import_export_engine_register()
    mdb_config_manage.on_import(function(...)
        return im_ex_eng.import(...)
    end)
    mdb_config_manage.on_export(function(...)
        return im_ex_eng.export(...)
    end)
end

function cpayload:init_reboot()
    reboot.on_prepare(function(...)
        return self:on_reboot_prepare()
    end)
    reboot.on_action(function(...)
        return self:on_reboot_action()
    end)
    reboot.on_cancel(function(...)
        return self:on_reboot_cancel()
    end)
end

function cpayload:on_reboot_prepare()
    log:notice('fructrl has no extra preparation for reboot.')
    return 0
end

function cpayload:on_reboot_action()
    log:notice('fructrl has no extra action for reboot.')
    return 0
end

function cpayload:on_reboot_cancel()
    log:notice('fructrl has no extra cancel for reboot.')
end

function cpayload:register_rpc_methods()
    self:ImplFruCtrlSystemsFruCtrlPowerCtrl(function(...)
        return self.fructrl_obj_mgnt:PowerCtrl(...)
    end)
    self:ImplFruCtrlSystemsPowerOnLockSetPowerOnLock(function(...)
        return self.fructrl_obj_mgnt:SetPowerOnLock(...)
    end)
    self:ImplFruCtrlPowerOffLockSetPowerOffLock(function(...)
        return self.fructrl_obj_mgnt:SetPowerOffLock(...)
    end)
    self:ImplFruCtrlSystemsFruCtrlSetPowerOnStrategyExceptions(function(...)
        return self.fructrl_obj_mgnt:SetPowerOnStrategyExceptions(...)
    end)
    self:ImplChassisFruCtrlChassisFruCtrlPowerCtrl(function (obj, ctx, PowerCtrlType, RestartCause)
        return self.fructrl_obj_mgnt:chassis_power_ctrl(ctx, PowerCtrlType, RestartCause)
    end)
    self:ImplChassisFruCtrlChassisPowerOnLockSetPowerOnLock(function (...)
        return self.fructrl_obj_mgnt:chassis_set_poweron_lock(...)
    end)
    self:ImplChassisFruCtrlPowerOffLockSetPowerOffLock(function (...)
        return self.fructrl_obj_mgnt:chassis_set_poweroff_lock(...)
    end)
    self:ImplChassisFruCtrlChassisFruCtrlSetPowerOnStrategyExceptions(function (...)
        return self.fructrl_obj_mgnt:chassis_set_poweron_strategy_exceptions(...)
    end)
    self:ImplFruCtrlSystemsFruCtrlSetACLost(function (...)
        return self.fructrl_obj_mgnt:SetACLost(...)
    end)
    self:ImplFruCtrlSystemsFruCtrlFruPowerCtrl(function (...)
        return self.fructrl_obj_mgnt:PowerCtrl(...)
    end)
end

return cpayload
