-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at:
--         http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.


local class = require 'mc.class'
local log = require 'mc.logging'
local manufacture_app = class()

local STATUS_COMPLETE = "Complete"
local STATUS_TESTING = "Testing"
local STATUS_UNSTART = "Unstart"

local RESULT_SUCCEED = "Succeed"
local RESULT_FAILED = "Failed"
local RESULT_NON = "Non"

function manufacture_app.new(app)
    return setmetatable({app = app}, manufacture_app)
end

function manufacture_app:init()
    self:register_dft_powerbutton_methods()
    self.button_test_result = RESULT_NON
end

function manufacture_app:register_dft_powerbutton_methods()
    self.app:ImplDftPowerButtonManufactureStart(function (obj, ...)
        return self:dft_powerbutton_start(obj)
    end)
    self.app:ImplDftPowerButtonManufactureStop(function (obj, ...)
        return self:dft_powerbutton_stop(obj)
    end)
    self.app:ImplDftPowerButtonManufactureGetResult(function (obj, ...)
        return self:dft_powerbutton_get_result(obj)
    end)
end

function manufacture_app:dft_powerbutton_start(obj)
    log:notice('start dft powerbutton')
    local fructrl_obj_mgnt = self.app.fructrl_obj_mgnt
    local fructrl = fructrl_obj_mgnt.host_obj[1].fructrl
    local button_evt = fructrl_obj_mgnt.button_evt
    -- 清0标记
    button_evt:set_button_evt_flag(0)

    -- 先清除按钮事件，保证测试不会误报
    button_evt:clear_PwrButtonEvt()
    -- 启动测试时，需要将电源按钮重新锁住，防止测试过程中下电
    fructrl:set_PanelPowerButtonEnabled(false)
    -- 启动测试时，设置为开始测试状态
    obj.Status = STATUS_TESTING
    self.button_test_result = RESULT_FAILED
end

function manufacture_app:dft_powerbutton_stop(obj)
    log:notice('stop dft powerbutton')
    -- 结束测试时，需要将电源按钮解锁，恢复正常功能
    local fructrl_obj_mgnt = self.app.fructrl_obj_mgnt
    local fructrl = fructrl_obj_mgnt.host_obj[1].fructrl
    fructrl:set_PanelPowerButtonEnabled(true)
    obj.Status = STATUS_UNSTART
    self.button_test_result = RESULT_NON
end

function manufacture_app:dft_powerbutton_get_result(obj)
    local description = ""
    local fructrl_obj_mgnt = self.app.fructrl_obj_mgnt
    local button_evt = fructrl_obj_mgnt.button_evt

    obj.Status = STATUS_COMPLETE
    log:notice('dft_powerbutton_get_result status=(%s)', self.button_test_result)
    -- 按过按钮，重复查询返回成功
    if self.button_test_result == RESULT_SUCCEED then
        return RESULT_SUCCEED, description
    end
    if self.button_test_result ==  RESULT_FAILED then
        -- 没按按钮，返回失败
        if button_evt:get_button_evt_flag() == 0 then
            return RESULT_FAILED, description
        end

        -- 测试中第一次按按钮，记录测试完成
        button_evt:set_button_evt_flag(0) -- 清0按钮事件标记
        self.button_test_result = RESULT_SUCCEED
        return RESULT_SUCCEED, description
    end
    return RESULT_NON, description
end

return manufacture_app