-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at:
--         http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.


local log = require 'mc.logging'
local chassis_def = require 'def.chassis_def'
local uid_state = chassis_def.UIDLEDSTATE
local uid_interval = chassis_def.UIDLEDINTERVAL

local led_object = {}
led_object.__index = led_object

function led_object.new_for_csr(obj)
    return setmetatable({uidled_info = obj, background_mode = 0, uid_control_right = chassis_def.UID_CONTROL_RIGHT.DEFAULT}, led_object)
end

-- 获取state
function led_object:get_led_state()
    return self.uidled_info.State
end

-- 获取寄存器的值
function led_object:get_led_ctrl_value()
    return self.uidled_info.CtrlValue
end

-- 设置灯的颜色
function led_object:set_capability(used_color)
    self.uidled_info.Capability = used_color
end

--只有当前state为灭，用户登录web,kvm,ssh时才会设置uid闪烁
--或者是用户登录设置的闪烁时，登出才会置灭
function led_object:update_uid_state(state, control_right)
    if control_right == chassis_def.UID_CONTROL_RIGHT.DEFAULT then
        self.uid_control_right = chassis_def.UID_CONTROL_RIGHT.DEFAULT
        return true
    end
    if (self.uidled_info.State == chassis_def.UIDLEDSTATE.LED_OFF and state == chassis_def.UIDLEDSTATE.LED_BLINK) or
            self.uid_control_right == chassis_def.UID_CONTROL_RIGHT.BLINK_BY_REMOTE_LOGIN then
        self.uid_control_right = chassis_def.UID_CONTROL_RIGHT.BLINK_BY_REMOTE_LOGIN
        return true
    end
    return false
end

-- 根据传入的state，计算亮灯持续时间和关灯持续时间，并更新
function led_object:update_led_info(state, control_right)
    if control_right == nil then
        control_right = chassis_def.UID_CONTROL_RIGHT.DEFAULT
    end
    if not self:update_uid_state(state, control_right) then
        log:notice('UID current state is not supported to set by remote login')
        return
    end
    local tmp_state = self.uidled_info.State
    local tmp_on_duration = self.uidled_info.OnDuration
    local tmp_off_duration = self.uidled_info.OffDuration
    self.uidled_info.State = state
    local on_duration, off_duration = self:get_onoffduration_by_state(state)
    -- 设置on_duration，off_duration
    self.uidled_info.OnDuration = on_duration
    self.uidled_info.OffDuration = off_duration
    local led_control = ((on_duration // 10) << 4) + (off_duration // 10) & 0xFF
    -- 修改寄存器的值，当扩展板CPLD被擦空之后，写寄存器会失败
    local ok = pcall(function ()
        self.uidled_info.CtrlValue = led_control
    end)
    -- 设置失败之后需要恢复状态，避免资源树状态和实际不符
    if not ok then
        log:error("set the CtrlValue %s failed", led_control)
        self.uidled_info.State = tmp_state
        self.uidled_info.OnDuration = tmp_on_duration
        self.uidled_info.OffDuration = tmp_off_duration
        return
    end

    return on_duration, off_duration
end

-- 根据state获取闪烁的时间间隔
function led_object:get_onoffduration_by_state(state)
    local on_duration, off_duration
    if state == uid_state.LED_OFF then
        -- 关闭状态下的时间间隔
        on_duration = 0
        off_duration = 0
    elseif state == uid_state.LED_BLINK then
        -- 闪烁状态下的时间间隔
        on_duration = uid_interval.LED_IDENTIFY_INTERVAL_ON
        off_duration = uid_interval.LED_IDENTIFY_INTERVAL_OFF
    elseif state == uid_state.LED_ON then
        -- 常亮状态下的时间间隔
        on_duration = uid_interval.LED_INTERVAl_ON
        off_duration = 0
    elseif state == 7 then
        on_duration = 10
        off_duration = 10
    end
    log:info('value of the on_duration: %d, off_duration: %d', on_duration, off_duration)
    return on_duration, off_duration
end

return led_object
