-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at:
--         http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

local file_service = require 'service.file_service'
local bs_util = require 'util.base_util'
local xml_util = require 'util.xml_util'
local prop_def = require 'macros.property_def'
local registry_service = require 'service.registry_service'
local setting_service = require 'service.setting_service'
local bios_service = require 'service.bios_service'
local bios_factory = require 'factory.bios_factory'
local cfg_handler = require 'handler.config_handler'
local pro_global = require 'macros.property_global'
local struct = require 'pojo.struct_def'
local project_dir = os.getenv('PROJECT_DIR')

local test_common = require 'test_common.utils'
local bus = test_common.dbus_launch('../.dbus', nil, true)

test_import_bios_config = {}

function test_import_bios_config:test_bean()
    local bios_ser = bios_service.get_instance(bus, {})
    assert(bios_ser ~= nil)

    bios_factory.register_bean('bios_service', bios_ser)

    bios_ser:set_bios_prop('CurrentValueFileName', project_dir .. '/test/test_data/currentvalue.json')
    bios_ser:set_bios_prop('ResultFileName', project_dir .. '/test/test_data/result.json')
    bios_ser:set_bios_prop('SettingFileName', project_dir .. '/test/test_data/setting.json')
    bios_ser:set_bios_prop('RegistryFileName', project_dir .. '/test/test_data/registry.json')

end

function test_import_bios_config:test_z_clear()
    local bios_ser = bios_service.get_instance(bus, {})
    assert(bios_ser ~= nil)

    bios_ser:set_bios_prop('CurrentValueFileName', '/data/opt/currentvalue.json')
    bios_ser:set_bios_prop('ResultFileName', '/data/opt/result.json')
    bios_ser:set_bios_prop('SettingFileName', '/data/opt/setting.json')
    bios_ser:set_bios_prop('RegistryFileName', '/data/opt/registry.json')
end

function test_import_bios_config:test_xml_util()
    local message = xml_util.xml_2_json()
    assert(message.ErrCode == prop_def.RESPONSE_XML_CONTENT_NULL)

    local registry_path = project_dir .. '/test/test_data/registry.json'
    local registry_json = bs_util.get_file_json(registry_path)

    local registry_ser = registry_service.new(registry_json)
    message = xml_util.xml_2_json(project_dir .. '/test/test_data/config.xml', registry_ser)
    assert(message.ErrCode == prop_def.E_OK)
    local res_data = message.Data
    local xml_json = res_data.xml_json_data
    local traverse_list = res_data.traverse_list_data
    assert(xml_json ~= nil)
    assert(traverse_list ~= nil)
end

function test_import_bios_config:test_setting_service()
    local setting_json = bs_util.get_file_json(project_dir .. '/test/test_data/currentvalue.json')
    local setting_ser = setting_service.new(setting_json)
    assert(setting_ser ~= nil)

    local setting_obj = setting_ser:get_setting_obj()
    assert(setting_obj ~= nil)

    local prop_val = setting_ser:get_prop_val('PCIEMaxPayloadSizePort[28]')
    assert(prop_val == 'LEVEL-256B')

    local valid = setting_ser:check_boot_order()
    assert(valid == true)

    local err_code = setting_ser:compare_setting('WarmBootFastSupport', 'Enabled')
    assert(err_code == prop_def.E_OK)

    err_code = setting_ser:compare_setting('WarmBootFastSupport', 'Disabled')
    assert(err_code == prop_def.E_FAILURE)
end

function test_import_bios_config:test_config_handler()
    local traverse_cfg = {}
    traverse_cfg['a'] = 'b'
    local output_cfg = {}
    output_cfg['a'] = 'b'
    cfg_handler.export_handler.handle(traverse_cfg, output_cfg)
    assert(output_cfg['a'] == nil)

    output_cfg['c'] = 'e'
    cfg_handler.export_handler.handle(traverse_cfg, output_cfg)
    assert(output_cfg['c'] == 'e')

    local res = cfg_handler.import_handler.handle(traverse_cfg, output_cfg)
    assert(res == 'a')

    res = cfg_handler.import_handler.handle(traverse_cfg, output_cfg)
    assert(res == 'a')

    res = cfg_handler.import_handler.handle(nil, output_cfg)
    assert(res == nil)
end
