-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at: http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

local bios_service = require 'service.bios_service'
local bios_enum = require 'bios.types.enums'
local bios_obj = require 'pojo.bios_object_mutihost'
local bs_util = require 'util.base_util'
local lu = require 'luaunit'
local ipmi = require 'ipmi'
local comp_code = ipmi.types.Cc

TestBiosService = {}

function TestBiosService:test_set_vrd_type_map()
    local val = 1
    bios_service:set_vrd_type_map(val)
    lu.assertEquals(bios_service.vrd_type_map, val)
end

function TestBiosService:test_get_cpu_physical_id()
    local logical_id = 1
    local ret = bios_service:get_cpu_physical_id(logical_id, 1)
    lu.assertEquals(ret, 2)

    local mock_1 = bios_service.get_obj
    bios_service.get_obj = function()
        return {
            get_cpu_info = function()
                return {
                    CPU = {
                        LogicalSocketId = logical_id,
                        PhysicalSocketId = 2,
                        DeviceLocator = 3
                    }
                }
            end
        }
    end
    local ret1, ret2 = bios_service:get_cpu_physical_id(logical_id, nil)
    lu.assertEquals(ret1, 2)
    lu.assertEquals(ret2, 3)

    bios_service.get_obj = mock_1
end

function TestBiosService:test_get_power_type()
    local power_type = 2
    bios_service.vrd_type_map = {
        IdRange = {
            power_type
        },
        NameRange = {
            power_type
        }
    }
    lu.assertEquals(bios_service:get_power_type(2), power_type)
end

function TestBiosService:test_write_data()
    local req = {}
    local data_opt = {}
    local info = {}
    data_opt.get_offset = function ()
        return 1
    end
    info.OffsetToWrite = 0
    local data = bios_service:write_data(req, info, data_opt, 1)
    lu.assertEquals(data.CompletionCode, bios_enum.BiosErrCode.BIOS_ERR_NORMALLY:value())
end

function TestBiosService:test_bios_fun()
    local tmp= {
        object_service = {},
        system_id = {},
        path = {},
        multihost_card_config = {},
        pcie_obj_array = {},
        nic_obj_array = {},
        obj_list = {},
        obj = {}
    }
    local ok, _ = pcall(function()
        bios_service:update_pcie_nic_obj_array(tmp.object_service, tmp.system_id, tmp.path, 
        tmp.tmp.multihost_card_config, tmp.pcie_obj_array, tmp.nic_obj_array, tmp.obj_list)
    end)
    lu.assertEquals(ok, false)

    local ok, _ = pcall(function()
        bios_service:get_pcie_silk_config(tmp.system_id, tmp.multihost_card_config, tmp.obj)
    end)
    lu.assertEquals(ok, true)

    local ok, _ = pcall(function()
        bios_service:get_disk_silk_config(tmp.system_id, tmp.obj)
    end)
    lu.assertEquals(ok, true)

    
    bios_obj.config = nil
    local ok, _ = pcall(function()
        bios_obj:update_denpendency()
    end)
    lu.assertEquals(ok, true)
    local ok, _ = pcall(function()
        bios_obj:clear_setting()
    end)
    lu.assertEquals(ok, true)

    bios_obj.config = {
        clear_setting_value = function ()
        end,
        update_current_value_by_file = function ()
        end,
        update_denpendency_after_clear = function ()
        end,
        update_setting_value_by_file = function ()
        end
    }
    local get_prop_mock = bios_obj.get_prop
    bios_obj.get_prop = function ()
    end

    local ok, _ = pcall(function()
        bios_obj:clear_setting()
    end)
    lu.assertEquals(ok, true)

    local ok, _ = pcall(function()
        bios_obj:update_denpendency()
    end)
    lu.assertEquals(ok, true)
    bios_obj.get_prop = get_prop_mock
end

function TestBiosService:test_get_pcie_silk_node_dpu_slot()
    local obj = {
        ['ComponentType'] = {
            ['value'] = function()
                return 8
            end
        },
        ['SlotID'] = {
            ['value'] = function()
                return 11
            end
        },
        ['SlotType'] = {
            ['value'] = function()
                return 'OEM'
            end
        }
    }
    lu.assertIsNil(bios_service.get_pcie_silk_node(obj, nil, nil))
end

function TestBiosService:test_check_pcie_card_silk_dpu_slot()
    local obj = {
        ['ComponentType'] = {
            ['value'] = function()
                return 8
            end
        },
        ['SlotID'] = {
            ['value'] = function()
                return 11
            end
        },
        ['SlotType'] = {
            ['value'] = function()
                return 'OEM'
            end
        }
    }
    lu.assertIsNil(bios_service.check_pcie_card_silk(obj, nil, {1}))
end

function TestBiosService:test_ipmi_set_base_os_boot_event()
    local bios_ser = bios_service.get_instance()
    local mdb_os_boot_mock = bios_ser.mdb_os_boot_mock
    bios_ser.mdb_os_boot_mock = nil
    local complete_code, manu_id = bios_ser:ipmi_set_base_os_boot_event()
    lu.assertEquals(complete_code, comp_code.CommandNotAvailable)
    lu.assertEquals(manu_id, bios_ser:get_manu_id())
    bios_ser.mdb_os_boot_mock = mdb_os_boot_mock
end