-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at:
--         http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

local log = require 'mc.logging'
local file_sec = require 'utils.file'
local utils_core = require 'utils.core'
local bios_factory = require 'factory.bios_factory'
local skynet = require 'skynet'
local utils = require 'mc.utils'

local log_collector = {}

local tree_map<const> = {
    [1] = "boot_options_service",
    [2] = "bios_service",
    [3] = "smbios_service",
    [4] = "boot_service"
}

local function get_sysid_list()
    local service = bios_factory.get_service("bios_service")
    local objs = service:get_all_objs()
    local sysid_list = {}
    for sysid, _ in pairs(objs) do
        table.insert(sysid_list, sysid)
    end

    return sysid_list
end

function log_collector._collect_bios_info(dst_path, sysid)
    local log_info = {}
    for _, v in pairs(tree_map) do
        local service = bios_factory.get_service(v)
        local dump_info = service:get_dump_info(sysid)
        if dump_info then
            table.insert(log_info, dump_info)
        end
        skynet.sleep(10)
    end

    local bios_info_path = dst_path .. '/bios_info'
    local f_log = file_sec.open_s(bios_info_path, 'w+')

    utils.safe_close_file(f_log, function ()
        if not f_log then
            error('[bios]log dump: open file fail, system_id: %s', sysid)
        end

        f_log:write(table.concat(log_info, '\r\n'))
    end)
end

function log_collector._collect_json_file(path, sysid)
    for _, v in pairs(tree_map) do
        local ser = bios_factory.get_service(v)
        ser:collect_json_file(path, sysid)
        skynet.sleep(10)
    end
end

function log_collector._start(path)
    local ok, err = pcall(function()
        local sysid_list = get_sysid_list()
        log:notice('[bios]start collect bios info')
        for _, sysid in pairs(sysid_list) do
            local dst_path = path .. '/' .. sysid
            utils.mkdir_with_parents(dst_path, utils.S_IRWXU | utils.S_IRGRP | utils.S_IXGRP)
            log_collector._collect_json_file(dst_path ,sysid)
            log_collector._collect_bios_info(dst_path, sysid)
            skynet.sleep(10)
        end
    end)
    if not ok then
        log:error('[bios]log dump: dump fail, err: %s', err)
    end
end

function log_collector.log_dump(ctx, path)
    -- 目录不存在则返回
    if not utils_core.is_dir(path) then
        log:error('[bios]log dump: the path doesn\'t exist')
        return
    end
    local ret = file_sec.check_realpath_before_open_s(path)
    if not ret then
        log:error('[bios]log dump: the path is illegal')
        return
    end
    log:notice('[bios]start to dump bios log')
    log_collector._start(path)
    log:notice('[bios]finish to dump bios log')
end

return log_collector
