#!/bin/bash
# Copyright © Huawei Technologies Co., Ltd. 2023-2024. All rights reserved.

# ========================== 全局配置 ==========================
readonly STUDIO_DIR="/usr/share/bmc_studio"
readonly SERVER_DIR="${STUDIO_DIR}/server"
readonly PORTS_CONFIG="${SERVER_DIR}/ports.conf"
readonly LOG_DIR="${HOME}/bmc_studio/log"
readonly NGINX_LOADER="/usr/share/bmc_studio/load_nginx.sh"
_trap_ran=0

# ========================== 全局变量 ==========================
declare -i USER_ID=${EUID}
declare STUDIO_PID=""
# ========================== 函数定义 ==========================

function show_usage() {
    echo $"Usage: $0 {start|stop|restart} [backend]"
}

function init_environment() {
    # 处理 sudo 执行场景
    [[ -n "${SUDO_UID}" && "${SUDO_UID}" != 0 ]] && USER_ID=${SUDO_UID}

    # 设置 Cangjie 环境
    if [[ -d "/opt/cangjie" ]]; then
        export CANGJIE_HOME="/opt/cangjie"
        export PATH="${CANGJIE_HOME}/bin:${CANGJIE_HOME}/tools/bin:${CANGJIE_HOME}/debugger/bin:${PATH}"
        export LD_LIBRARY_PATH="${CANGJIE_HOME}/runtime/lib/linux_x86_64_llvm:${LD_LIBRARY_PATH}"
    else
        echo "Warning: Cangjie environment not found at /opt/cangjie" >&2
    fi

    # 添加依赖库路径
    export LD_LIBRARY_PATH="/usr/share/bmc_studio/server:${LD_LIBRARY_PATH}"
    return 0
}

function is_wsl_environment() {
    # 综合判断 WSL 环境
    grep -qi "microsoft" /proc/version 2>/dev/null && return 0
    uname -a | grep -qi "microsoft" && return 0
    [[ -n "${WSL_DISTRO_NAME}" || -n "${WSL_INTEROP}" ]] && return 0
    grep -qi "microsoft" /sys/class/dmi/id/sys_vendor 2>/dev/null && return 0
    return 1
}

function get_frontend_port() {
    # 优先从生成的配置文件获取实时端口
    local port=$("${NGINX_LOADER}" get-frontend-port "${STUDIO_DIR}" 2>/dev/null)
    echo "${port:-0}"
}

function get_backend_port() {
    # 优先从配置文件获取实时端口
    local port=$("${NGINX_LOADER}" get-backend-port "${STUDIO_DIR}" 2>/dev/null)
    echo "${port:-0}"
}

function show_service_url() {
    local port=$(get_frontend_port)
    [[ "${port}" -le 0 ]] && { echo "Error: Failed to determine service port" >&2; return 1; }

    local url="http://localhost:${port}"
    echo "BMC Studio is running at: ${url}"

    if is_wsl_environment; then
        xdg-open "${url}" 2>/dev/null || echo "Failed to auto-open in WSL" >&2
    else
        echo "Please manually open: ${url}"
    fi
}

function start_nginx_service() {
    echo "Starting Nginx service..."
    local port=$("${NGINX_LOADER}" start "${STUDIO_DIR}")
    [[ "${port}" -gt 0 ]] || { echo "Error: Nginx failed to start" >&2; return 1; }
    return 0
}

function nginx_cleanup() {
    if [[ $_trap_ran -gt 0 ]]; then
        return
    fi
    ((_trap_ran++))

    echo ""
    echo "接收到中断信号，正在执行清理程序..." >&2

    if ! "${NGINX_LOADER}" stop "${STUDIO_DIR}"; then
        echo "清理脚本在执行 stop 时遇到问题，但这不会阻止退出。" >&2
    fi
    echo "清理完毕。脚本退出。" >&2
}

function run_server_foreground() {
    echo "Starting BMC Studio in foreground..."
    cd "${SERVER_DIR}" || return 1
    export cjHeapSize="1024MB"
    ./bmc_studio_server "$1"
}

function run_server_background() {
    echo "Starting BMC Studio in background..."
    mkdir -p "${LOG_DIR}" || return 1
    cd "${SERVER_DIR}" || return 1
    export cjHeapSize="1024MB"
    nohup ./bmc_studio_server "$1"> "${LOG_DIR}/bmc_studio.log" 2>&1 &
}

function start_service() {
    has_quiet=0
    has_backend=0
    for arg in "$@"; do
        case "$arg" in
            "quiet") has_quiet=1 ;;
            "backend") has_backend=1 ;;
        esac
    done
    # 检查现有进程
    if pgrep -u "${USER_ID}" -f bmc_studio_server >/dev/null; then
        echo "Service is already running"
        if [[ $has_quiet -eq 0 ]]; then
            show_service_url
        fi
        return 1
    fi
    # 启动依赖服务
    start_nginx_service || return 1
    # 打开studio页面
    if [[ $has_quiet -eq 0 ]]; then
        show_service_url
    fi
    # 运行主服务
    if [[ $has_backend -eq 1 ]]; then
        trap 'nginx_cleanup' INT TERM
        run_server_background $(get_backend_port)|| { echo "Failed to start background service" >&2; return 1; }
    else
        trap 'nginx_cleanup' EXIT INT TERM
        run_server_foreground $(get_backend_port)|| { echo "Failed to start foreground service" >&2; return 1; }
    fi
}

function stop_service() {
    echo "Stopping nginx process..."
    "${NGINX_LOADER}" stop "${STUDIO_DIR}" 2>/dev/null
    local pids=$(pgrep -u "${USER_ID}" -f bmc_studio_server)
    [[ -z "${pids}" ]] && { echo "Service is not running"; return 0; }
    echo "Stopping BMC Studio (PID: ${pids})..."
    kill -9 ${pids} 2>/dev/null
    return $?
}

function restart_service() {
    stop_service || return 1
    sleep 1  # 等待进程完全退出
    start_service "$@"
}

# ========================== 主执行逻辑 ==========================
init_environment || exit 1

case "$1" in
    "start")
        start_service "$@"
        ;;
    "stop")
        stop_service
        ;;
    "restart")
        restart_service "$@"
        ;;
    *)
        show_usage
        exit 2
        ;;
esac

exit $?
