#!/usr/bin/env python
# -*- coding: utf-8 -*-
# Copyright (c) Huawei Technologies Co., Ltd. 2024. All rights reserved.
from typing import Optional


class MessageBuilder:
    @staticmethod
    def format_message(msg: str, args: Optional[list] = None) -> str:
        if args is None:
            return msg
        for i, arg in enumerate(args):
            if str(i + 1) + "%" not in msg:
                raise Exception("""Failed to format message, placeholder not found. 
                                Please check rule definition in file grammar_check.json.""")
            msg = msg.replace(str(i + 1) + "%", arg)
        return msg


class GrammarIssue:
    def __init__(self, rule_id: str, config: dict, model: str, file_path: str, file_type: str):
        self.rule_id = rule_id
        self.config = config
        self.model = model

        self.error_msg = ""
        self.repair_suggestion = ""
        self.file_path = file_path
        self.file_type = file_type
        self.is_complex_rule = False
        self.item_id = 0
        self.need_append = False

    def err(self, template: str, args: Optional[list] = None) -> 'GrammarIssue':
        msg = MessageBuilder.format_message(template, args)
        if self.error_msg:
            self.item_id += 1
            msg = f"\n{self.item_id}、{msg}"
            self.need_append = True
        self.error_msg += msg
        return self

    def suggest(self, template: str, args: Optional[list] = None) -> 'GrammarIssue':
        msg = MessageBuilder.format_message(template, args)
        if self.repair_suggestion:
            msg = f"\n{self.item_id}、{msg}"
        self.repair_suggestion += msg
        return self

    def format_issue(self, error_obj: str = "") -> dict:
        return {
            "ruleId": self.rule_id,
            "rule": self.config.get("name", ""),
            "model": self.model,
            "category": self.config.get("category", ""),
            "filepath": self.file_path,
            "filetype": self.file_type,
            "alarmLevel": self.config.get("alarmLevel", "严重"), 
            "errorObject": error_obj,
            "errorMessage": self.error_msg,
            "repairSuggestion": self.repair_suggestion,
            "ciEnabled": self.config.get("ciEnabled", False)
        }
