#!/bin/bash
# Copyright © Huawei Technologies Co., Ltd. 2023-2024. All rights reserved.

SERVER_DIR="/usr/share/bmc_studio/server"
PORTS_CONFIG="${SERVER_DIR}/ports.conf"
g_user_id="${EUID}"

function usage()
{
    echo $"Usage: $0 {start|stop|restart}"
}

function start_nginx()
{
    NGINX_PID=$(pidof nginx)
    if [ -n "${NGINX_PID}" ]; then
        return 1
    fi
    
    sudo service nginx start
    if [ $? -ne 0 ]; then
        echo "nginx start failed, you can start manually"
        return 1
    fi

    return 0
}

function print_url()
{
    local port=12888
    if [[ "${g_user_id}" != "0" ]] && [[ -f "${PORTS_CONFIG}" ]]; then
        local user_port_config=$(grep -oP "^\d+\s+${g_user_id}\s+\d+\s+\d+$" "${PORTS_CONFIG}" || :)
        if [[ -n "${user_port_config}" ]]; then
            port=$(echo "${user_port_config}" | awk '{print $3}')
        else
            echo "Failed to get port config"
        fi
    fi
    local url="http://localhost:${port}"
    echo "BMC Studio starting at ${url}"
    local is_wsl=false
    # 检查 /proc/version 或 uname 输出
    if { grep -qi "microsoft\|wsl" /proc/version || uname -a | grep -qi "microsoft"; }; then
        is_wsl=true
    fi
    # 检查环境变量
    if [[ -n "$WSL_DISTRO_NAME" || -n "$WSL_INTEROP" ]]; then
        is_wsl=true
    fi
    # 检查系统供应商信息
    if grep -qi "microsoft" /sys/class/dmi/id/sys_vendor 2>/dev/null; then
        is_wsl=true
    fi
    if [ "$is_wsl" = true ]; then
        xdg-open "${url}" || :
    else
        echo "Not running in WSL environment, please open ${url} manually"
    fi
}

function start()
{
    if pgrep -u "${g_user_id}" -f bmc_studio_server; then
        echo "bmc_studio service is already running"
        return 1 
    fi

    start_nginx
    print_url
    cd "${SERVER_DIR}"
    export cjHeapSize=1024MB
    if [ "$2" == "backend" ]; then
        mkdir -p ~/bmc_studio/log
        nohup ./bmc_studio_server > ~/bmc_studio/log/bmc_studio.log 2>&1 &
    else
        ./bmc_studio_server
    fi

    return 0
}

function stop()
{
    STUDIO_PID=$(pgrep -u "${g_user_id}" -f bmc_studio_server)
    if [ -z "${STUDIO_PID}" ]; then
       echo "bmc_studio service is not running"
       return 1
    fi

    kill -9 ${STUDIO_PID} > /dev/null

    echo "bmc_studio stop successfully"
    return 0
}

function restart()
{
    stop

    start "$@"
    if [ $? -ne 0 ]; then
        echo "Failed to restart bmc_studio server, failed stage is start"
        return 1
    fi

    return 0
}

function env_init()
{
    if [[ -n "${SUDO_UID}" ]] && [[ "${SUDO_UID}" != 0 ]]; then
        g_user_id="${SUDO_UID}"
    fi
    if [[ ! -d /opt/cangjie ]]; then
        return 1
    fi
    export CANGJIE_HOME=/opt/cangjie
    export PATH=$CANGJIE_HOME/bin:$CANGJIE_HOME/tools/bin:$CANGJIE_HOME/debugger/bin:$PATH:
    export LD_LIBRARY_PATH=$CANGJIE_HOME/runtime/lib/linux_x86_64_llvm:/usr/share/bmc_studio/server:${LD_LIBRARY_PATH}
    return 0
}

env_init

case "$1" in
    "stop")
        stop
        exit $?
        ;;
    "start")
        start "$@"
        exit $?
        ;;
    "restart")
        restart "$@"
        exit $?
        ;;
    *)
        usage
        RET_VAL=2
        ;;
esac