//  Copyright (c) 2024 Huawei Technologies Co., Ltd.
//  openUBMC is licensed under Mulan PSL v2.
//  You can use this software according to the terms and conditions of the Mulan PSL v2.
//  You may obtain a copy of Mulan PSL v2 at:
//        #  http://license.coscl.org.cn/MulanPSL2
//  THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
//  EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
//  MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
//  See the Mulan PSL v2 for more details.
import { traduction } from '@/utils/language';
import { IHealthState, IOptions, ISwitch, IRadio, IStorageRadio } from '../storage-interface';
import { StorageStaticData, DriveStaticModel } from '../static-data';
import { RaidType } from '../index';
import { getStoreData } from '@/utils/composition';
import { computed } from 'vue';

const fdmSupport = computed(() => getStoreData('loct', 'fdmspt'));
export class Driver {
  private interType!: [string, string];
  private healthState: [string, IHealthState];
  private manufacturer!: [string, string];
  private model!: [string, string];
  private serialNumber!: [string, string];
  private firmwareVersion!: [string, string];
  private mediaType!: [string, string];
  private temperature!: [string, string];
  private firmwareStatus: [string, IOptions | null];
  private sasAddress0!: [string, string];
  private sasAddress1!: [string, string];
  private capacity!: [string, string];
  private capableSpeed: [string, string];
  private negotiatedSpeed: [string, string];
  private powerState: [string, string];
  private hotspareType: [string, IRadio | null];
  private construction!: [string, string];
  private patrolState?: [string, any];
  private indicator: [string, ISwitch | null];
  private powerSupply: [string, string];
  private wearRate: [string, string];
  private partNum: [string, string];
  private resourceOwner: [string, string];
  private diskSpeed: [string, number];
  private isEPD: boolean;
  private bootEnabled: [string, IStorageRadio | null];
  private bootPriority: IOptions;
  private patrolStateEn?: any;
  private estimatedRemainingLifespan?: [string, string];
  private position: [string, string];
  private presence?: [string, string | null];

  constructor(
    interType: string,
    healthState: string,
    manufacturer: string,
    model: string,
    serialNumber: string,
    firmwareVersion: string,
    mediaType: string,
    temperature: string,
    firmwareStatus: string,
    sasAddress0: string,
    sasAddress1: string,
    capacity: string,
    capableSpeed: string,
    negotiatedSpeed: string,
    powerState: string,
    hotspareType: string,
    construction: string,
    patrolState: string,
    indicator: string,
    powerSupply: string,
    wearRate: string,
    partNum: string,
    resourceOwner: string,
    isRaid: boolean,
    diskSpeed: number,
    isEPD: boolean,
    raidType: string,
    bootEnabled: boolean,
    bootPriority: string,
    estimatedRemainingLifespan: string,
    position: string,
    presence: number,
  ) {
    this.setStatus(interType, manufacturer, model, serialNumber,
      firmwareVersion, mediaType, temperature, sasAddress0, sasAddress1, capacity);
    const state = StorageStaticData.getState(healthState);
    this.healthState = ['NET_HEALTH_STATUS', state];
    this.firmwareStatus = ['FDM_FIRMWARE_STATUS', DriveStaticModel.getInstance().getFirmwareStatus(firmwareStatus)];
    this.capableSpeed = ['FDM_SUPPORTED_RATE', capableSpeed];
    this.negotiatedSpeed = ['FDM_NEGOTIATED_SPEEDGBS', negotiatedSpeed];
    this.powerState = ['FDM_POWER_STATUS', powerState];
    this.hotspareType = ['FDM_HOT_STATUS', DriveStaticModel.getInstance().getHotspareType(hotspareType)];
    this.indicator = ['STORE_LOCATION_STATUS', DriveStaticModel.getInstance().getIndicator(indicator)];
    this.powerSupply = ['FDM_TOTAL_POWER_TIME', powerSupply];
    this.wearRate = ['WEAR_RATE', wearRate];
    this.partNum = ['OTHER_PART_NUMBER', partNum];
    this.resourceOwner = ['COMMON_RESOURCE_OWNER', resourceOwner];
    this.diskSpeed = ['FAN_HEAT_ROTATIONAL_SPEED', diskSpeed];    
    if (fdmSupport.value) {
      this.estimatedRemainingLifespan = ['ESTIMATED_LIFESPAN', estimatedRemainingLifespan];
    }
    this.bootEnabled = ['STORE_BOOT_DISK', DriveStaticModel.getIsBootDisk(bootEnabled)];    
    this.bootPriority = DriveStaticModel.getInstance().getBootPriority(bootPriority);
    this.position = ['OTHER_POSITION', position];
    if (presence !== null) {
      this.presence = ['STORE_IN_POSITION', presence === 1 ? traduction('COMMON_YES') : traduction('COMMON_NO')];
    }
    // 只有Raid卡下面才会显示的属性
    if (isRaid) {
      this.construction = ['FDM_CONSTRUCTION_STATE', construction];
      this.patrolState = ['STORE_PATROL_STATE', raidType === RaidType.BRCM || raidType === RaidType.ARIES ? traduction(this.setPatrolState(patrolState)) : null];
      this.patrolStateEn = this.setPatrolState(patrolState);
    }
    this.isEPD = isEPD;
  }

  private setPatrolState(key: string): string {
    let str = '';
    if (key === 'DoneOrNotPatrolled') {
      str = 'FDM_STOPPED';
    } else if (key === 'Patrolling') {
      str = 'FDM_CHECKING';
    }
    return str;
  }

  get getIndicator(): ISwitch | null {
    return this.indicator[1];
  }

  get getHotspareType(): IRadio | null {
    return this.hotspareType[1];
  }

  get getFirmwareStatus(): IOptions | null {
    return this.firmwareStatus[1];
  }

  get getBootDisk(): IStorageRadio | null {
    return this.bootEnabled[1];
  }

  get getIsEPD(): boolean {
    return this.isEPD;
  }

  get getMediaType(): string {
    return this.mediaType[1];
  }

  get getPatrolState(): IStorageRadio | null {        
    return DriveStaticModel.getPatrol(StorageStaticData.getPatrolState(this.patrolState ? this.patrolStateEn : null));
  }

  get getBootPriority(): IOptions {
    return this.bootPriority;
  }

  public setStatus(
    interType: string,
    manufacturer: string,
    model: string,
    serialNumber: string,
    firmwareVersion: string,
    mediaType: string,
    temperature: string,
    sasAddress0: string,
    sasAddress1: string,
    capacity: string,
  ): void {
    this.interType = ['FDM_INTERFACE_TYPE', interType];
    this.manufacturer = ['COMMON_MANUFACTURER', manufacturer];
    this.model = ['FDM_ALARM_MODEL', model];
    this.serialNumber = ['COMMON_SERIALNUMBER', serialNumber];
    this.firmwareVersion = ['STORE_FIRMWARE_VERSION', firmwareVersion];
    this.mediaType = ['FDM_MEDIA_TYPE', mediaType];
    this.temperature = ['FDM_TEMPERATURE', temperature];
    this.sasAddress0 = ['FDM_SAS_ADDRESS0', sasAddress0];
    this.sasAddress1 = ['FDM_SAS_ADDRESS1', sasAddress1];
    this.capacity = ['COMMON_CAPACITY', capacity];
  }
}
