-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at: http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.


require 'skynet.manager'
local skynet = require 'skynet'
local log = require 'mc.logging'
local mc_context = require 'mc.context'
local json = require 'cjson'
local lu = require 'luaunit'
local mdb = require 'mc.mdb'
local prop = require 'basic_cooling.define.cooling_properties'
local enums = require 'basic_cooling.define.cooling_enums'
local mc_utils = require 'mc.utils'

local service = 'bmc.kepler.thermal_mgmt'
local path = '/bmc/kepler/thermal_mgmt/MicroComponent'
local intf = 'bmc.kepler.MicroComponent.ConfigManage'
local smart_cooling_map = {
    [16] = 1,
    [17] = 3,
    [18] = 2
}
local context = mc_context.new('IT', 'Admin', '127.0.0.1')

local test_business_config = {}

local function test_init(bus)
    local c_requirement_70_path =
        '/bmc/kepler/Systems/1/CoolingRequirement/CoolingRequirement_1_70_0101'
    local c_requirement_70_obj
    local ok
    ok, c_requirement_70_obj = pcall(mdb.get_object, bus, c_requirement_70_path,
        'bmc.kepler.Systems.CoolingRequirement')
    assert(ok)

    return {
        ['req_70'] = c_requirement_70_obj
    }
end

local function test_cooling_and_policy1_class_import(bus, service, path, intf)
    local data = json.encode({
        ConfigData = {
            Cooling = {
                CustomCpuCoremTobj = { Value = 80, Import = true},
                CustomDiskTobj = { Value = 80, Import = true},
                CustomMemoryTobj = { Value = 80, Import = true},
                CustomVRDTobj = { Value = 80, Import = true},
                CustomVDDQTobj = { Value = 80, Import = true},
                CustomNpuAiCoreTobj = { Value = 80, Import = true},
                CustomNpuHbmTobj = { Value = 80, Import = true},
                CustomSocBoardInletTobj = { Value = 80, Import = true},
                CustomSocBoardOutletTobj = { Value = 80, Import = true}
            }
        }
    })
    lu.assertErrorMsgContains("CollectingConfigurationErrorDesc", function()
        bus:call(service, path, intf, 'Import', 'a{ss}ss', context, data, 'configuration')
    end)
    data = json.encode({
        ConfigData = {
            Cooling = {
                Mode = { Value = 1, Import = true },
                Level = { Value = 20, Import = true },
                SmartCoolingMode = { Value = 17, Import = true },
                CustomOutletTobj = { Value = 30, Import = true }
            },
            Policy1Class = {
                EnvRangeRdL = { Value = { 4, 0, 20, 30, 40, 20, 50, 70, 80, 90 }, Import = true }
            }
        }
    })
    bus:call(service, path, intf, 'Import', 'a{ss}ss', context, data, 'configuration')
end

local function test_array_ascend(bus, service, path, intf)
    local data = json.encode({
        ConfigData = {
            CoolingPolicy = {
                [1] = {
                    Id = { Value = 3, Import = false },
                    TemperatureRangeCelsiusLow = { Value = { -127, 26, 26 }, Import = true },
                    TemperatureRangeCelsiusHigh = { Value = { 25, 26, 127 }, Import = true },
                    PWMRangeLow = { Value = { 50, 51, 52 }, Import = true },
                    PWMRangeHigh = { Value = { 51, 52, 53 }, Import = true }
                }
            }
        }
    })
    lu.assertErrorMsgContains("CollectingConfigurationErrorDesc", function()
        bus:call(service, path, intf, 'Import', 'a{ss}ss', context, data, 'configuration')
    end)

    data = json.encode({
        ConfigData = {
            CoolingPolicy = {
                [1] = {
                    Id = { Value = 3, Import = false },
                    TemperatureRangeCelsiusLow = { Value = { -127, 25, 26 }, Import = true },
                    TemperatureRangeCelsiusHigh = { Value = { 25, 26, 127 }, Import = true },
                    PWMRangeLow = { Value = { 50, 51, 52 }, Import = true },
                    PWMRangeHigh = { Value = { 51, 55, 53 }, Import = true }
                }
            }
        }
    })
    lu.assertErrorMsgContains("CollectingConfigurationErrorDesc", function()
        bus:call(service, path, intf, 'Import', 'a{ss}ss', context, data, 'configuration')
    end)
end

local function test_array_length(bus, service, path, intf)
    local data = json.encode({
        ConfigData = {
            CoolingPolicy = {
                [1] = {
                    Id = { Value = 3, Import = false },
                    TemperatureRangeCelsiusLow = { Value = {}, Import = true },
                    TemperatureRangeCelsiusHigh = { Value = { 25, 26, 127 }, Import = true },
                    PWMRangeLow = { Value = { 50, 51, 52 }, Import = true },
                    PWMRangeHigh = { Value = { 51, 52, 53 }, Import = true }
                }
            }
        }
    })
    lu.assertErrorMsgContains("CollectingConfigurationErrorDesc", function()
        bus:call(service, path, intf, 'Import', 'a{ss}ss', context, data, 'configuration')
    end)

    local pwm_table = {}
    for i = 1, 64, 1 do
        table.insert(pwm_table, 20 + i)
    end
    data = json.encode({
        ConfigData = {
            CoolingPolicy = {
                [1] = {
                    Id = { Value = 3, Import = false },
                    TemperatureRangeCelsiusLow = { Value = { -127, 25, 26 }, Import = true },
                    TemperatureRangeCelsiusHigh = { Value = { 25, 26, 127 }, Import = true },
                    PWMRangeLow = { Value = { 50, 51, 52 }, Import = true },
                    PWMRangeHigh = { Value = pwm_table, Import = true }
                }
            }
        }
    })
    lu.assertErrorMsgContains("CollectingConfigurationErrorDesc", function()
        bus:call(service, path, intf, 'Import', 'a{ss}ss', context, data, 'configuration')
    end)
end

local function test_temp_array_digit(bus, service, path, intf)
    local data = json.encode({
        ConfigData = {
            CoolingPolicy = {
                [1] = {
                    Id = { Value = 3, Import = false },
                    TemperatureRangeCelsiusLow = { Value = { -127, 25, 26, 27 }, Import = true },
                    TemperatureRangeCelsiusHigh = { Value = { 25, 26, 27 }, Import = true },
                    PWMRangeLow = { Value = { 50, 51, 52 }, Import = true },
                    PWMRangeHigh = { Value = { 51, 52, 53 }, Import = true }
                }
            }
        }
    })
    lu.assertErrorMsgContains("CollectingConfigurationErrorDesc", function()
        bus:call(service, path, intf, 'Import', 'a{ss}ss', context, data, 'configuration')
    end)

    data = json.encode({
        ConfigData = {
            CoolingPolicy = {
                [1] = {
                    Id = { Value = 3, Import = false },
                    TemperatureRangeCelsiusLow = { Value = { 24, 25, 26, 27 }, Import = true },
                    TemperatureRangeCelsiusHigh = { Value = { 25, 26, 127 }, Import = true },
                    PWMRangeLow = { Value = { 50, 51, 52 }, Import = true },
                    PWMRangeHigh = { Value = { 51, 52, 53 }, Import = true }
                }
            }
        }
    })
    lu.assertErrorMsgContains("CollectingConfigurationErrorDesc", function()
        bus:call(service, path, intf, 'Import', 'a{ss}ss', context, data, 'configuration')
    end)
end

local function test_high_and_low_array(bus, service, path, intf)
    local data = json.encode({
        ConfigData = {
            CoolingPolicy = {
                [1] = {
                    Id = { Value = 3, Import = false },
                    TemperatureRangeCelsiusLow = { Value = { -127, 25, 26 }, Import = true },
                    TemperatureRangeCelsiusHigh = { Value = { 25, 26, 27, 127 }, Import = true },
                    PWMRangeLow = { Value = { 50, 51, 52 }, Import = true },
                    PWMRangeHigh = { Value = { 51, 52, 53 }, Import = true }
                }
            }
        }
    })
    lu.assertErrorMsgContains("CollectingConfigurationErrorDesc", function()
        bus:call(service, path, intf, 'Import', 'a{ss}ss', context, data, 'configuration')
    end)

    data = json.encode({
        ConfigData = {
            CoolingPolicy = {
                [1] = {
                    Id = { Value = 3, Import = false },
                    TemperatureRangeCelsiusLow = { Value = { -127, 25, 26 }, Import = true },
                    TemperatureRangeCelsiusHigh = { Value = { 25, 26, 127 }, Import = true },
                    PWMRangeLow = { Value = { 50, 51, 52, 53 }, Import = true },
                    PWMRangeHigh = { Value = { 51, 52, 53 }, Import = true }
                }
            }
        }
    })
    lu.assertErrorMsgContains("CollectingConfigurationErrorDesc", function()
        bus:call(service, path, intf, 'Import', 'a{ss}ss', context, data, 'configuration')
    end)
end

local function test_temp_and_speed_array(bus, service, path, intf)
    local data = json.encode({
        ConfigData = {
            CoolingPolicy = {
                [1] = {
                    Id = { Value = 3, Import = false },
                    TemperatureRangeCelsiusLow = { Value = { -127, 25, 26, 27 }, Import = true },
                    TemperatureRangeCelsiusHigh = { Value = { 25, 26, 27, 127 }, Import = true },
                    PWMRangeLow = { Value = { 50, 51, 52 }, Import = true },
                    PWMRangeHigh = { Value = { 51, 52, 53 }, Import = true }
                }
            }
        }
    })
    lu.assertErrorMsgContains("CollectingConfigurationErrorDesc", function()
        bus:call(service, path, intf, 'Import', 'a{ss}ss', context, data, 'configuration')
    end)

    data = json.encode({
        ConfigData = {
            CoolingPolicy = {
                [1] = {
                    Id = { Value = 3, Import = false },
                    TemperatureRangeCelsiusHigh = { Value = { 25, 26, 127 }, Import = true },
                    PWMRangeLow = { Value = { 50, 51, 52 }, Import = true },
                    PWMRangeHigh = { Value = { 51, 52, 53 }, Import = true }
                }
            }
        }
    })
    lu.assertErrorMsgContains("CollectingConfigurationErrorDesc", function()
        bus:call(service, path, intf, 'Import', 'a{ss}ss', context, data, 'configuration')
    end)
end

local function test_cooling_policy_import(bus, service, path, intf)
    test_array_ascend(bus, service, path, intf)
    test_array_length(bus, service, path, intf)
    test_temp_array_digit(bus, service, path, intf)
    test_high_and_low_array(bus, service, path, intf)
    test_temp_and_speed_array(bus, service, path, intf)
end

local function test_on_import(bus)
    log:notice('test on import start')
    test_cooling_and_policy1_class_import(bus, service, path, intf)
    test_cooling_policy_import(bus, service, path, intf)
    log:notice('test on import end')
end

local function test_cooling_and_policy1_class_export(data)
    assert(data)
    assert(data.ConfigData)
    assert(data.ConfigData.Cooling)
    assert(data.ConfigData.Cooling.Mode == 1, 'actual: '.. data.ConfigData.Cooling.Mode)
    assert(data.ConfigData.Cooling.Level == 20, 'actual: '.. data.ConfigData.Cooling.Level)
    assert(data.ConfigData.Cooling.SmartCoolingMode == 17, 'actual: '.. data.ConfigData.Cooling.Level)
    assert(data.ConfigData.Cooling.CustomCpuCoremTobjRangeMin == 50,
        'actual: ' .. data.ConfigData.Cooling.CustomCpuCoremTobjRangeMin)
    assert(data.ConfigData.Cooling.CustomCpuCoremTobjRangeMax == 95,
        'actual: ' .. data.ConfigData.Cooling.CustomCpuCoremTobjRangeMax)
    assert(table.concat(data.ConfigData.Cooling.CustomInletSpeedRange, ' ') == '20 100',
        'actual: ' .. table.concat(data.ConfigData.Cooling.CustomInletSpeedRange, ' '))
    
    assert(data.ConfigData.Policy1Class)
    assert(data.ConfigData.Policy1Class.EnvRangeRdL[1] == 4)
    assert(data.ConfigData.Policy1Class.EnvRangeRdL[2] == 0)
    assert(data.ConfigData.Policy1Class.EnvRangeRdL[3] == 20)
    assert(data.ConfigData.Policy1Class.EnvRangeRdL[4] == 30)
    assert(data.ConfigData.Policy1Class.EnvRangeRdL[5] == 40)
    assert(data.ConfigData.Policy1Class.EnvRangeRdL[6] == 20)
    assert(data.ConfigData.Policy1Class.EnvRangeRdL[7] == 50)
    assert(data.ConfigData.Policy1Class.EnvRangeRdL[8] == 70)
    assert(data.ConfigData.Policy1Class.EnvRangeRdL[9] == 80)
    assert(data.ConfigData.Policy1Class.EnvRangeRdL[10] == 90)
end

local function test_on_export(bus)
    log:notice('test on export start')
    local ret = bus:call(service, path, intf, 'Export', 'a{ss}s', context, 'configuration')
    assert(ret)
    local data = json.decode(ret)
    test_cooling_and_policy1_class_export(data)
    log:notice('test on export end')
end

local function test_import_target_temp(bus)
    local data
    data = json.encode({
        ConfigData = {
            CoolingRequirement = {
                [1] = {
                    Id =  {Value = 71, Import = false },
                    -- 目标调速值CSR配置为45
                    TargetTemperatureCelsius = { Value = 46, Import = true }
                }
            }
        }
    })
    lu.assertErrorMsgContains("CollectingConfigurationErrorDesc", function()
        bus:call(service, path, intf, 'Import', 'a{ss}ss', context, data, 'configuration')
    end)

    data = json.encode({
        ConfigData = {
            CoolingRequirement = {
                [1] = {
                    Id =  {Value = 71, Import = false },
                    -- 目标调速值CSR配置为45
                    TargetTemperatureCelsius = { Value = -1, Import = true }
                }
            }
        }
    })
    lu.assertErrorMsgContains("CollectingConfigurationErrorDesc", function()
        bus:call(service, path, intf, 'Import', 'a{ss}ss', context, data, 'configuration')
    end)

    -- 70温度点支持SmartCooling,导入目标温度值会直接报错
    data = json.encode({
        ConfigData = {
            CoolingRequirement = {
                [1] = {
                    Id =  {Value = 70, Import = false },
                    -- 目标调速值CSR配置为[45, 42, 48]
                    TargetTemperatureCelsius = { Value = 46, Import = true }
                }
            }
        }
    })
    lu.assertErrorMsgContains("CollectingConfigurationErrorDesc", function()
        bus:call(service, path, intf, 'Import', 'a{ss}ss', context, data, 'configuration')
    end)
end

local function test_import_threshold_temp(bus)
    local data
    data = json.encode({
        ConfigData = {
            CoolingRequirement = {
                [1] = {
                    Id =  {Value = 70, Import = false },
                    -- 全速阈值CSR配置为70
                    ThresholdTemperatureCelsius = { Value = 71, Import = true }
                }
            }
        }
    })
    lu.assertErrorMsgContains("CollectingConfigurationErrorDesc", function()
        bus:call(service, path, intf, 'Import', 'a{ss}ss', context, data, 'configuration')
    end)

    data = json.encode({
        ConfigData = {
            CoolingRequirement = {
                [1] = {
                    Id =  {Value = 70, Import = false },
                    -- 全速阈值CSR配置为70
                    ThresholdTemperatureCelsius = { Value = -1, Import = true }
                }
            }
        }
    })
    lu.assertErrorMsgContains("CollectingConfigurationErrorDesc", function()
        bus:call(service, path, intf, 'Import', 'a{ss}ss', context, data, 'configuration')
    end)
end

local function test_import_smart_cooling_tar(bus)
    local data
    data = json.encode({
        ConfigData = {
            CoolingRequirement = {
                [1] = {
                    Id =  {Value = 70, Import = false },
                    -- 目标值CSR配置为[45, 42, 48]
                    SmartCoolingTargetTemperatureCelsius = { Value = { 46, 42, 48 }, Import = true }
                }
            }
        }
    })
    lu.assertErrorMsgContains("CollectingConfigurationErrorDesc", function()
        bus:call(service, path, intf, 'Import', 'a{ss}ss', context, data, 'configuration')
    end)

    -- 71温度点不支持SmartCooling,导入应该报错
    data = json.encode({
        ConfigData = {
            CoolingRequirement = {
                [1] = {
                    Id =  {Value = 71, Import = false },
                    SmartCoolingTargetTemperatureCelsius = { Value = { 46, 42, 48 }, Import = true }
                }
            }
        }
    })
    lu.assertErrorMsgContains("CollectingConfigurationErrorDesc", function()
        bus:call(service, path, intf, 'Import', 'a{ss}ss', context, data, 'configuration')
    end)
end

local function test_import_failed_value(bus)
    local data
    data = json.encode({
        ConfigData = {
            CoolingRequirement = {
                [1] = {
                    Id =  {Value = 70, Import = false },
                    -- CSR配置为80
                    FailoverPercent = { Value = 101, Import = true }
                }
            }
        }
    })
    lu.assertErrorMsgContains("CollectingConfigurationErrorDesc", function()
        bus:call(service, path, intf, 'Import', 'a{ss}ss', context, data, 'configuration')
    end)

    data = json.encode({
        ConfigData = {
            CoolingRequirement = {
                [1] = {
                    Id =  {Value = 71, Import = false },
                    FailoverPercent = { Value = 79, Import = true }
                }
            }
        }
    })
    lu.assertErrorMsgContains("CollectingConfigurationErrorDesc", function()
        bus:call(service, path, intf, 'Import', 'a{ss}ss', context, data, 'configuration')
    end)
end

local function test_import_cooling_requirement(bus)
    -- 正常值导入导出校验
    local data = json.encode({
        ConfigData = {
            CoolingRequirement = {
                [1] = {
                    Id =  {Value = 70, Import = false },
                    -- 目标值CSR配置为[45, 42, 48]
                    SmartCoolingTargetTemperatureCelsius = { Value = { 44, 41, 47 }, Import = true }
                }
            }
        }
    })
    bus:call(service, path, intf, 'Import', 'a{ss}ss', context, data, 'configuration')

    local ret = bus:call(service, path, intf, 'Export', 'a{ss}s', context, 'configuration')
    assert(ret)
    data = json.decode(ret)
    local req_74_smartcooling_temp_compare_flag = false
    for _, obj in pairs(data.ConfigData.CoolingRequirement) do
        if obj.Id == 70 then
            -- 带有SmartCooling数组的对象不会导出目标温度，需要根据资源树的属性进行对比
            assert(mc_utils.table_compare(obj.SmartCoolingTargetTemperatureCelsius, { 44, 41, 47}))
            req_74_smartcooling_temp_compare_flag = true
        end
    end
    assert(req_74_smartcooling_temp_compare_flag)
    -- 正常值导入导出校验
    data = json.encode({
        ConfigData = {
            CoolingRequirement = {
                [1] = {
                    Id =  {Value = 71, Import = false },
                    TargetTemperatureCelsius = { Value = 44, Import = true },
                    ThresholdTemperatureCelsius = { Value = 69, Import = true },
                    FailoverPercent = { Value = 90, Import = true }
                }
            }
        }
    })
    bus:call(service, path, intf, 'Import', 'a{ss}ss', context, data, 'configuration')
    ret = bus:call(service, path, intf, 'Export', 'a{ss}s', context, 'configuration')
    assert(ret)
    data = json.decode(ret)
    for _, obj in pairs(data.ConfigData.CoolingRequirement) do
        if obj.Id == 71 then
            assert(obj.TargetTemperatureCelsius == 44)
            assert(obj.ThresholdTemperatureCelsius == 69)
            assert(obj.FailoverPercent == 90)
        end
    end
end

local function test_import_hysteresis(bus)
    local data = json.encode({
        ConfigData = {
            CoolingPolicy = {
                [1] = {
                    Id =  {Value = 71, Import = false },
                    TemperatureRangeCelsiusLow = { Value = { -127, 25, 30, 35, 40 }, Import = false },
                    TemperatureRangeCelsiusHigh = { Value = { 25, 30, 35, 40, 127 }, Import = false },
                    PWMRangeLow = { Value = { 45, 60, 70, 80, 100 }, Import = false },
                    PWMRangeHigh = { Value = { 45, 60, 70, 80, 100 }, Import = false },
                    Hysteresis = { Value = 6, Import = true }
                }
            }
        }
    })
    lu.assertErrorMsgContains("CollectingConfigurationErrorDesc", function()
        bus:call(service, path, intf, 'Import', 'a{ss}ss', context, data, 'configuration')
    end)
end

local function test_import_custom_outlet_tar(bus)
    -- Range范围为[30,60]
    local data = json.encode({
        ConfigData = {
            Cooling = {
                CustomOutletTobj = { Value = 29, Import = true}
            }
        }
    })
    lu.assertErrorMsgContains("CollectingConfigurationErrorDesc", function()
        bus:call(service, path, intf, 'Import', 'a{ss}ss', context, data, 'configuration')
    end)
    data = json.encode({
        ConfigData = {
            Cooling = {
                CustomOutletTobj = { Value = 61, Import = true}
            }
        }
    })
    lu.assertErrorMsgContains("CollectingConfigurationErrorDesc", function()
        bus:call(service, path, intf, 'Import', 'a{ss}ss', context, data, 'configuration')
    end)
    data = json.encode({
        ConfigData = {
            Cooling = {
                CustomOutletTobj = { Value = 50, Import = true}
            }
        }
    })
    bus:call(service, path, intf, 'Import', 'a{ss}ss', context, data, 'configuration')
    -- 导出校验
    local ret = bus:call(service, path, intf, 'Export', 'a{ss}s', context, 'configuration')
    assert(ret)
    data = json.decode(ret)
    assert(data.ConfigData.Cooling.CustomOutletTobj == 50)
    -- 数据恢复
    data = json.encode({
        ConfigData = {
            Cooling = {
                CustomOutletTobj = { Value = 52, Import = true}
            }
        }
    })
    bus:call(service, path, intf, 'Import', 'a{ss}ss', context, data, 'configuration')
    -- 导出校验
    local ret = bus:call(service, path, intf, 'Export', 'a{ss}s', context, 'configuration')
    assert(ret)
    data = json.decode(ret)
    assert(data.ConfigData.Cooling.CustomOutletTobj == 52)
end

local function test_invalid_cooling_policy_import(bus)
    -- 存在温度点下的转速计算不出来的曲线会报错
    local data = json.encode({
        ConfigData = {
            CoolingPolicy = {
                [1] = {
                    Id =  {Value = 71, Import = false },
                    TemperatureRangeCelsiusLow = { Value = { -127, 26, 30, 35, 40 }, Import = true },
                    TemperatureRangeCelsiusHigh = { Value = { 25, 30, 35, 40, 127 }, Import = true },
                    PWMRangeLow = { Value = { 45, 60, 70, 80, 100 }, Import = true },
                    PWMRangeHigh = { Value = { 45, 60, 70, 80, 100 }, Import = true }
                }
            }
        }
    })
    lu.assertErrorMsgContains("CollectingConfigurationErrorDesc", function()
        bus:call(service, path, intf, 'Import', 'a{ss}ss', context, data, 'configuration')
    end)
    -- 转速与原来相比下降趋势的曲线导入报错
    data = json.encode({
        ConfigData = {
            CoolingPolicy = {
                [1] = {
                    Id =  {Value = 71, Import = false },
                    TemperatureRangeCelsiusLow = { Value = { -127, 25, 30, 35, 40 }, Import = true },
                    TemperatureRangeCelsiusHigh = { Value = { 25, 30, 35, 40, 127 }, Import = true },
                    PWMRangeLow = { Value = { 40, 60, 70, 80, 100 }, Import = true },
                    PWMRangeHigh = { Value = { 40, 60, 70, 80, 100 }, Import = true }
                }
            }
        }
    })
    lu.assertErrorMsgContains("CollectingConfigurationErrorDesc", function()
        bus:call(service, path, intf, 'Import', 'a{ss}ss', context, data, 'configuration')
    end)
end

local function test_valid_cooling_policy_import(bus)
    log:notice('===================== test valid cooling policy import start ====================')
    local data = json.encode({
        ConfigData = {
            CoolingPolicy = {
                [1] = {
                    Id =  {Value = 71, Import = false },
                    TemperatureRangeCelsiusLow = { Value = { -127, 25, 30, 35, 40 }, Import = true },
                    TemperatureRangeCelsiusHigh = { Value = { 25, 30, 35, 40, 127 }, Import = true },
                    PWMRangeLow = { Value = { 50, 60, 70, 80, 100 }, Import = true },
                    PWMRangeHigh = { Value = { 50, 60, 70, 80, 100 }, Import = true }
                }
            }
        }
    })
    bus:call(service, path, intf, 'Import', 'a{ss}ss', context, data, 'configuration')
    skynet.sleep(300)
    -- 导出校验
    local ret = bus:call(service, path, intf, 'Export', 'a{ss}s', context, 'configuration')
    assert(ret)
    data = json.decode(ret)
    for _, obj in pairs(data.ConfigData.CoolingPolicy) do
        if obj.Id == 71 then
            assert(table.concat(obj.TemperatureRangeCelsiusLow, ' ') == '-127 25 30 35 40')
            assert(table.concat(obj.TemperatureRangeCelsiusHigh, ' ') == '25 30 35 40 127')
            assert(table.concat(obj.PWMRangeLow, ' ') == '50 60 70 80 100',
                'actual: ' .. table.concat(obj.PWMRangeLow, ' '))
            assert(table.concat(obj.PWMRangeHigh, ' ') == '50 60 70 80 100',
                'actual: ' .. table.concat(obj.PWMRangeHigh, ' '))
        end
    end
    log:notice('===================== test valid cooling policy import end ====================')
end

local function test_import_same_value_to_persistence(bus)
    log:notice('===================== test import same value to persistence start ====================')
    local data = json.encode({
        ConfigData = {
            Cooling = {
                CustomPCHTobj = { Value = 52, Import = true }
            },
            CoolingRequirement = {
                [1] = {
                    Id =  {Value = 72, Import = false },
                    ThresholdTemperatureCelsius = { Value = 70, Import = true },
                    FailoverPercent = { Value = 80, Import = true },
                    SmartCoolingTargetTemperatureCelsius = { Value = { 45, 42, 48 }, Import = true }
                }
            },
            CoolingPolicy = {
                [1] = {
                    Id =  {Value = 72, Import = false },
                    Hysteresis = { Value = 1, Import = true },
                    TemperatureRangeCelsiusLow = { Value = { -127, 25, 30, 35, 40 }, Import = true },
                    TemperatureRangeCelsiusHigh = { Value = { 25, 30, 35, 40, 127 }, Import = true },
                    PWMRangeLow = { Value = { 45, 60, 70, 80, 100 }, Import = true },
                    PWMRangeHigh = { Value = { 45, 60, 70, 80, 100 }, Import = true }
                }
            }
        }
    })
    bus:call(service, path, intf, 'Import', 'a{ss}ss', context, data, 'configuration')
    log:notice('===================== test import same value to persistence end ====================')
end

function test_business_config.main(bus)
    log:notice('================ test config import and export start ================')
    local objs = test_init(bus)
    -- 等待coolingarea对象加载完成
    test_on_import(bus)
    test_on_export(bus)
    -- 目标温度调速配置导入
    test_import_target_temp(bus)
    test_import_threshold_temp(bus)
    test_import_smart_cooling_tar(bus)
    test_import_failed_value(bus)
    test_import_cooling_requirement(bus)
    skynet.sleep(100)
    test_import_hysteresis(bus)
    test_import_custom_outlet_tar(bus)
    test_invalid_cooling_policy_import(bus)
    test_valid_cooling_policy_import(bus)
    test_import_same_value_to_persistence(bus)
    log:notice('================ test config import and export complete ================')
end

return test_business_config