-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at:
--         http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

local class = require 'mc.class'
local sml = require 'sml'
local c_tasks = require 'mc.orm.tasks'

local STOP_LOCATE<const> = 0
local LOCATE<const> = 1
local RET_OK = 0
local RET_INVALID = 1
local LOCATE_SLEEP_TIME<const> = 2100
local PD_OPERATION_SMART_TEST <const> = 7

local log = require 'mc.logging'

local function make_pd_key(controller_id, pd)
    return string.format('%d:%d:%d:%d', controller_id, pd.device_id, pd.slot_num, pd.enclosure_id)
end

-- 定位灯类，析构时会自动 STOP_LOCATE
local auto_locate = {}
auto_locate.__index = auto_locate

function auto_locate:__gc()
    self:stop()
end

function auto_locate:__close()
    self:stop()
end

function auto_locate.new(pd)
    return setmetatable({pd = pd, is_locate = false}, auto_locate)
end

function auto_locate:start()
    if self.is_locate or not self.pd:is_valid() then
        return false
    end

    self.is_locate = true
    local retval = sml.pd_operation(self.pd.controller_id, self.pd.device_id, LOCATE, '')
    if retval ~= RET_OK then
        log:info("sml_pd_operation failed. Return %d", retval)
    end
    c_tasks.get_instance().sleep_ms(LOCATE_SLEEP_TIME)
    return self.pd:is_valid()
end

function auto_locate:stop()

    if not self.pd:is_valid() then
        return
    end

    if not self.is_locate then
        return
    end

    self.is_locate = false
    local retval = sml.pd_operation(self.pd.controller_id, self.pd.device_id, STOP_LOCATE, '')
    if retval ~= RET_OK then
        log:info("sml_pd_operation failed. Return %d", retval)
    end
    c_tasks.get_instance().sleep_ms(LOCATE_SLEEP_TIME)
end

-- pd 对象类
local pd_object = class()

function pd_object:ctor(controller_id, pd, controller_type)
    self.key = make_pd_key(controller_id, pd)
    self.controller_id = controller_id
    self.device_id = pd.device_id
    self.slot_num = pd.slot_num
    self.enclosure_id = pd.enclosure_id
    self.valid_status = true
    self.controller_type = controller_type
end

function pd_object:is_valid()
    return self.valid_status
end

function pd_object:invalid()
    self.valid_status = false
end

function pd_object:locate()
    return auto_locate.new(self)
end

function pd_object:pd_object_operation(operation, data)
    if not self:is_valid() then
        return RET_INVALID
    end

    local retval = sml.pd_operation(self.controller_id, self.device_id, operation, data)
    if retval ~= RET_OK then
        log:error("sml_pd_operation failed. Return %d", retval)
    end

    return retval
end

function pd_object:set_short_dst(test_type)
    return self:pd_object_operation(PD_OPERATION_SMART_TEST, string.pack('B', test_type))
end

return pd_object
