-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at: http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.


local log = require 'mc.logging'
local c_drives_object = require 'drives.drives_object'

local customize_cfg = {}

-- 装备定制化默认配置表
local config_default_map = {
    ['BMCSet_DrivesLogAutoCollectEnable'] = 1,
    ['BMCSet_DrivesLogCollectInterval'] = 24
}

-- 装备定制化配置表
local config_map = {
    ['BMCSet_DrivesLogAutoCollectEnable'] = 1,
    ['BMCSet_DrivesLogCollectInterval'] = 24
}

local function record_customize_operation_log(ctx, msg)
    if ctx and ctx.get_initiator then
        log:operation(ctx:get_initiator(), 'storage', msg)
    end
end

local function set_drives_log_auto_collect_enable(val)
    if val == 1 then
        c_drives_object.get_instance().obj.LogAutoCollectEnable = true
    elseif val == 0 then
        c_drives_object.get_instance().obj.LogAutoCollectEnable = false
    else
        log:error('[Storage] LogAutoCollectEnable wrong. LogAutoCollectEnable is %d', val)
        return false
    end
    return true
end

local function set_drives_log_collect_interval(val)
    if val < 1 or val > 255 then
        log:error('[Storage] LogAutoCollectInterval wrong. LogAutoCollectInterval is %d', val)
        return false
    end
    c_drives_object.get_instance().obj.LogAutoCollectInterval = val
    return true
end

local import_callback_map  = {
    ['BMCSet_DrivesLogAutoCollectEnable'] = set_drives_log_auto_collect_enable,
    ['BMCSet_DrivesLogCollectInterval'] = set_drives_log_collect_interval
}

-- 配置导入接口
function customize_cfg.on_import(ctx, customize_data)
    -- 更新装备定制化默认配置表，合法项传入更新设置值
    for item, v in pairs(customize_data) do
        if not import_callback_map[item] then
            log:error('Import %s has no object to process.', item)
        end

        -- 空白定制化执行默认值
        if v.Value == nil or v.Value == "" then
            config_map[item] = config_default_map[item]
        else
            config_map[item] = v.Value
        end
    end

    -- 执行所有定制化
    local k
    for k, fun in pairs(import_callback_map) do
        local ok, ret = pcall(function()
            return fun(config_map[k])
        end)
        if not ok or not ret then
            record_customize_operation_log(ctx, string.format('Set %s customization failed', k))
            log:error('Import %s has error occured (%s) in customize config.', k, ret)
            return false
        end
    end
    record_customize_operation_log(ctx, string.format('Set %s customization successfully', k))
end

-- 配置导出接口
function customize_cfg.on_export(ctx)
    local data = {}
    local log_auto_collect_enable =  c_drives_object.get_instance().obj.LogAutoCollectEnable
    if log_auto_collect_enable then
        log_auto_collect_enable = 1
    else
        log_auto_collect_enable = 0
    end
    data.BMCSet_DrivesLogAutoCollectEnable = log_auto_collect_enable
    log:notice("Export BMCSet_DrivesLogAutoCollectEnable %s successfully", log_auto_collect_enable)

    local log_auto_collect_interval =  c_drives_object.get_instance().obj.LogAutoCollectInterval
    data.BMCSet_DrivesLogCollectInterval = log_auto_collect_interval
    log:notice("Export BMCSet_DrivesLogCollectInterval %s successfully", log_auto_collect_interval)

    if ctx and ctx.get_initiator then
        log:operation(ctx:get_initiator(), 'storage', 'Export customize config successfully')
    end
    log:notice('Export customize config successfully')
    return data
end

return customize_cfg