-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at:
--         http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

-- Description: The unit test for sensor common interfaces.
local lu = require 'luaunit'
local common = require 'sensor.sensor_common'

test_sensor_common = {}

function test_sensor_common:test_parse_enable_req()
    local data = {0x01, 0x00, 0x80, 0x72, 0x80, 0x72}
    local datas = {}
    for i = 1, #data do
        datas[i] = string.char(data[i])
    end

    local req = common.parse_enable_req(table.concat(datas, ''))
    lu.assertEquals(req.number, 1)
    lu.assertEquals(req.assert_mask, 0x7280)
    lu.assertEquals(req.deassert_mask, 0x7280)
    lu.assertEquals(req.change_mode, 0)
end

function test_sensor_common:test_check_updation()
    lu.assertEquals(common.check_updation('Reading'), true)
    lu.assertEquals(common.check_updation('ReadingRaw'), false)
    lu.assertEquals(common.check_updation('UpperNoncritical'), true)
    lu.assertEquals(common.check_updation('Upper'), false)
    lu.assertEquals(common.check_updation('EntityId'), true)
    lu.assertEquals(common.check_updation('SensorName'), true)
    lu.assertEquals(common.check_updation('Capabilities'), false)
    lu.assertEquals(common.check_updation('OriginalReading'), true)
end

function test_sensor_common:test_compare()
    -- 无符号数比较大小，预期结果：127(0x7F) < 129(0x81)
    lu.assertEquals(common.compare(0x00, 127, 129), false)
    -- 有符号数比较大小，预期结果：127(0x7F) > -129(0x81)
    lu.assertEquals(common.compare(0xC0, 127, -129), true)
end

function test_sensor_common.test_get_scan_enabled()
    lu.assertEquals(common.get_scan_enabled(0xC0), false)
    lu.assertEquals(common.get_scan_enabled(0x03), false)
    lu.assertEquals(common.get_scan_enabled(0x07), true)
end

function test_sensor_common.test_convert_raw_to_normal()
    local o = {
        Unit = 128,
        M = 100,
        MT = 0,
        B = 0,
        BA = 0,
        Accuracy = 0,
        RBExp = 224,
        Analog = 1,
        Linearization = 0,
        Reading = 35,
        UpperNoncritical = 46
    }
    lu.assertEquals(common.convert_raw_to_normal(o, o.Reading), '35.000')
    lu.assertEquals(common.convert_raw_to_normal(o, o.UpperNoncritical), '46.000')
end

function test_sensor_common.test_convert_normal_to_raw()
    local o = {
        Unit = 128,
        M = 100,
        MT = 0,
        B = 0,
        BA = 0,
        Accuracy = 0,
        RBExp = 224,
        Analog = 1,
        Linearization = 0,
        Reading = 35,
        UpperNoncritical = 46
    }
    lu.assertEquals(common.convert_normal_to_raw(o, 35.000), 35)
    lu.assertEquals(common.convert_normal_to_raw(o, 46.0), 46)
end

function test_sensor_common.test_convert_health()
    lu.assertEquals(common.convert_health(0), 'OK')
    lu.assertEquals(common.convert_health(1), 'Minor')
    lu.assertEquals(common.convert_health(2), 'Major')
    lu.assertEquals(common.convert_health(3), 'Critical')
    lu.assertEquals(common.convert_health(4), 'OK')
end

function test_sensor_common.test_convert_sensor_status()
    lu.assertEquals(common.convert_sensor_status('Disabled', 'InTest'), 'InTest')
    lu.assertEquals(common.convert_sensor_status('Starting', 'InTest'), 'InTest')
    lu.assertEquals(common.convert_sensor_status('Disabled', 'Starting'), 'Starting')
    lu.assertEquals(common.convert_sensor_status('Disabled', 'Enabled'), 'Enabled')
    lu.assertEquals(common.convert_sensor_status('Enabled', 'Starting'), 'Starting')
end

function test_sensor_common.test_get_sensor_unit_str()
    lu.assertEquals(common.get_sensor_unit_str(0, 6, 0), 'Watts')
    lu.assertEquals(common.get_sensor_unit_str(128, 1, 0), 'degrees C')
    lu.assertEquals(common.get_sensor_unit_str(0, 4, 0), 'Volts')
    lu.assertEquals(common.get_sensor_unit_str(0, 18, 0), 'RPM')
end

return test_sensor_common