-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at:
--         http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

-- Description: The unit test for sensor database interfaces.
local lu = require 'luaunit'
local open_db = require 'sensor.db'
local open_local_db = require 'sensor.local_db'
local datas = require 'test_data'
local utils = require 'mc.utils'
local sensor_db = require 'sensor_database'

test_sensor_database = {}

function test_sensor_database:setupClass()
    self.db_path = './sensor.test.db'
    self.local_db_path = './sensor.test.local.db'
    utils.remove_file(self.db_path)
    utils.remove_file(self.local_db_path)
    self.db = open_db(self.db_path, datas)
    self.local_db = open_local_db(self.local_db_path, datas, 'poweroff')
    self.db_obj = sensor_db.new(self.db, self.local_db)
end

function test_sensor_database:teardownClass()
    self.db.db:close()
    self.local_db.db:close()
    utils.remove_file(self.db_path)
    utils.remove_file(self.local_db_path)
    self.db_obj = nil
end

function test_sensor_database:test_sel_list()
    local sel = {
        RecordId = 11,
        SelType = 1,
        Timestamp = os.time(),
        GenerateId = 2,
        SelMsgVersion = 0x51,
        SensorType = 0x04,
        SensorNumber = 0xFF,
        SelEventType = 0x01,
        SelData1 = 0x0A,
        SelData2 = 0x0B,
        SelData3 = 0x0C,
        SensorId = 'TestInsert'
    }
    self.db_obj:insert_sel_list(sel, "IpmiSelList")
    local o = self.db_obj:query_sel_list(11)
    lu.assertEquals(o.SelType, 1)
    lu.assertEquals(o.GenerateId, 2)
    lu.assertEquals(o.SelMsgVersion, 0x51)
    lu.assertEquals(o.SelData2, 0x0B)
    lu.assertEquals(o.SensorId, 'TestInsert')
    local count = self.db_obj:query_sel_count("IpmiSelList")
    lu.assertEquals(count, 2)
    self.db_obj:delete_sel_list(11, "IpmiSelList")
    o = self.db_obj:query_sel_list(11)
    lu.assertTrue(o == nil or #o == 0)
end

function test_sensor_database:test_insert_pef_control()
    local new = {
        InProgress = 0,
        PostponeTimeout = 0,
        StartupDelayTID = 0,
        Action = 0,
        UnqueuedEvent = 0,
        LastEventBMC = 0,
        LastEventSMS = 0,
        NextEvent = 0,
        Id = 1
    }
    self.db_obj:insert_pef_control(new)
    local o = self.db_obj:query_pef_control()
    lu.assertEquals(o.InProgress, 0)
    lu.assertEquals(o.LastEventBMC, 0)
    lu.assertEquals(o.NextEvent, 0)
    lu.assertEquals(o.Id, 1)
end


function test_sensor_database:test_delete_sel()
    for i = 1, 10 do
        local sel = {
            RecordId = 11 + i,
            SelType = 1,
            Timestamp = os.time(),
            GenerateId = 2,
            SelMsgVersion = 0x51,
            SensorType = 0x04,
            SensorNumber = 0xFF,
            SelEventType = 0x01,
            SelData1 = 0x0A + i,
            SelData2 = 0x0B + i,
            SelData3 = 0x0C + i,
            SensorId = 'TestInsert'
        }
        self.db_obj:insert_sel_list(sel, "IpmiSelList")
    end

    local o = self.db_obj:query_sel_list(13)
    lu.assertEquals(o.SelType, 1)
    lu.assertEquals(o.SelData1, 0x0A + 2)

    self.db_obj:delete_sel(13, 20, "IpmiSelList")
    o = self.db_obj:query_sel_list(13)
    lu.assertTrue(o == nil or #o == 0)
    o = self.db_obj:query_sel_list(21)
    lu.assertEquals(o.SelType, 1)
    lu.assertEquals(o.SelData1, 0x0A + 10)
    self.db_obj:delete_sel(11, 21, "IpmiSelList")
end

function test_sensor_database:test_query_sel_filter()
    local o = self.db_obj:query_sel_filter(1, 1, 2)
    lu.assertNotIsNil(o)
    lu.assertEquals(o.FilterMask1, 2708)
    o = self.db_obj:query_sel_filter(1, 7, 0)
    lu.assertIsNil(o)
    o = self.db_obj:query_sel_filter(4, 2, 2580)
    lu.assertIsNil(o)
    o = self.db_obj:query_sel_filter(6, 111, 1)
    lu.assertIsNil(o)
end

function test_sensor_database:test_query_sel_desc()
    local descs = self.db_obj:query_sel_desc(1, 1, 0)
    lu.assertNotIsNil(descs)
    lu.assertEquals(descs[1].AlarmLevel, 1)

    descs = self.db_obj:query_sel_desc(1, 1, 1)
    lu.assertNotIsNil(descs)
    lu.assertEquals(descs[1].ShieldFlag, 0)

    descs = self.db_obj:query_sel_desc(1, 1, 4)
    lu.assertNotIsNil(descs)
    lu.assertEquals(descs[1].AlarmLevel, 3)
end

return test_sensor_database