-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at:
--         http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

local skynet = require 'skynet'
local sdbus = require 'sd_bus'
local utils = require 'mc.utils'
local log = require 'mc.logging'
local test_common = require 'test_common.utils'
local common = require 'test_sensor_common'
require 'skynet.manager'
log:setLevel(log.INFO)

local function prepare_test_data()
    log:info('== prepare test data')

    -- 设置测试的数据路径
    local project_dir = os.getenv('PROJECT_DIR')
    local test_dir = project_dir .. '/test/integration/.test_temp_data/'
    skynet.setenv('PROG_APP_PATH', test_dir .. 'apps')
    skynet.setenv('PROG_CSR_PATH', test_dir .. 'sr')
    skynet.setenv('DATA_CSR_PATH', test_dir .. 'data/sr')

    -- 准备测试的数据路径和对应的数据文件
    local test_data_dir = skynet.getenv('TEST_DATA_DIR')
    os.execute('mkdir -p ' .. test_data_dir)
    os.execute('mkdir -p ' .. test_data_dir .. '/dump')

    -- 准备 ipmi_core 的测试数据
    os.execute('mkdir -p test/integration/ipmi_core')
    utils.copy_file('test/ipmi_core/datas.lua', 'test/integration/ipmi_core/datas.lua')
    os.execute('mkdir -p ' .. test_data_dir .. 'usr/lib64')
    utils.copy_file('temp/usr/lib64/mock/libsoc_adapter_it.so', test_data_dir .. 'usr/lib64/libsoc_adapter.so')
    os.execute('mkdir -p ' .. test_data_dir .. '/apps/ipmi_core/mds')
    utils.copy_file('temp/opt/bmc/apps/ipmi_core/mds/schema.json', test_data_dir .. '/apps/ipmi_core/mds/schema.json')

    -- 准备 fructrl 的测试数据
    os.execute('mkdir -p test/integration/fructrl')
    os.execute('mkdir -p ' .. test_data_dir .. '/apps/fructrl/mds')
    utils.copy_file('temp/opt/bmc/apps/fructrl/mds/schema.json', test_data_dir .. '/apps/fructrl/mds/schema.json')

    -- 准备 hwdiscovery 的测试数据
    os.execute('mkdir -p ' .. test_data_dir .. '/apps/sensor/mds')
    os.execute('mkdir -p ' .. test_data_dir .. '/apps/event/mds')
    os.execute('mkdir -p ' .. test_data_dir .. '/apps/hwdiscovery/mds')
    os.execute('mkdir -p ' .. test_data_dir .. '/sr')
    utils.copy_file('test/sr/root.sr', test_data_dir .. '/sr/root.sr')
    utils.copy_file('test/sr/platform.sr', test_data_dir .. '/sr/platform.sr')
    utils.copy_file('test/sr/14100513_Sensor_0.sr', test_data_dir .. '/sr/14100513_Sensor_0.sr')
    utils.copy_file('test/sr/14100513_Sensor_1.sr', test_data_dir .. '/sr/14100513_Sensor_1.sr')
    utils.copy_file('mds/schema.json', test_data_dir .. '/apps/sensor/mds/schema.json')
    utils.copy_file('mds/service.json', test_data_dir .. '/apps/sensor/mds/service.json')
    utils.copy_file('temp/opt/bmc/apps/hwdiscovery/mds/schema.json',
        test_data_dir .. '/apps/hwdiscovery/mds/schema.json')
    -- 准备 event 资源
    utils.copy_file('temp/opt/bmc/apps/event/mds/schema.json', test_data_dir .. '/apps/event/mds/schema.json')
    utils.copy_file('temp/opt/bmc/apps/event/mds/service.json', test_data_dir .. '/apps/event/mds/service.json')
end

local function clear_test_data(exit_test)
    log:info('== clear test data')
    local test_data_dir = skynet.getenv('TEST_DATA_DIR')
    if exit_test then
        skynet.timeout(0, function()
            skynet.sleep(300)
            skynet.abort()
            utils.remove_file(test_data_dir)
            utils.remove_file('test/integration/ipmi_core')
        end)
    else
        utils.remove_file(test_data_dir)
        utils.remove_file('test/integration/ipmi_core')
    end
end

local function setup_bus()
    local bus = sdbus.open_user(true)

    -- 开启 INFO 日志
    local apps = {'hwdiscovery', 'ipmi_core', 'sensor', 'persistence'}
    common.set_log_level(bus, apps, 'info')

    -- 第一步：等待资源就位
    local path = '/bmc/kepler/Systems/1/DiscreteSensors/DiscreteSensor_SelStatus_0101'
    common.wait_resource(bus, 'bmc.kepler.sensor', path)
    path = '/bmc/kepler/Systems/1/DiscreteSensors/DiscreteSensor_MngmntHealth_0101'
    common.wait_resource(bus, 'bmc.kepler.sensor', path)
    path = '/bmc/kepler/Systems/1/ThresholdSensors/ThresholdSensor_InletTemp_0101'
    common.wait_resource(bus, 'bmc.kepler.sensor', path)
    path = '/bmc/kepler/Systems/1/Entities/Entity_MainBoard_0101'
    common.wait_resource(bus, 'bmc.kepler.sensor', path)
    path = '/bmc/kepler/Chassis/1/Sensors'
    common.wait_resource(bus, 'bmc.kepler.sensor', path)

    -- 第二步：等待资源更新
    require 'sensor.json_types.ThresholdSensorDisplay'
    path = '/bmc/kepler/Systems/1/ThresholdSensors/ThresholdSensor_InletTemp_0101'
    common.wait_interface(bus, path, 'bmc.kepler.Systems.ThresholdSensorDisplay')

    return bus
end

local function test_entry()
    log:info('================ test sensor application start ================')

    -- step 1: setup the bus
    local bus = setup_bus()

    -- 等待资源加载
    skynet.sleep(500)

    -- step 2: test the sensor interfaces
    local cases = require 'sensor.test_sensor_interface'
    cases.test_entry(bus)

    -- step 3: test the ipmi sensor interfaces
    cases = require 'sensor.test_ipmi_interface'
    cases.test_entry(bus)

    -- step 4: test the ipmi sel interfaces
    cases = require 'sel.test_ipmi_interface'
    cases.test_entry(bus)

    -- step 5: test the ipmi pef interfaces
    cases = require 'pef.test_ipmi_interface'
    cases.test_entry(bus)

    -- step 6: test the ipmi sdr interfaces part1
    cases = require 'sdr.test_ipmi_interface'
    cases.test_entry_part1(bus)

    -- step 7: test the multihost
    cases = require 'test_multihost_interface'
    cases.test_entry(bus)

    -- step 8: test the ipmi sdr interfaces part2
    -- 由于SDR有清除和添加操作，会导致系统中的SDR数据发生变化，因此该部分最后测试
    cases = require 'sdr.test_ipmi_interface'
    cases.test_entry_part2(bus)

    -- step 9: test the ipmi entity
    cases = require 'entity.test_entity_interface'
    cases.test_entry(bus)

    -- step 10: test the discrete events and sensors
    cases = require 'sel.test_sel_interface'
    cases.test_entry(bus)

    -- step 11: test the global ipmi interfaces
    cases = require 'global.test_ipmi_interface'
    cases.test_entry(bus)

    log:info('================ test sensor application complete ================')
    skynet.call('sensor', 'lua', 'exit')
end

skynet.start(function()
    clear_test_data(false)
    prepare_test_data()
    test_common.dbus_launch()
    skynet.uniqueservice('sd_bus')
    skynet.uniqueservice('persistence/service/main')
    skynet.uniqueservice('maca/service/main')
    skynet.uniqueservice('hwdiscovery/service/main')
    skynet.uniqueservice('fructrl/service/main')
    skynet.uniqueservice('ipmi_core/service/main')
    skynet.uniqueservice('event/service/main')
    skynet.uniqueservice('main')
    skynet.fork(function()
        local ok, err = pcall(test_entry)
        clear_test_data(true)
        if not ok then
            print(debug.traceback(nil, nil, 15))
            error(err)
        end
    end)
end)
