-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at:
--         http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

-- Description: test the external IPMI and RPC interfaces of sensor in multihost.
local skynet = require 'skynet'
local log = require 'mc.logging'
local ctx = require 'mc.context'
local mdb = require 'mc.mdb'
local ipmitool = require 'test_ipmitool'

local multihost_interface = {}
multihost_interface.__index = multihost_interface

function multihost_interface.enter_multihost(bus)
    log:info('== enter multihost ...')
    -- 获取connector
    local ret, connector = pcall(mdb.get_object, bus, '/bmc/kepler/Connector/Connector_Sensor_0_2_01',
    'bmc.kepler.Connector')
    assert(ret)
    assert(connector)

    connector.Presence = 1
    skynet.sleep(200)  -- 等待对象加载
end

function multihost_interface.exit_multihost(bus)
    log:info('== exit multihost ...')
    -- 获取connector
    local ret, connector = pcall(mdb.get_object, bus, '/bmc/kepler/Connector/Connector_Sensor_0_2_01',
    'bmc.kepler.Connector')
    assert(ret)
    assert(connector)

    connector.Presence = 0
    skynet.sleep(200)  -- 等待对象卸载
end

function multihost_interface.test_ipmi_sdr_list(bus)
    log:info('== test multihost ipmitool sdr list start ...')
    -- 模拟带内查询，查询host1
    local sdr_count = ipmitool.sdr_list(bus, 1)
    assert(sdr_count == 9, 'actual: '.. sdr_count)

    -- 模拟其他通道查询
    sdr_count = ipmitool.sdr_list(bus)
    assert(sdr_count == 14, 'actual: '.. sdr_count)
end

function multihost_interface.test_get_sensor_list(bus)
    log:info('== test multihost get sensor list start ...')
    local expected_res = {
        {SystemId = 1, SensorName = 'Inlet Temp'}, -- DT中为SystemId为1的公共传感器
        {SystemId = 1, SensorName = 'Power'},
        {SystemId = 1, SensorName = 'SYS 12V_2'},
        {SystemId = 1, SensorName = 'Boot Error'},
        {SystemId = 1, SensorName = 'Mngmnt Health'},
        {SystemId = 1, SensorName = 'Sel Status'},
        {SystemId = 2, SensorName = 'Inlet Temp'},
        {SystemId = 2, SensorName = 'Power'},
        {SystemId = 2, SensorName = 'SYS 12V_2'},
        {SystemId = 2, SensorName = 'Mngmnt Health'},
        {SystemId = 2, SensorName = 'Sel Status'}
    }

    local mobj = mdb.get_object(bus, '/bmc/kepler/Chassis/1/Sensors', 'bmc.kepler.Chassis.Sensors')
    local rsp = mobj:GetSensorList_PACKED(ctx.new())
    assert(#rsp.SensorList == 11, 'actual: ' .. tostring(#rsp.SensorList))
    for i, sensor in ipairs(rsp.SensorList) do
        assert(sensor.SystemId == expected_res[i].SystemId, 'sensor' .. i .. ' actual: ' .. sensor.SystemId)
        assert(sensor.SensorName == expected_res[i].SensorName, 'sensor' .. i .. ' actual: ' .. sensor.SensorName)
    end
end

function multihost_interface.test_entry(bus)
    log:info('================ test multihost interface start ================')
    multihost_interface.enter_multihost(bus)
    multihost_interface.test_ipmi_sdr_list(bus)
    multihost_interface.test_get_sensor_list(bus)
    multihost_interface.exit_multihost(bus)
    log:info('================ test multihost interface complete ================')
end

return multihost_interface