-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at:
--         http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.


local skynet = require 'skynet'
require 'skynet.manager'
local log = require 'mc.logging'
local service = require 'sensor_service'

local commands = {}
local app = nil

function commands.exit()
    skynet.timeout(0, function()
        log:info('- sensor service exit')
        app:exit()
        skynet.exit()
    end)
end

-- 当前命令仅为测试Entity的上下电状态
function commands.power_ctrl(id, instance, host_id, state)
    if skynet.getenv('TEST_DATA_DIR') then
        log:info('- sensor recv power ctrl: 0x%02x 0x%02X %d %d', id, instance, host_id, state)
        app.entity_mgr:power_ctrl(id, instance, host_id, state)
    end
end

-- 当前命令仅为测试SEL超限90%
function commands.sel_almost_full(value)
    if skynet.getenv('TEST_DATA_DIR') then
        log:info('- sensor recv sel almost full')
        app.sel_mgr:record_sel_almost_full(value)
    end
end

-- 当前命令仅为测试SEL超限
function commands.sel_full(value)
    if skynet.getenv('TEST_DATA_DIR') then
        log:info('- sensor recv sel full')
        app.sel_mgr:record_sel_full(value)
    end
end

-- 当前命令仅为测试管理健康状态
function commands.management_health(lun, number, state)
    if skynet.getenv('TEST_DATA_DIR') then
        log:info('- sensor recv management health: lun[%d], number[%d], state[%d]', lun, number, state)
        app.sensor_mgr:management_health(lun, number, state)
    end
end

-- 当前命令仅为测试模拟过程中的Boot Error传感器的状态
function commands.boot_error(data)
    if skynet.getenv('TEST_DATA_DIR') then
        log:info('- sensor recv boot error: event data[%d]', data)
        app.sel_mgr:update_boot_error(data)
    end
end

-- 当前命令仅为测试模拟场景修改sel最大记录量
function commands.sel_max_count(max_count)
    if skynet.getenv('TEST_DATA_DIR') then
        log:info('- sensor recv sel max count: %d', max_count)
        app.sel_mgr:set_sel_max_count(max_count)
    end
end

local function check_local_db()
    local open_local_db = require 'sensor.local_db'
    local path = '/data/opt/bmc/persistence.local/sensor.db'
    if skynet.getenv('TEST_DATA_DIR') then
        path = skynet.getenv('TEST_DATA_DIR') .. '/sensor.db'
    end

    -- 这地方仅检查数据库结构，不需要 datas
    local ok, err = pcall(open_local_db, path, nil, 'poweroff')
    if not ok then
        log:warn('[sensor] local db is incompatible and remove it. err: [%s]', err)
        os.execute('rm -f ' .. path)
    else
        -- 成功时需要关闭数据库
        log:info('[sensor] local db check ok.')
        err.db:close()
    end
end

skynet.start(function()
    skynet.uniqueservice('sd_bus')
    skynet.register('sensor')
    check_local_db()
    local ok, err = pcall(function ()
        app = service.new()
    end)
    if not ok then
        log:error('sensor app start failed, err: %s', err)
    end
    skynet.dispatch('lua', function(_, _, cmd, ...)
    local f = assert(commands[cmd])
    skynet.ret(skynet.pack(f(...)))
    end)
end)
