-- Copyright (c) 2025 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at:
--         http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

-- Description: the management of shm data.
local log = require 'mc.logging'
local core = require 'dbus.core'
local utils = require 'sensor_utils'
local shm = core.SharedMemory

local shm_data_management = {}
local shm_sdr_datas
local shm_sel_datas

function shm_data_management.init()
    local ok, sdr_datas, sel_datas
    ok, sdr_datas = pcall(shm.get_sdr_datas)
    if ok then
        shm_sdr_datas = sdr_datas
    else
        log:notice('get sdr_datas handler failed, ret=%s', sdr_datas)
    end

    ok, sel_datas = pcall(shm.get_sel_datas)
    if ok then
        shm_sel_datas = sel_datas
    else
        log:notice('get sel_datas handler failed, ret=%s', sel_datas)
    end
end

function shm_data_management.clear_shm_data()
    local ok, ret = pcall(utils.shm_lock_exec, shm, function()
        shm_sdr_datas:clear_sdr_datas()
        shm_sel_datas:clear_sel_datas()
    end)
    if not ok then
        log:info('clear shm data failed, ret=%s', ret)
    end
end

function shm_data_management.clear_static_records()
    local ok, ret = pcall(utils.shm_lock_exec, shm, function ()
        shm_sdr_datas:clear_static_records()
    end)
    if not ok then
        log:info('clear sdr static record in shm failed, ret=%s', ret)
    end
end

function shm_data_management.add_static_record(host_id, record)
    local ok, ret = pcall(utils.shm_lock_exec, shm, function ()
        shm_sdr_datas:add_static_record(host_id, record)
    end)
    if not ok then
        log:info('add sdr static record in shm failed, ret=%s', ret)
    end
end

function shm_data_management.del_static_record(host_id, index)
    local ok, ret = pcall(utils.shm_lock_exec, shm, function ()
        shm_sdr_datas:del_static_record(host_id, index)
    end)
    if not ok then
        log:info('delete sdr static record in shm failed, ret=%s', ret)
    end
end

function shm_data_management.update_static_record(host_id, index, record)
    local ok, ret = pcall(utils.shm_lock_exec, shm, function ()
        shm_sdr_datas:update_static_record(host_id, index, record)
    end)
    if not ok then
        log:info('update sdr static record in shm failed, ret=%s', ret)
    end
end

function shm_data_management.add_dynamic_data(sensor_id, sensor_number, data)
    local ok, ret = pcall(utils.shm_lock_exec, shm, function ()
        shm_sdr_datas:add_dynamic_data(sensor_id, sensor_number, data)
    end)
    if not ok then
        log:info('add sensor dynamic data to shm failed, ret=%s', ret)
    end
end

function shm_data_management.del_dynamic_data(sensor_id)
    local ok, ret = pcall(utils.shm_lock_exec, shm, function ()
        shm_sdr_datas:del_dynamic_data(sensor_id)
    end)
    if not ok then
        log:info('delete sensor dynamic data from shm failed, ret=%s', ret)
    end
end

function shm_data_management.update_sensor_number(sensor_id, new_number)
    local ok, ret = pcall(utils.shm_lock_exec, shm, function ()
        shm_sdr_datas:update_sensor_number(sensor_id, new_number)
    end)
    if not ok then
        log:info('update sensor number in shm failed, ret=%s', ret)
    end
end

function shm_data_management.update_sensor_prop(sensor_number, prop, value)
    local ok, ret = pcall(utils.shm_lock_exec, shm, function ()
        shm_sdr_datas:update_sensor_prop(sensor_number, prop, value)
    end)
    if not ok then
        log:info('update sensor property in shm failed, ret=%s', ret)
    end
end

function shm_data_management.update_max_query_count(max_query_count)
    local ok, ret = pcall(utils.shm_lock_exec, shm, function()
        shm_sel_datas:update_max_query_count(max_query_count)
    end)
    if not ok then
        log:info('update sel QueryMaxCount in shm failed, ret=%s', ret)
    end
end

function shm_data_management.switch_mode_type(type)
    local ok, ret = pcall(utils.shm_lock_exec, shm, function()
        shm_sel_datas:switch_mode_type(type)
    end)
    if not ok then
        log:info('switch mode type in shm failed, ret=%s', ret)
    end
end

function shm_data_management.init_sel_info(cur_record_id, max_count, max_query_count, add_timestamp, del_timestamp)
    local ok, ret = pcall(utils.shm_lock_exec, shm, function()
        shm_sel_datas:init_sel_count(cur_record_id, max_count, max_query_count)
        shm_sel_datas:init_sel_timestamp(add_timestamp, del_timestamp)
    end)
    if not ok then
        log:info('init sel info in shm failed, ret=%s', ret)
    end
end

function shm_data_management.add_sel_record(record_id, sensor_id, sel_record, timestamp)
    local ok, ret = pcall(utils.shm_lock_exec, shm, function()
        shm_sel_datas:add_sel_record(record_id, sensor_id, sel_record, timestamp)
    end)
    if not ok then
        log:info('add sel to shm failed, ret=%s', ret)
    end
end

function shm_data_management.clear_sel_record(del_timestamp)
    local ok, ret = pcall(utils.shm_lock_exec, shm, function()
        shm_sel_datas:clear_sel_record(del_timestamp)
    end)
    if not ok then
        log:info('clear sel in shm failed, ret=%s', ret)
    end
end

function shm_data_management.get_reserve_id(update_flag)
    local ok, shm_reserve_id = pcall(utils.shm_lock_exec, shm, function()
        return shm_sel_datas:get_reserve_id(update_flag)
    end)
    if not ok then
        log:info('get sel reserve id in shm failed, ret=%s', shm_reserve_id)
    end

    return ok and shm_reserve_id
end

return shm_data_management