-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at:
--         http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

local skynet = require 'skynet'
local sdbus = require 'sd_bus'
local utils = require 'mc.utils'
local log = require 'mc.logging'
local test_common = require 'test_common.utils'
require 'skynet.manager'

local project_dir = os.getenv('PROJECT_DIR')
local test_dir = project_dir .. '/test/integration/.test_temp_data/'
skynet.setenv('PROG_APP_PATH', test_dir .. 'apps')
skynet.setenv('PROG_CSR_PATH', test_dir .. 'sr')
skynet.setenv('DATA_CSR_PATH', test_dir .. 'data/sr')
skynet.setenv('DUMP_FILE_PATH', test_dir .. 'dump')

log:setLevel(log.INFO)

local function prepare_test_data()
    log:info('- prepare test data')
    local test_data_dir = skynet.getenv('TEST_DATA_DIR')

    -- prepare common mds files and folders
    local dir_list = {'apps/rack_mgmt/mds', 'apps/hwdiscovery/mds', 'sr', '/dump'}
    for _, path in pairs(dir_list) do
        os.execute('mkdir -p ' .. test_data_dir .. '/' .. path)
    end

    -- prepare hwdiscovery mds files
    utils.copy_file('temp/opt/bmc/apps/hwdiscovery/mds/schema.json',
        test_data_dir .. '/apps/hwdiscovery/mds/schema.json')
    utils.copy_file('temp/opt/bmc/apps/hwdiscovery/mds/service.json',
        test_data_dir .. '/apps/hwdiscovery/mds/service.json')

    -- prepare rack mgmt mds files
    utils.copy_file('mds/schema.json', test_data_dir .. '/apps/rack_mgmt/mds/schema.json')
    utils.copy_file('mds/service.json', test_data_dir .. '/apps/rack_mgmt/mds/service.json')

    -- prepare root.sr
    utils.copy_file('test/integration/test_data/root.sr', test_data_dir .. '/sr/root.sr')

    -- prepare persistence related
    local src = os.getenv('ROOT_DIR') .. '/test_data/apps/persistence/mmcblk0p8'
    local dst = test_data_dir .. '/mmcblk0p8'
    utils.copy_file(src, dst)
end

local function clear_test_data(exit_test)
    log:info('== clear test data')
    local test_data_dir = skynet.getenv('TEST_DATA_DIR')
    if exit_test then
        skynet.timeout(0, function()
            skynet.sleep(300)
            skynet.abort()
            utils.remove_file(test_data_dir)
        end)
    else
        utils.remove_file(test_data_dir)
    end
end

local function test_entry()
    log:info('================ test rack mgmt start ================')

    -- 1. setup the bus
    local bus = sdbus.open_user(true)
    bus:start()

    -- 2. test rack mgmt service
    local cases = require 'service_cases'
    cases.test_entry(bus)

    -- 3. test rack mgmt heartbeat
    cases = require 'heartbeat_cases'
    cases.test_entry(bus)

    log:info('================ test rack mgmt complete ================')

    skynet.call('.rack_mgmt', 'lua', 'exit')
end

skynet.start(function()
    clear_test_data(false)
    prepare_test_data()
    test_common.dbus_launch()
    skynet.uniqueservice('sd_bus')
    skynet.uniqueservice('maca/service/main')
    skynet.uniqueservice('persistence/service/main')
    skynet.uniqueservice('hwdiscovery/service/main')
    skynet.uniqueservice('main')
    skynet.fork(function()
        local ok, err = pcall(test_entry)
        clear_test_data(true)
        if not ok then
            print(debug.traceback())
            error(err)
        end
    end)
end)
