-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at: http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

local bs = require 'mc.bitstring'
local log = require 'mc.logging'
local core = require 'network.core'
local ncsi_protocol_intf = require 'ncsi_protocol_intf'
local ncsi_utils = require 'ncsi.ncsi_protocol.ncsi_utils'
local ncsi_def = require 'ncsi.ncsi_protocol.ncsi_def'
local ncsi_packet = require 'ncsi.ncsi_protocol.ncsi_packet'
local ncsi_parameter = require 'ncsi.ncsi_protocol.ncsi_parameter'

local ncsi_package = {}

local SELECT_PACKAGE = 0x01
local SELECT_PACKAGE_RSP = 0x81
local DESELECT_PACKAGE = 0x02
local DESELECT_PACKAGE_RSP = 0x82

local SELECT_PACKAGE_REQ_LEN = 4
local DESELECT_PACKAGE_REQ_LEN = 0

local SEL_PACKAGE_SIZE = 34
local DESEL_PACKAGE_SIZE = 34
local ARBITRATION_DISABLE = 1
local PACKAGE_SUPPORT = 1

local desel_package_bs = bs.new([[<<
    check_sum:32,
    data:26/string,
    fcs:32
>>]])

local sel_package_bs = bs.new([[<<
    reserved1:3/string,
    hd_arbitration:1,
    reserved2:7,
    check_sum:32,
    data:22/string,
    fcs:32
>>]])

-- 填充选择包请求的payload
local function fill_select_package_payload(req_packet, arbitration_flag)
    -- 定义公共的payload数据结构
    local payload_data = {
        reserved1 = string.rep('\0', 3),
        hd_arbitration = arbitration_flag,
        reserved2 = 0,
        check_sum = 0,
        data = string.rep('\0', 22),
        fcs = 0
    }

    -- 首先创建一个check_sum为0的初始payload
    req_packet.payload = sel_package_bs:pack(payload_data)

    -- 计算校验和和CRC32
    local check_sum = ncsi_utils.get_checksum(req_packet, ncsi_def.PACKET_HEAD_LEN + SELECT_PACKAGE_REQ_LEN)
    local crc32 = ncsi_utils.get_crc32(req_packet, ncsi_def.PACKET_HEAD_LEN + SEL_PACKAGE_SIZE - 4)

    -- 更新checksum和FCS字段
    payload_data.check_sum = core.htonl(check_sum)
    payload_data.fcs = core.htonl(crc32)

    -- 返回最终的payload
    return sel_package_bs:pack(payload_data)
end

-- 写入选择包命令
local function write_select_package_cmd(req_packet, eth_name, arbitration_flag)
    ncsi_utils.ncsi_cmd_common_config(req_packet)
    req_packet.packet_head.payload_len_hi = (SELECT_PACKAGE_REQ_LEN >> 8) & 0x0f
    req_packet.packet_head.payload_len_lo = SELECT_PACKAGE_REQ_LEN & 0xff
    req_packet.payload = fill_select_package_payload(req_packet, arbitration_flag)

    local req_data = ncsi_utils.ncsi_packet_bs:pack(req_packet)
    return ncsi_protocol_intf.send_ncsi_cmd(req_data, 
        SEL_PACKAGE_SIZE + ncsi_def.ETHERNET_HEAD_LEN + ncsi_def.PACKET_HEAD_LEN, eth_name)
end

local function write_deselect_package_cmd(req_packet, eth_name)
    ncsi_utils.ncsi_cmd_common_config(req_packet)
    req_packet.packet_head.payload_len_hi = (DESELECT_PACKAGE_REQ_LEN >> 8) & 0x0f
    req_packet.packet_head.payload_len_lo = DESELECT_PACKAGE_REQ_LEN & 0xff
    local check_sum = ncsi_utils.get_checksum(req_packet, ncsi_def.PACKET_HEAD_LEN + DESELECT_PACKAGE_REQ_LEN)
    local crc32 = ncsi_utils.get_crc32(req_packet, ncsi_def.PACKET_HEAD_LEN + DESEL_PACKAGE_SIZE - 4)
    local payload_data = desel_package_bs:pack({
        check_sum = core.htonl(check_sum),
        data = '',
        fcs = core.htonl(crc32)
    })
    req_packet.payload = payload_data
    local req_data = ncsi_utils.ncsi_packet_bs:pack(req_packet)
    return ncsi_protocol_intf.send_ncsi_cmd(req_data, 
        DESEL_PACKAGE_SIZE + ncsi_def.ETHERNET_HEAD_LEN + ncsi_def.PACKET_HEAD_LEN, eth_name)
end

-- 读取选择包响应
local function read_select_package_rsp(rsp)
    return ncsi_packet.read_common_rsp(rsp, 'select package')
end

-- 读取取消选择包响应
local function read_deselect_package_rsp(rsp)
    return ncsi_packet.read_common_rsp(rsp, 'deselect package')
end

-- 命令处理表
local ncsi_package_table = {
    [SELECT_PACKAGE] = write_select_package_cmd,
    [SELECT_PACKAGE_RSP] = read_select_package_rsp,
    [DESELECT_PACKAGE] = write_deselect_package_cmd,
    [DESELECT_PACKAGE_RSP] = read_deselect_package_rsp
}

-- 选择包
function ncsi_package.ncsi_select_package(package_id, eth_name)
    local channel_id = 0x1f
    local req_packet = ncsi_packet.create_request_packet(package_id, channel_id, SELECT_PACKAGE)

    local custom_cmd_table = ncsi_utils.create_custom_cmd_table(
        ncsi_package_table, SELECT_PACKAGE, write_select_package_cmd, ARBITRATION_DISABLE
    )

    local ret = ncsi_utils.ncsi_cmd_ctrl(package_id, channel_id, req_packet, eth_name, custom_cmd_table)
    if ret ~= ncsi_def.NCSI_SUCCESS then
        log:error('ncsi cmd ctrl select package failed, package_id = %s, channel_id = %s, eth_name = %s',
            package_id, channel_id, eth_name)
        return ret
    end

    ncsi_parameter.get_instance():get_ncsi_parameter().package_exist[package_id] = PACKAGE_SUPPORT
    return ret
end

-- 取消选择包
function ncsi_package.ncsi_deselect_package(package_id, eth_name)
    local channel_id = 0x1f
    local req_packet = ncsi_packet.create_request_packet(package_id, channel_id, DESELECT_PACKAGE)
    local ret = ncsi_utils.ncsi_cmd_ctrl(package_id, channel_id, req_packet, eth_name, ncsi_package_table)

    if ret ~= ncsi_def.NCSI_SUCCESS then
        log:error('ncsi cmd ctrl deselect package failed, package_id = %s, channel_id = %s, eth_name = %s',
            package_id, channel_id, eth_name)
    end

    return ret
end

return ncsi_package