-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at:
--          http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.
local ipmi = require 'ipmi'
local mdb = require 'mc.mdb'
local skynet = require 'skynet'
local class = require 'mc.class'
local context = require 'mc.context'
local open_db = require 'network_adapter.db'
local app_base = require 'mc.service_app_base'
local open_local_db = require 'network_adapter.local_db'
local object_manage = require 'mc.mdb.object_manage'
local persist_client = require 'persistence.persist_client_lib'
local orm_classes = require 'network_adapter.orm_classes'
local ok, datas = pcall(require, 'network_adapter.datas')
if not ok then
    datas = nil -- 如果没有datas配置，证明当前组件不需要datas，仅打开数据库
end

local keplerSystemsNetworkAdaptersTypes = require 'network_adapter.json_types.keplerSystemsNetworkAdapters'
local NetworkBondingsTypes = require 'network_adapter.json_types.NetworkBondings'
local PropertiesTypes = require 'network_adapter.json_types.Properties'
local NetworkAdapterTypes = require 'network_adapter.json_types.NetworkAdapter'
local NetworkAdapterMetricsTypes = require 'network_adapter.json_types.NetworkAdapterMetrics'
local StatusTypes = require 'network_adapter.json_types.Status'
local HardwareTypes = require 'network_adapter.json_types.Hardware'
local NetworkPortTypes = require 'network_adapter.json_types.NetworkPort'
local MetricsTypes = require 'network_adapter.json_types.Metrics'
local DataCenterBridgingTypes = require 'network_adapter.json_types.DataCenterBridging'
local LLDPReceiveTypes = require 'network_adapter.json_types.LLDPReceive'
local FibreChannelTypes = require 'network_adapter.json_types.FibreChannel'
local FaultTypes = require 'network_adapter.json_types.Fault'
local VLANsTypes = require 'network_adapter.json_types.VLANs'
local VLANTypes = require 'network_adapter.json_types.VLAN'
local IPv4AddressTypes = require 'network_adapter.json_types.IPv4Address'
local IPv6AddressTypes = require 'network_adapter.json_types.IPv6Address'
local OpticalModuleTypes = require 'network_adapter.json_types.OpticalModule'
local BiasCurrentTypes = require 'network_adapter.json_types.BiasCurrent'
local PowerTypes = require 'network_adapter.json_types.Power'
local TemperatureTypes = require 'network_adapter.json_types.Temperature'
local VoltageTypes = require 'network_adapter.json_types.Voltage'
local DiagnoseTypes = require 'network_adapter.json_types.Diagnose'
local MetricTypes = require 'network_adapter.json_types.Metric'
local ChannelTypes = require 'network_adapter.json_types.Channel'
local ManufactureTypes = require 'network_adapter.json_types.Manufacture'
local NCSITypes = require 'network_adapter.json_types.NCSI'
local NetworkBridgeTypes = require 'network_adapter.json_types.NetworkBridge'
local ContainedPortTypes = require 'network_adapter.json_types.ContainedPort'
local NetworkBondingTypes = require 'network_adapter.json_types.NetworkBonding'
local PRBSTestTypes = require 'network_adapter.json_types.PRBSTest'
local DebugSystemsNetworkAdaptersTypes = require 'network_adapter.json_types.DebugSystemsNetworkAdapters'

local NetworkAdapters = mdb.register_object('/bmc/kepler/Systems/:SystemId/NetworkAdapters', {
    {name = 'bmc.kepler.Systems.NetworkAdapters', interface = keplerSystemsNetworkAdaptersTypes.interface},
    {name = 'bmc.kepler.Systems.NetworkBondings', interface = NetworkBondingsTypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function NetworkAdapters:ctor(SystemId)
    self.path = '/bmc/kepler/Systems/' .. SystemId .. '/NetworkAdapters'
end

local NetworkAdapter = mdb.register_object('/bmc/kepler/Systems/:SystemID/NetworkAdapters/:ID', {
    {name = 'bmc.kepler.Systems.NetworkAdapter', interface = NetworkAdapterTypes.interface},
    {name = 'bmc.kepler.Systems.NetworkAdapter.NetworkAdapterMetrics', interface = NetworkAdapterMetricsTypes.interface},
    {name = 'bmc.kepler.Systems.NetworkAdapter.Status', interface = StatusTypes.interface},
    {name = 'bmc.kepler.Inventory.Hardware', interface = HardwareTypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function NetworkAdapter:ctor(SystemID, ID)
    self.path = '/bmc/kepler/Systems/' .. SystemID .. '/NetworkAdapters/' .. ID
end

local NetworkPort = mdb.register_object('/bmc/kepler/Systems/:SystemID/NetworkAdapters/:ID1/Ports/:ID2', {
    {name = 'bmc.kepler.Systems.NetworkPort', interface = NetworkPortTypes.interface},
    {name = 'bmc.kepler.Systems.NetworkPort.Metrics', interface = MetricsTypes.interface},
    {name = 'bmc.kepler.Systems.NetworkPort.DataCenterBridging', interface = DataCenterBridgingTypes.interface},
    {name = 'bmc.kepler.Systems.NetworkPort.LLDPReceive', interface = LLDPReceiveTypes.interface},
    {name = 'bmc.kepler.Systems.NetworkPort.FibreChannel', interface = FibreChannelTypes.interface},
    {name = 'bmc.kepler.Systems.NetworkPort.Fault', interface = FaultTypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function NetworkPort:ctor(SystemID, ID1, ID2)
    self.path = '/bmc/kepler/Systems/' .. SystemID .. '/NetworkAdapters/' .. ID1 .. '/Ports/' .. ID2
end

local VLANs = mdb.register_object('/bmc/kepler/Systems/:SystemID/NetworkAdapters/:ID1/Ports/:ID2/VLANs', {
    {name = 'bmc.kepler.Systems.NetworkPort.VLANs', interface = VLANsTypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function VLANs:ctor(SystemID, ID1, ID2)
    self.path = '/bmc/kepler/Systems/' .. SystemID .. '/NetworkAdapters/' .. ID1 .. '/Ports/' .. ID2 .. '/VLANs'
end

local VLAN = mdb.register_object('/bmc/kepler/Systems/:SystemID/NetworkAdapters/:ID1/Ports/:ID2/VLANs/:ID3', {
    {name = 'bmc.kepler.Systems.NetworkPort.VLAN', interface = VLANTypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function VLAN:ctor(SystemID, ID1, ID2, ID3)
    self.path = '/bmc/kepler/Systems/' .. SystemID .. '/NetworkAdapters/' .. ID1 .. '/Ports/' .. ID2 .. '/VLANs/' .. ID3
end

local IPv4Address = mdb.register_object(
    '/bmc/kepler/Systems/:SystemID/NetworkAdapters/:ID1/Ports/:ID2/IPv4Addresses/:ID3', {
        {name = 'bmc.kepler.Systems.IPv4Address', interface = IPv4AddressTypes.interface},
        {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
    })

function IPv4Address:ctor(SystemID, ID1, ID2, ID3)
    self.path = '/bmc/kepler/Systems/' .. SystemID .. '/NetworkAdapters/' .. ID1 .. '/Ports/' .. ID2 ..
                    '/IPv4Addresses/' .. ID3
end

local IPv6Address = mdb.register_object(
    '/bmc/kepler/Systems/:SystemID/NetworkAdapters/:ID1/Ports/:ID2/IPv6Addresses/:ID3', {
        {name = 'bmc.kepler.Systems.IPv6Address', interface = IPv6AddressTypes.interface},
        {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
    })

function IPv6Address:ctor(SystemID, ID1, ID2, ID3)
    self.path = '/bmc/kepler/Systems/' .. SystemID .. '/NetworkAdapters/' .. ID1 .. '/Ports/' .. ID2 ..
                    '/IPv6Addresses/' .. ID3
end

local VLANIPv4Address = mdb.register_object(
    '/bmc/kepler/Systems/:SystemID/NetworkAdapters/:ID1/Ports/:ID2/VLANs/:ID3/IPv4Addresses/:ID4', {
        {name = 'bmc.kepler.Systems.IPv4Address', interface = IPv4AddressTypes.interface},
        {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
    })

function VLANIPv4Address:ctor(SystemID, ID1, ID2, ID3, ID4)
    self.path =
        '/bmc/kepler/Systems/' .. SystemID .. '/NetworkAdapters/' .. ID1 .. '/Ports/' .. ID2 .. '/VLANs/' .. ID3 ..
            '/IPv4Addresses/' .. ID4
end

local VLANIPv6Address = mdb.register_object(
    '/bmc/kepler/Systems/:SystemID/NetworkAdapters/:ID1/Ports/:ID2/VLANs/:ID3/IPv6Addresses/:ID4', {
        {name = 'bmc.kepler.Systems.IPv6Address', interface = IPv6AddressTypes.interface},
        {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
    })

function VLANIPv6Address:ctor(SystemID, ID1, ID2, ID3, ID4)
    self.path =
        '/bmc/kepler/Systems/' .. SystemID .. '/NetworkAdapters/' .. ID1 .. '/Ports/' .. ID2 .. '/VLANs/' .. ID3 ..
            '/IPv6Addresses/' .. ID4
end

local OpticalModule = mdb.register_object('/bmc/kepler/Systems/:SystemID/NetworkAdapters/:ID1/Ports/:ID2/OpticalModule',
    {
        {name = 'bmc.kepler.Systems.OpticalModule', interface = OpticalModuleTypes.interface},
        {name = 'bmc.kepler.Systems.OpticalModule.BiasCurrent', interface = BiasCurrentTypes.interface},
        {name = 'bmc.kepler.Systems.OpticalModule.Power', interface = PowerTypes.interface},
        {name = 'bmc.kepler.Systems.OpticalModule.Temperature', interface = TemperatureTypes.interface},
        {name = 'bmc.kepler.Systems.OpticalModule.Voltage', interface = VoltageTypes.interface},
        {name = 'bmc.kepler.Systems.OpticalModule.Diagnose', interface = DiagnoseTypes.interface},
        {name = 'bmc.kepler.Metric', interface = MetricTypes.interface},
        {name = 'bmc.kepler.Inventory.Hardware', interface = HardwareTypes.interface},
        {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
    })

function OpticalModule:ctor(SystemID, ID1, ID2)
    self.path = '/bmc/kepler/Systems/' .. SystemID .. '/NetworkAdapters/' .. ID1 .. '/Ports/' .. ID2 .. '/OpticalModule'
end

local OpticalModuleChannel = mdb.register_object(
    '/bmc/kepler/Systems/:SystemID/NetworkAdapters/:ID1/Ports/:ID2/OpticalModule/Channel/:ID3', {
        {name = 'bmc.kepler.Systems.OpticalModule.Channel', interface = ChannelTypes.interface},
        {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
    })

function OpticalModuleChannel:ctor(SystemID, ID1, ID2, ID3)
    self.path = '/bmc/kepler/Systems/' .. SystemID .. '/NetworkAdapters/' .. ID1 .. '/Ports/' .. ID2 ..
                    '/OpticalModule/Channel/' .. ID3
end

local DftNCSI = mdb.register_object('/bmc/kepler/Manufacture/PrepareTest/DftNCSI/:Id', {
    {name = 'bmc.kepler.Manufacture', interface = ManufactureTypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function DftNCSI:ctor(Id)
    self.path = '/bmc/kepler/Manufacture/PrepareTest/DftNCSI/' .. Id
end

local NCSI = mdb.register_object('/bmc/kepler/Debug/Systems/:SystemId/NCSI', {
    {name = 'bmc.kepler.Debug.Systems.NCSI', interface = NCSITypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function NCSI:ctor(SystemId)
    self.path = '/bmc/kepler/Debug/Systems/' .. SystemId .. '/NCSI'
end

local NetworkBridge = mdb.register_object('/bmc/kepler/Systems/:SystemId/NetworkBridges/:ID', {
    {name = 'bmc.kepler.Systems.NetworkBridge', interface = NetworkBridgeTypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function NetworkBridge:ctor(SystemId, ID)
    self.path = '/bmc/kepler/Systems/' .. SystemId .. '/NetworkBridges/' .. ID
end

local NetworkBridgePort = mdb.register_object('/bmc/kepler/Systems/:SystemID/NetworkBridges/:ID/Ports/:PortID', {
    {name = 'bmc.kepler.Systems.ContainedPort', interface = ContainedPortTypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function NetworkBridgePort:ctor(SystemID, ID, PortID)
    self.path = '/bmc/kepler/Systems/' .. SystemID .. '/NetworkBridges/' .. ID .. '/Ports/' .. PortID
end

local NetworkBonding = mdb.register_object('/bmc/kepler/Systems/:SystemId/NetworkBondings/:ID', {
    {name = 'bmc.kepler.Systems.NetworkBonding', interface = NetworkBondingTypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function NetworkBonding:ctor(SystemId, ID)
    self.path = '/bmc/kepler/Systems/' .. SystemId .. '/NetworkBondings/' .. ID
end

local NetworkBondingPort = mdb.register_object('/bmc/kepler/Systems/:SystemID/NetworkBondings/:ID/Ports/:PortID', {
    {name = 'bmc.kepler.Systems.ContainedPort', interface = ContainedPortTypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function NetworkBondingPort:ctor(SystemID, ID, PortID)
    self.path = '/bmc/kepler/Systems/' .. SystemID .. '/NetworkBondings/' .. ID .. '/Ports/' .. PortID
end

local PRBSTest = mdb.register_object('/bmc/kepler/Managers/:Id/DiagnosticService/PRBSTest', {
    {name = 'bmc.kepler.Managers.DiagnosticService.PRBSTest', interface = PRBSTestTypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function PRBSTest:ctor(Id)
    self.path = '/bmc/kepler/Managers/' .. Id .. '/DiagnosticService/PRBSTest'
end

local NetworkAdaptersDebug = mdb.register_object('/bmc/kepler/Debug/Systems/:SystemId/NetworkAdaptersDebug', {
    {name = 'bmc.kepler.Debug.Systems.NetworkAdapters', interface = DebugSystemsNetworkAdaptersTypes.interface},
    {name = 'bmc.kepler.Object.Properties', interface = PropertiesTypes.interface}
})

function NetworkAdaptersDebug:ctor(SystemId)
    self.path = '/bmc/kepler/Debug/Systems/' .. SystemId .. '/NetworkAdaptersDebug'
end

local model = require 'class.model'

local network_adapter_service = class(app_base.Service)

network_adapter_service.package = 'network_adapter'

function network_adapter_service:CreateNetworkAdapters(SystemId, prop_setting_cb)
    local path = '/bmc/kepler/Systems/' .. SystemId .. '/NetworkAdapters'
    return object_manage.create_object('NetworkAdapters', path, path, prop_setting_cb)
end

function network_adapter_service:CreateNetworkAdapter(SystemID, ID, prop_setting_cb)
    local path = '/bmc/kepler/Systems/' .. SystemID .. '/NetworkAdapters/' .. ID
    return object_manage.create_object('NetworkAdapter', path, path, prop_setting_cb)
end

function network_adapter_service:CreateNetworkPort(SystemID, ID1, ID2, prop_setting_cb)
    local path = '/bmc/kepler/Systems/' .. SystemID .. '/NetworkAdapters/' .. ID1 .. '/Ports/' .. ID2
    return object_manage.create_object('NetworkPort', path, path, prop_setting_cb)
end

function network_adapter_service:CreateVLANs(SystemID, ID1, ID2, prop_setting_cb)
    local path = '/bmc/kepler/Systems/' .. SystemID .. '/NetworkAdapters/' .. ID1 .. '/Ports/' .. ID2 .. '/VLANs'
    return object_manage.create_object('VLANs', path, path, prop_setting_cb)
end

function network_adapter_service:CreateVLAN(SystemID, ID1, ID2, ID3, prop_setting_cb)
    local path = '/bmc/kepler/Systems/' .. SystemID .. '/NetworkAdapters/' .. ID1 .. '/Ports/' .. ID2 .. '/VLANs/' ..
                     ID3
    return object_manage.create_object('VLAN', path, path, prop_setting_cb)
end

function network_adapter_service:CreateIPv4Address(SystemID, ID1, ID2, ID3, prop_setting_cb)
    local path = '/bmc/kepler/Systems/' .. SystemID .. '/NetworkAdapters/' .. ID1 .. '/Ports/' .. ID2 ..
                     '/IPv4Addresses/' .. ID3
    return object_manage.create_object('IPv4Address', path, path, prop_setting_cb)
end

function network_adapter_service:CreateIPv6Address(SystemID, ID1, ID2, ID3, prop_setting_cb)
    local path = '/bmc/kepler/Systems/' .. SystemID .. '/NetworkAdapters/' .. ID1 .. '/Ports/' .. ID2 ..
                     '/IPv6Addresses/' .. ID3
    return object_manage.create_object('IPv6Address', path, path, prop_setting_cb)
end

function network_adapter_service:CreateVLANIPv4Address(SystemID, ID1, ID2, ID3, ID4, prop_setting_cb)
    local path = '/bmc/kepler/Systems/' .. SystemID .. '/NetworkAdapters/' .. ID1 .. '/Ports/' .. ID2 .. '/VLANs/' ..
                     ID3 .. '/IPv4Addresses/' .. ID4
    return object_manage.create_object('VLANIPv4Address', path, path, prop_setting_cb)
end

function network_adapter_service:CreateVLANIPv6Address(SystemID, ID1, ID2, ID3, ID4, prop_setting_cb)
    local path = '/bmc/kepler/Systems/' .. SystemID .. '/NetworkAdapters/' .. ID1 .. '/Ports/' .. ID2 .. '/VLANs/' ..
                     ID3 .. '/IPv6Addresses/' .. ID4
    return object_manage.create_object('VLANIPv6Address', path, path, prop_setting_cb)
end

function network_adapter_service:CreateOpticalModule(SystemID, ID1, ID2, prop_setting_cb)
    local path = '/bmc/kepler/Systems/' .. SystemID .. '/NetworkAdapters/' .. ID1 .. '/Ports/' .. ID2 ..
                     '/OpticalModule'
    return object_manage.create_object('OpticalModule', path, path, prop_setting_cb)
end

function network_adapter_service:CreateOpticalModuleChannel(SystemID, ID1, ID2, ID3, prop_setting_cb)
    local path = '/bmc/kepler/Systems/' .. SystemID .. '/NetworkAdapters/' .. ID1 .. '/Ports/' .. ID2 ..
                     '/OpticalModule/Channel/' .. ID3
    return object_manage.create_object('OpticalModuleChannel', path, path, prop_setting_cb)
end

function network_adapter_service:CreateDftNCSI(Id, prop_setting_cb)
    local path = '/bmc/kepler/Manufacture/PrepareTest/DftNCSI/' .. Id
    return object_manage.create_object('DftNCSI', path, path, prop_setting_cb)
end

function network_adapter_service:CreateNCSI(SystemId, prop_setting_cb)
    local path = '/bmc/kepler/Debug/Systems/' .. SystemId .. '/NCSI'
    return object_manage.create_object('NCSI', path, path, prop_setting_cb)
end

function network_adapter_service:CreateNetworkBridge(SystemId, ID, prop_setting_cb)
    local path = '/bmc/kepler/Systems/' .. SystemId .. '/NetworkBridges/' .. ID
    return object_manage.create_object('NetworkBridge', path, path, prop_setting_cb)
end

function network_adapter_service:CreateNetworkBridgePort(SystemID, ID, PortID, prop_setting_cb)
    local path = '/bmc/kepler/Systems/' .. SystemID .. '/NetworkBridges/' .. ID .. '/Ports/' .. PortID
    return object_manage.create_object('NetworkBridgePort', path, path, prop_setting_cb)
end

function network_adapter_service:CreateNetworkBonding(SystemId, ID, prop_setting_cb)
    local path = '/bmc/kepler/Systems/' .. SystemId .. '/NetworkBondings/' .. ID
    return object_manage.create_object('NetworkBonding', path, path, prop_setting_cb)
end

function network_adapter_service:CreateNetworkBondingPort(SystemID, ID, PortID, prop_setting_cb)
    local path = '/bmc/kepler/Systems/' .. SystemID .. '/NetworkBondings/' .. ID .. '/Ports/' .. PortID
    return object_manage.create_object('NetworkBondingPort', path, path, prop_setting_cb)
end

function network_adapter_service:CreatePRBSTest(Id, prop_setting_cb)
    local path = '/bmc/kepler/Managers/' .. Id .. '/DiagnosticService/PRBSTest'
    return object_manage.create_object('PRBSTest', path, path, prop_setting_cb)
end

function network_adapter_service:CreateNetworkAdaptersDebug(SystemId, prop_setting_cb)
    local path = '/bmc/kepler/Debug/Systems/' .. SystemId .. '/NetworkAdaptersDebug'
    return object_manage.create_object('NetworkAdaptersDebug', path, path, prop_setting_cb)
end

function network_adapter_service:ImplNetworkAdapterskeplerSystemsNetworkAdaptersGetNetCardPortNum(cb)
    model.ImplNetworkAdapterskeplerSystemsNetworkAdaptersGetNetCardPortNum(cb)
end

function network_adapter_service:ImplNetworkAdapterskeplerSystemsNetworkAdaptersGetPortSpeed(cb)
    model.ImplNetworkAdapterskeplerSystemsNetworkAdaptersGetPortSpeed(cb)
end

function network_adapter_service:ImplNetworkAdapterskeplerSystemsNetworkAdaptersGetNetworkAdapterNum(cb)
    model.ImplNetworkAdapterskeplerSystemsNetworkAdaptersGetNetworkAdapterNum(cb)
end

function network_adapter_service:ImplNetworkAdapterskeplerSystemsNetworkAdaptersGetNetworkPortNum(cb)
    model.ImplNetworkAdapterskeplerSystemsNetworkAdaptersGetNetworkPortNum(cb)
end

function network_adapter_service:ImplNetworkAdapterskeplerSystemsNetworkAdaptersGetBandwidthHistory(cb)
    model.ImplNetworkAdapterskeplerSystemsNetworkAdaptersGetBandwidthHistory(cb)
end

function network_adapter_service:ImplNetworkAdapterskeplerSystemsNetworkAdaptersClearBandwidthHistory(cb)
    model.ImplNetworkAdapterskeplerSystemsNetworkAdaptersClearBandwidthHistory(cb)
end

function network_adapter_service:ImplNetworkAdapterskeplerSystemsNetworkAdaptersSetBandwidthThreshold(cb)
    model.ImplNetworkAdapterskeplerSystemsNetworkAdaptersSetBandwidthThreshold(cb)
end

function network_adapter_service:ImplNetworkAdaptersNetworkBondingsCreateBond(cb)
    model.ImplNetworkAdaptersNetworkBondingsCreateBond(cb)
end

function network_adapter_service:ImplNetworkAdaptersNetworkBondingsDeleteBond(cb)
    model.ImplNetworkAdaptersNetworkBondingsDeleteBond(cb)
end

function network_adapter_service:ImplNetworkAdaptersNetworkBondingsSetLinkMonitorPeriodMS(cb)
    model.ImplNetworkAdaptersNetworkBondingsSetLinkMonitorPeriodMS(cb)
end

function network_adapter_service:ImplNetworkAdapterNetworkAdapterStartRemovingDevice(cb)
    model.ImplNetworkAdapterNetworkAdapterStartRemovingDevice(cb)
end

function network_adapter_service:ImplNetworkAdapterNetworkAdapterDumpBlackBox(cb)
    model.ImplNetworkAdapterNetworkAdapterDumpBlackBox(cb)
end

function network_adapter_service:ImplNetworkPortNetworkPortConfigure(cb)
    model.ImplNetworkPortNetworkPortConfigure(cb)
end

function network_adapter_service:ImplVLANsVLANsCreateVLAN(cb)
    model.ImplVLANsVLANsCreateVLAN(cb)
end

function network_adapter_service:ImplVLANVLANConfigureVLAN(cb)
    model.ImplVLANVLANConfigureVLAN(cb)
end

function network_adapter_service:ImplVLANVLANDeleteVLAN(cb)
    model.ImplVLANVLANDeleteVLAN(cb)
end

function network_adapter_service:ImplOpticalModuleMetricGetData(cb)
    model.ImplOpticalModuleMetricGetData(cb)
end

function network_adapter_service:ImplOpticalModuleMetricGetItems(cb)
    model.ImplOpticalModuleMetricGetItems(cb)
end

function network_adapter_service:ImplOpticalModuleMetricSynDataAcquisitionEnable(cb)
    model.ImplOpticalModuleMetricSynDataAcquisitionEnable(cb)
end

function network_adapter_service:ImplDftNCSIManufactureStart(cb)
    model.ImplDftNCSIManufactureStart(cb)
end

function network_adapter_service:ImplDftNCSIManufactureStop(cb)
    model.ImplDftNCSIManufactureStop(cb)
end

function network_adapter_service:ImplDftNCSIManufactureGetResult(cb)
    model.ImplDftNCSIManufactureGetResult(cb)
end

function network_adapter_service:ImplNCSINCSISendCmdOverMCTP(cb)
    model.ImplNCSINCSISendCmdOverMCTP(cb)
end

function network_adapter_service:ImplPRBSTestPRBSTestQueryInfo(cb)
    model.ImplPRBSTestPRBSTestQueryInfo(cb)
end

function network_adapter_service:ImplPRBSTestPRBSTestConfig(cb)
    model.ImplPRBSTestPRBSTestConfig(cb)
end

function network_adapter_service:ImplPRBSTestPRBSTestClearStatistics(cb)
    model.ImplPRBSTestPRBSTestClearStatistics(cb)
end

function network_adapter_service:ImplPRBSTestPRBSTestShutdown(cb)
    model.ImplPRBSTestPRBSTestShutdown(cb)
end

function network_adapter_service:ImplNetworkAdaptersDebugDebugSystemsNetworkAdaptersDump(cb)
    model.ImplNetworkAdaptersDebugDebugSystemsNetworkAdaptersDump(cb)
end

---@param mdb_object object
---@param MetricName string
---@param TagPath string
function network_adapter_service:OpticalModuleMetricCollectSignal(mdb_object, MetricName, TagPath)
    self.bus:signal(mdb_object.path, 'bmc.kepler.Metric', 'CollectSignal', 'a{ss}ss', context.get_context() or {},
        MetricName, TagPath)
end

function network_adapter_service:get_bus()
    return self.bus
end

function network_adapter_service:register_ipmi_cmd(ipmi_cmd, cb)
    self.ipmi_cmds[ipmi_cmd.name] = ipmi.register_ipmi_cmd(self.bus, self.service_name, ipmi_cmd,
        cb or self[ipmi_cmd.name])
end

function network_adapter_service:unregister_ipmi_cmd(ipmi_cmd)
    local cmd_obj = self.ipmi_cmds[ipmi_cmd.name]
    if not cmd_obj then
        return
    end

    cmd_obj:unregister()
    self.ipmi_cmds[ipmi_cmd.name] = nil
end

function network_adapter_service:ctor()
    self.ipmi_cmds = {}
    self.signal_slots = {}
    self.name = self.name or network_adapter_service.package
    self.db = open_db(':memory:', datas)
    if skynet.getenv('TEST_DATA_DIR') then
        self.reset_local_db = open_local_db(skynet.getenv('TEST_DATA_DIR') .. '/network_adapter_reset.db', datas,
            'reset')
    else
        self.reset_local_db = open_local_db('/opt/bmc/pram/persistence.local/network_adapter.db', datas, 'reset')
    end

    orm_classes.init(self.db)
    self.bus:request_name(app_base.Service.get_service_name(self.name))
    model.init(self.bus)
    network_adapter_service.bus = self.bus
end

function network_adapter_service:pre_init()
    network_adapter_service.super.pre_init(self)
    self.persist = persist_client.new(self.bus, self.db, self, {
        ['t_network_adapter'] = true,
        ['t_network_port'] = true,
        ['t_optical_module'] = true,
        ['t_optical_module_channel'] = true,
        ['t_ncsi_nc_info'] = true,
        ['t_ncsi_channel_mgmt'] = true
    })
    object_manage.set_persist_client(self.persist)
end

function network_adapter_service:init()
    network_adapter_service.super.init(self)
end

return network_adapter_service
