-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at:
--         http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

local validate = require 'mc.validate'
local utils = require 'mc.utils'

local ObjectManageMsg = {}

---@class ObjectManageMsg.AddObjectReq
---@field ClassName string
---@field ObjectName string
---@field Props string
---@field Position string
---@field ExtraParams string
local TAddObjectReq = {}
TAddObjectReq.__index = TAddObjectReq
TAddObjectReq.group = {}

local function TAddObjectReq_from_obj(obj)
    return setmetatable(obj, TAddObjectReq)
end

function TAddObjectReq.new(ClassName, ObjectName, Props, Position, ExtraParams)
    return TAddObjectReq_from_obj({
        ClassName = ClassName,
        ObjectName = ObjectName,
        Props = Props,
        Position = Position,
        ExtraParams = ExtraParams
    })
end
---@param obj ObjectManageMsg.AddObjectReq
function TAddObjectReq:init_from_obj(obj)
    self.ClassName = obj.ClassName
    self.ObjectName = obj.ObjectName
    self.Props = obj.Props
    self.Position = obj.Position
    self.ExtraParams = obj.ExtraParams
end

function TAddObjectReq:remove_error_props(errs, obj)
    utils.remove_obj_error_property(obj, errs, TAddObjectReq.group)
end

TAddObjectReq.from_obj = TAddObjectReq_from_obj

TAddObjectReq.proto_property = {'ClassName', 'ObjectName', 'Props', 'Position', 'ExtraParams'}

TAddObjectReq.default = {'', '', '', '', ''}

TAddObjectReq.struct = {
    {name = 'ClassName', is_array = false, struct = nil},
    {name = 'ObjectName', is_array = false, struct = nil},
    {name = 'Props', is_array = false, struct = nil},
    {name = 'Position', is_array = false, struct = nil},
    {name = 'ExtraParams', is_array = false, struct = nil}
}

function TAddObjectReq:validate(prefix, errs)
    prefix = prefix or ''

    validate.Required(prefix .. 'ClassName', self.ClassName, 'string', false, errs)
    validate.Required(prefix .. 'ObjectName', self.ObjectName, 'string', false, errs)
    validate.Required(prefix .. 'Props', self.Props, 'string', false, errs)
    validate.Required(prefix .. 'Position', self.Position, 'string', false, errs)
    validate.Required(prefix .. 'ExtraParams', self.ExtraParams, 'string', false, errs)

    TAddObjectReq:remove_error_props(errs, self)
    validate.CheckUnknowProperty(self, TAddObjectReq.proto_property, errs)
    return self
end

function TAddObjectReq:unpack(_)
    return self.ClassName, self.ObjectName, self.Props, self.Position, self.ExtraParams
end

ObjectManageMsg.AddObjectReq = TAddObjectReq

---@class ObjectManageMsg.DeleteObjectReq
---@field ClassName string
---@field ObjectName string
---@field Position string
local TDeleteObjectReq = {}
TDeleteObjectReq.__index = TDeleteObjectReq
TDeleteObjectReq.group = {}

local function TDeleteObjectReq_from_obj(obj)
    return setmetatable(obj, TDeleteObjectReq)
end

function TDeleteObjectReq.new(ClassName, ObjectName, Position)
    return TDeleteObjectReq_from_obj({
        ClassName = ClassName,
        ObjectName = ObjectName,
        Position = Position
    })
end
---@param obj ObjectManageMsg.DeleteObjectReq
function TDeleteObjectReq:init_from_obj(obj)
    self.ClassName = obj.ClassName
    self.ObjectName = obj.ObjectName
    self.Position = obj.Position
end

function TDeleteObjectReq:remove_error_props(errs, obj)
    utils.remove_obj_error_property(obj, errs, TDeleteObjectReq.group)
end

TDeleteObjectReq.from_obj = TDeleteObjectReq_from_obj

TDeleteObjectReq.proto_property = {'ClassName', 'ObjectName', 'Position'}

TDeleteObjectReq.default = {'', '', ''}

TDeleteObjectReq.struct = {
    {name = 'ClassName', is_array = false, struct = nil},
    {name = 'ObjectName', is_array = false, struct = nil},
    {name = 'Position', is_array = false, struct = nil}
}

function TDeleteObjectReq:validate(prefix, errs)
    prefix = prefix or ''

    validate.Required(prefix .. 'ClassName', self.ClassName, 'string', false, errs)
    validate.Required(prefix .. 'ObjectName', self.ObjectName, 'string', false, errs)
    validate.Required(prefix .. 'Position', self.Position, 'string', false, errs)

    TDeleteObjectReq:remove_error_props(errs, self)
    validate.CheckUnknowProperty(self, TDeleteObjectReq.proto_property, errs)
    return self
end

function TDeleteObjectReq:unpack(_)
    return self.ClassName, self.ObjectName, self.Position
end

ObjectManageMsg.DeleteObjectReq = TDeleteObjectReq

---@class ObjectManageMsg.AddObjectCompleteReq
---@field Position string
local TAddObjectCompleteReq = {}
TAddObjectCompleteReq.__index = TAddObjectCompleteReq
TAddObjectCompleteReq.group = {}

local function TAddObjectCompleteReq_from_obj(obj)
    return setmetatable(obj, TAddObjectCompleteReq)
end

function TAddObjectCompleteReq.new(Position)
    return TAddObjectCompleteReq_from_obj({Position = Position})
end
---@param obj ObjectManageMsg.AddObjectCompleteReq
function TAddObjectCompleteReq:init_from_obj(obj)
    self.Position = obj.Position
end

function TAddObjectCompleteReq:remove_error_props(errs, obj)
    utils.remove_obj_error_property(obj, errs, TAddObjectCompleteReq.group)
end

TAddObjectCompleteReq.from_obj = TAddObjectCompleteReq_from_obj

TAddObjectCompleteReq.proto_property = {'Position'}

TAddObjectCompleteReq.default = {''}

TAddObjectCompleteReq.struct = {{name = 'Position', is_array = false, struct = nil}}

function TAddObjectCompleteReq:validate(prefix, errs)
    prefix = prefix or ''

    validate.Required(prefix .. 'Position', self.Position, 'string', false, errs)

    TAddObjectCompleteReq:remove_error_props(errs, self)
    validate.CheckUnknowProperty(self, TAddObjectCompleteReq.proto_property, errs)
    return self
end

function TAddObjectCompleteReq:unpack(_)
    return self.Position
end

ObjectManageMsg.AddObjectCompleteReq = TAddObjectCompleteReq

---@class ObjectManageMsg.DeleteObjectCompleteReq
---@field Position string
local TDeleteObjectCompleteReq = {}
TDeleteObjectCompleteReq.__index = TDeleteObjectCompleteReq
TDeleteObjectCompleteReq.group = {}

local function TDeleteObjectCompleteReq_from_obj(obj)
    return setmetatable(obj, TDeleteObjectCompleteReq)
end

function TDeleteObjectCompleteReq.new(Position)
    return TDeleteObjectCompleteReq_from_obj({Position = Position})
end
---@param obj ObjectManageMsg.DeleteObjectCompleteReq
function TDeleteObjectCompleteReq:init_from_obj(obj)
    self.Position = obj.Position
end

function TDeleteObjectCompleteReq:remove_error_props(errs, obj)
    utils.remove_obj_error_property(obj, errs, TDeleteObjectCompleteReq.group)
end

TDeleteObjectCompleteReq.from_obj = TDeleteObjectCompleteReq_from_obj

TDeleteObjectCompleteReq.proto_property = {'Position'}

TDeleteObjectCompleteReq.default = {''}

TDeleteObjectCompleteReq.struct = {{name = 'Position', is_array = false, struct = nil}}

function TDeleteObjectCompleteReq:validate(prefix, errs)
    prefix = prefix or ''

    validate.Required(prefix .. 'Position', self.Position, 'string', false, errs)

    TDeleteObjectCompleteReq:remove_error_props(errs, self)
    validate.CheckUnknowProperty(self, TDeleteObjectCompleteReq.proto_property, errs)
    return self
end

function TDeleteObjectCompleteReq:unpack(_)
    return self.Position
end

ObjectManageMsg.DeleteObjectCompleteReq = TDeleteObjectCompleteReq

return ObjectManageMsg
