-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at: http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

local lu = require 'luaunit'
local prog_stats = require "progress_statistics"

TestProgStats = {}

-- 参数异常测试
function TestProgStats:test_progress_statistic_exception_params()
    -- 参数异常测试：min_progress 要小于 max_progress
    lu.assertIsTrue(prog_stats.new(85, 20, 3) == nil)
    lu.assertIsTrue(prog_stats.new(85, 85, 3) == nil)

    -- 参数异常测试：sub_task_cnt 不能超过160
    lu.assertIsTrue(prog_stats.new(20, 85, 160) ~= nil)
    lu.assertIsTrue(prog_stats.new(20, 85, 161) == nil)

    -- 参数异常测试：设置某个任务进度，不能低于上次的设置值，也不能超过100
    local prog_stats_i = prog_stats.new(20, 85, 16)
    lu.assertIsTrue(prog_stats_i ~= nil)
    lu.assertIsTrue(prog_stats_i:set_sub_task_progress(1, -1) == -1)
    lu.assertIsTrue(prog_stats_i:set_sub_task_progress(1, 0) == 20)
    lu.assertIsTrue(prog_stats_i:set_sub_task_progress(1, 100) ~= -1)
    lu.assertIsTrue(prog_stats_i:set_sub_task_progress(1, 101) == -1)
end

-- 3个任务进度值计算
function TestProgStats:test_progress_statistic_of_three_task()
    local prog_stats_i = prog_stats.new(20, 85, 3)
    local one_task_prog = (85 - 20) / 3
    local cur_prog = prog_stats_i:get_current_progress()
    lu.assertIsTrue(cur_prog == 20)

    for set_progress=0, 100, 10 do
        lu.assertIsTrue(math.abs(
            prog_stats_i:set_sub_task_progress(1, set_progress) - (20 + one_task_prog * set_progress / 100))
            < 1)
    end
    lu.assertIsTrue(prog_stats_i:get_current_progress() == 41)
    for set_progress=0, 100, 10 do
        lu.assertIsTrue(math.abs(
            prog_stats_i:set_sub_task_progress(2, set_progress) - (41 + one_task_prog * set_progress / 100))
            < 1)
    end
    lu.assertIsTrue(prog_stats_i:get_current_progress() == 63)
    for set_progress=0, 100, 10 do
        lu.assertIsTrue(math.abs(
            prog_stats_i:set_sub_task_progress(3, set_progress) - (63 + one_task_prog * set_progress / 100))
            < 1)
    end
    lu.assertIsTrue(prog_stats_i:get_current_progress() == 85)
end

function TestProgStats:test_progress_statistic_of_sixteen_task()
    local min_prog, max_prog = 12, 93
    local one_task_prog = (max_prog - min_prog) / 16
    local prog_stats_i = prog_stats.new(min_prog, max_prog, 16)
    lu.assertIsTrue(prog_stats_i:get_current_progress() == min_prog)

    for set_progress=0, 100, 10 do
        lu.assertIsTrue(math.abs(
            prog_stats_i:set_sub_task_progress(16, set_progress) - (min_prog + one_task_prog * set_progress / 100)) < 1)
    end
    lu.assertIsTrue(prog_stats_i:get_current_progress() == 17)

    local base_prog = prog_stats_i:get_current_progress()
    for set_progress=0, 100, 20 do
        lu.assertIsTrue(math.abs(
            prog_stats_i:set_sub_task_progress(15, set_progress) - (base_prog + one_task_prog * set_progress / 100)) < 1)
    end
    lu.assertIsTrue(prog_stats_i:get_current_progress() == 22)

    base_prog = prog_stats_i:get_current_progress()
    for set_progress=0, 100, 50 do
        lu.assertIsTrue(math.abs(
            prog_stats_i:set_sub_task_progress(14, set_progress) - (base_prog + one_task_prog * set_progress / 100)) < 1)
    end
    lu.assertIsTrue(prog_stats_i:get_current_progress() == 27)

    base_prog = prog_stats_i:get_current_progress()
    for set_progress=0, 100, 20 do
        lu.assertIsTrue(math.abs(
            prog_stats_i:set_sub_task_progress(13, set_progress) - (base_prog + one_task_prog * set_progress / 100)) < 1)
    end
    lu.assertIsTrue(prog_stats_i:get_current_progress() == 32)

    base_prog = prog_stats_i:get_current_progress()
    for set_progress=0, 100, 10 do
        lu.assertIsTrue(math.abs(
            prog_stats_i:set_sub_task_progress(12, set_progress) - (base_prog + one_task_prog * set_progress / 100)) < 1)
    end
    lu.assertIsTrue(prog_stats_i:get_current_progress() == 37)

    base_prog = prog_stats_i:get_current_progress()
    for set_progress=0, 100, 20 do
        lu.assertIsTrue(math.abs(
            prog_stats_i:set_sub_task_progress(11, set_progress) - (base_prog + one_task_prog * set_progress / 100)) < 1)
    end
    lu.assertIsTrue(prog_stats_i:get_current_progress() == 42)
    lu.assertIsTrue(prog_stats_i:set_sub_task_progress(10, 100) == 47)
    lu.assertIsTrue(prog_stats_i:set_sub_task_progress(9, 100) == 52)
    lu.assertIsTrue(prog_stats_i:set_sub_task_progress(8, 100) == 57)
    lu.assertIsTrue(prog_stats_i:set_sub_task_progress(7, 100) == 62)
    lu.assertIsTrue(prog_stats_i:set_sub_task_progress(6, 100) == 67)
    lu.assertIsTrue(prog_stats_i:set_sub_task_progress(5, 100) == 72)
    lu.assertIsTrue(prog_stats_i:set_sub_task_progress(4, 100) == 77)
    lu.assertIsTrue(prog_stats_i:set_sub_task_progress(3, 100) == 82)
    lu.assertIsTrue(prog_stats_i:set_sub_task_progress(2, 100) == 87)
    lu.assertIsTrue(prog_stats_i:set_sub_task_progress(1, 100) == 93)
end

-- 160个任务进度值计算
function TestProgStats:test_progress_statistic_of_one_hundred_and_sixty_task()
    local prog_stats_i = prog_stats.new(20, 85, 160)
    local one_task_prog = (85 - 20) / 160
    local cur_prog = prog_stats_i:get_current_progress()
    lu.assertIsTrue(cur_prog == 20)

    local set_progress
    for task_id=1, 160 do

        set_progress = 50
        lu.assertIsTrue(math.abs(
            prog_stats_i:set_sub_task_progress(task_id, set_progress) -
            (20+ one_task_prog * (task_id - 1) + one_task_prog * set_progress / 100)) < 1)

        set_progress = 100
        lu.assertIsTrue(math.abs(
            prog_stats_i:set_sub_task_progress(task_id, set_progress) -
            (20 + one_task_prog * (task_id - 1) + one_task_prog * set_progress / 100)) < 1)
    end

    lu.assertIsTrue(prog_stats_i:get_current_progress() == 85)
end