-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at:
--          http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.
local mdb = require 'mc.mdb'
local class = require 'mc.class'
local app_base = require 'mc.client_app_base'
local mdb_service = require 'mc.mdb.mdb_service'
local subscribe_signal = require 'mc.mdb.subscribe_signal'
local org_freedesktop_dbus = require 'sd_bus.org_freedesktop_dbus'

local match_rule = org_freedesktop_dbus.MatchRule
local get_non_virtual_interface_objects = mdb_service.get_non_virtual_interface_objects
local foreach_non_virtual_interface_objects = mdb_service.foreach_non_virtual_interface_objects

local SystemControl = require 'general_hardware.json_types.SystemControl'
local SystemsFruCtrl = require 'general_hardware.json_types.SystemsFruCtrl'
local UpdateService = require 'general_hardware.json_types.UpdateService'
local FirmwareInventory = require 'general_hardware.json_types.FirmwareInventory'
local FirmwareInfo = require 'general_hardware.json_types.FirmwareInfo'
local Time = require 'general_hardware.json_types.Time'
local BlockIO = require 'general_hardware.json_types.BlockIO'
local Chip = require 'general_hardware.json_types.Chip'
local EthernetInterfaces = require 'general_hardware.json_types.EthernetInterfaces'
local Ipv4 = require 'general_hardware.json_types.Ipv4'
local Ipv6 = require 'general_hardware.json_types.Ipv6'
local SerialManagement = require 'general_hardware.json_types.SerialManagement'
local Accessor = require 'general_hardware.json_types.Accessor'
local Component = require 'general_hardware.json_types.Component'
local Frudata = require 'general_hardware.json_types.Frudata'
local JtagTarget = require 'general_hardware.json_types.JtagTarget'
local SystemsPowerOnLock = require 'general_hardware.json_types.SystemsPowerOnLock'
local SmBios = require 'general_hardware.json_types.SmBios'
local Pmu = require 'general_hardware.json_types.Pmu'
local BiosUpgradeService = require 'general_hardware.json_types.BiosUpgradeService'
local FirmwareActive = require 'general_hardware.json_types.FirmwareActive'
local Events = require 'general_hardware.json_types.Events'
local EyeDiagram = require 'general_hardware.json_types.EyeDiagram'
local ChassisPowerOnLock = require 'general_hardware.json_types.ChassisPowerOnLock'
local ChassisFruCtrl = require 'general_hardware.json_types.ChassisFruCtrl'
local Multihost = require 'general_hardware.json_types.Multihost'
local File = require 'general_hardware.json_types.File'
local SmsStatus = require 'general_hardware.json_types.SmsStatus'
local Sms = require 'general_hardware.json_types.Sms'
local Connector = require 'general_hardware.json_types.Connector'
local SystemsProcessor = require 'general_hardware.json_types.SystemsProcessor'
local CPU = require 'general_hardware.json_types.CPU'
local Driver = require 'general_hardware.json_types.Driver'

---@class general_hardware_client: BasicClient
local general_hardware_client = class(app_base.Client)

function general_hardware_client:GetSystemsFruCtrlObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.FruCtrl', true)
end

function general_hardware_client:ForeachSystemsFruCtrlObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.FruCtrl', cb, true)
end

function general_hardware_client:GetFirmwareInfoObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.UpdateService.FirmwareInfo', true)
end

function general_hardware_client:ForeachFirmwareInfoObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.UpdateService.FirmwareInfo', cb, true)
end

function general_hardware_client:GetTimeObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Managers.Time', true)
end

function general_hardware_client:ForeachTimeObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Managers.Time', cb, true)
end

function general_hardware_client:GetBlockIOObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Chip.BlockIO', true)
end

function general_hardware_client:ForeachBlockIOObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Chip.BlockIO', cb, true)
end

function general_hardware_client:GetChipObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Chip', true)
end

function general_hardware_client:ForeachChipObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Chip', cb, true)
end

function general_hardware_client:GetIpv4Objects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Managers.EthernetInterfaces.Ipv4', true)
end

function general_hardware_client:ForeachIpv4Objects(cb)
    return
        foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Managers.EthernetInterfaces.Ipv4', cb, true)
end

function general_hardware_client:GetIpv6Objects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Managers.EthernetInterfaces.Ipv6', true)
end

function general_hardware_client:ForeachIpv6Objects(cb)
    return
        foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Managers.EthernetInterfaces.Ipv6', cb, true)
end

function general_hardware_client:GetSerialManagementObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Managers.SerialManagement', true)
end

function general_hardware_client:ForeachSerialManagementObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Managers.SerialManagement', cb, true)
end

function general_hardware_client:GetAccessorObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Accessor', true)
end

function general_hardware_client:ForeachAccessorObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Accessor', cb, true)
end

function general_hardware_client:GetComponentObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.Component', true)
end

function general_hardware_client:ForeachComponentObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.Component', cb, true)
end

function general_hardware_client:GetFrudataObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.FrudataService.Frudata', true)
end

function general_hardware_client:ForeachFrudataObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.FrudataService.Frudata', cb, true)
end

function general_hardware_client:GetJtagTargetObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Chip.JtagTarget', true)
end

function general_hardware_client:ForeachJtagTargetObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Chip.JtagTarget', cb, true)
end

function general_hardware_client:GetSystemsPowerOnLockObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.PowerOnLock', true)
end

function general_hardware_client:ForeachSystemsPowerOnLockObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.PowerOnLock', cb, true)
end

function general_hardware_client:GetSmBiosObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.SmBios', true)
end

function general_hardware_client:ForeachSmBiosObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.SmBios', cb, true)
end

function general_hardware_client:GetBiosUpgradeServiceObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.BiosUpgradeService', true)
end

function general_hardware_client:ForeachBiosUpgradeServiceObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.BiosUpgradeService', cb, true)
end

function general_hardware_client:GetEventsObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.Events', true)
end

function general_hardware_client:ForeachEventsObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.Events', cb, true)
end

function general_hardware_client:GetChassisPowerOnLockObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Chassis.PowerOnLock', true)
end

function general_hardware_client:ForeachChassisPowerOnLockObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Chassis.PowerOnLock', cb, true)
end

function general_hardware_client:GetChassisFruCtrlObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Chassis.FruCtrl', true)
end

function general_hardware_client:ForeachChassisFruCtrlObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Chassis.FruCtrl', cb, true)
end

function general_hardware_client:GetMultihostObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Managers.Multihost', true)
end

function general_hardware_client:ForeachMultihostObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Managers.Multihost', cb, true)
end

function general_hardware_client:GetConnectorObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Connector', true)
end

function general_hardware_client:ForeachConnectorObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Connector', cb, true)
end

function general_hardware_client:GetSystemsProcessorObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.Processor', true)
end

function general_hardware_client:ForeachSystemsProcessorObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.Processor', cb, true)
end

function general_hardware_client:GetCPUObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.Processor.CPU', true)
end

function general_hardware_client:ForeachCPUObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.Processor.CPU', cb, true)
end

function general_hardware_client:GetMacaServiceSystemControlObject()
    return mdb.try_get_object(self:get_bus(), '/bmc/kepler/MacaService', 'bmc.kepler.SystemControl')
end

function general_hardware_client:OnSystemControlInterfacesAdded(cb)
    local path_namespace = '/bmc/kepler/MacaService'
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), path_namespace, cb,
        'bmc.kepler.SystemControl')
end

function general_hardware_client:OnSystemControlInterfacesRemoved(cb)
    local path_namespace = '/bmc/kepler/MacaService'
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), path_namespace,
        cb, 'bmc.kepler.SystemControl')
end

function general_hardware_client:OnSystemsFruCtrlPropertiesChanged(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_properties_changed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.FruCtrl', {'PowerState', 'SysResetDetected'})
end

function general_hardware_client:OnSystemsFruCtrlInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.FruCtrl')
end

function general_hardware_client:OnSystemsFruCtrlInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.FruCtrl')
end

function general_hardware_client:GetUpdateServiceUpdateServiceObject()
    return mdb.try_get_object(self:get_bus(), '/bmc/kepler/UpdateService', 'bmc.kepler.UpdateService')
end

function general_hardware_client:OnUpdateServiceInterfacesAdded(cb)
    local path_namespace = '/bmc/kepler/UpdateService'
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), path_namespace, cb,
        'bmc.kepler.UpdateService')
end

function general_hardware_client:OnUpdateServiceInterfacesRemoved(cb)
    local path_namespace = '/bmc/kepler/UpdateService'
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), path_namespace,
        cb, 'bmc.kepler.UpdateService')
end

function general_hardware_client:GetFirmwareInventoryFirmwareInventoryObject()
    return mdb.try_get_object(self:get_bus(), '/bmc/kepler/UpdateService/FirmwareInventory',
        'bmc.kepler.UpdateService.FirmwareInventory')
end

function general_hardware_client:OnFirmwareInventoryInterfacesAdded(cb)
    local path_namespace = '/bmc/kepler/UpdateService/FirmwareInventory'
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), path_namespace, cb,
        'bmc.kepler.UpdateService.FirmwareInventory')
end

function general_hardware_client:OnFirmwareInventoryInterfacesRemoved(cb)
    local path_namespace = '/bmc/kepler/UpdateService/FirmwareInventory'
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), path_namespace,
        cb, 'bmc.kepler.UpdateService.FirmwareInventory')
end

function general_hardware_client:OnFirmwareInfoPropertiesChanged(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_properties_changed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.UpdateService.FirmwareInfo', {'Version'})
end

function general_hardware_client:OnFirmwareInfoInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
        'bmc.kepler.UpdateService.FirmwareInfo')
end

function general_hardware_client:OnFirmwareInfoInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.UpdateService.FirmwareInfo')
end

function general_hardware_client:OnTimeInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Managers.Time')
end

function general_hardware_client:OnTimeInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Managers.Time')
end

function general_hardware_client:OnBlockIOPropertiesChanged(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_properties_changed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Chip.BlockIO')
end

function general_hardware_client:OnBlockIOInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Chip.BlockIO')
end

function general_hardware_client:OnBlockIOInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Chip.BlockIO')
end

function general_hardware_client:OnChipPropertiesChanged(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_properties_changed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Chip')
end

function general_hardware_client:OnChipInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Chip')
end

function general_hardware_client:OnChipInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Chip')
end

function general_hardware_client:GetEthernetInterfacesEthernetInterfacesObject(path_params)
    return mdb.try_get_object(self:get_bus(),
        path_params and ('/bmc/kepler/Managers/' .. path_params['Id'] .. '/EthernetInterfaces') or
            '/bmc/kepler/Managers/:Id/EthernetInterfaces', 'bmc.kepler.Managers.EthernetInterfaces')
end

function general_hardware_client:GetEthernetInterfacesObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Managers.EthernetInterfaces', true,
        {'/bmc/kepler/Managers/:Id/EthernetInterfaces'})
end

function general_hardware_client:ForeachEthernetInterfacesObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Managers.EthernetInterfaces', cb, true,
        {'/bmc/kepler/Managers/:Id/EthernetInterfaces'})
end

function general_hardware_client:OnEthernetInterfacesPropertiesChanged(cb, path_params)
    local path_namespace = path_params and ('/bmc/kepler/Managers/' .. path_params['Id'] .. '/EthernetInterfaces') or
                               '/bmc/kepler/Managers/:Id/EthernetInterfaces'
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_properties_changed(self:get_bus(), path_namespace,
        cb, 'bmc.kepler.Managers.EthernetInterfaces', {'VLANId'})
end

function general_hardware_client:OnEthernetInterfacesInterfacesAdded(cb, path_params)
    local path_namespace = path_params and ('/bmc/kepler/Managers/' .. path_params['Id'] .. '/EthernetInterfaces') or
                               '/bmc/kepler/Managers/:Id/EthernetInterfaces'
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), path_namespace, cb,
        'bmc.kepler.Managers.EthernetInterfaces')
end

function general_hardware_client:OnEthernetInterfacesInterfacesRemoved(cb, path_params)
    local path_namespace = path_params and ('/bmc/kepler/Managers/' .. path_params['Id'] .. '/EthernetInterfaces') or
                               '/bmc/kepler/Managers/:Id/EthernetInterfaces'
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), path_namespace,
        cb, 'bmc.kepler.Managers.EthernetInterfaces')
end

function general_hardware_client:OnIpv4PropertiesChanged(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_properties_changed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Managers.EthernetInterfaces.Ipv4', {'IpAddr', 'SubnetMask'})
end

function general_hardware_client:OnIpv4InterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Managers.EthernetInterfaces.Ipv4')
end

function general_hardware_client:OnIpv4InterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Managers.EthernetInterfaces.Ipv4')
end

function general_hardware_client:OnIpv6PropertiesChanged(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_properties_changed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Managers.EthernetInterfaces.Ipv6', {'IpAddr', 'PrefixLength'})
end

function general_hardware_client:OnIpv6InterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Managers.EthernetInterfaces.Ipv6')
end

function general_hardware_client:OnIpv6InterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Managers.EthernetInterfaces.Ipv6')
end

function general_hardware_client:OnSerialManagementInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Managers.SerialManagement')
end

function general_hardware_client:OnSerialManagementInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Managers.SerialManagement')
end

function general_hardware_client:OnAccessorPropertiesChanged(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_properties_changed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Accessor')
end

function general_hardware_client:OnAccessorInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Accessor')
end

function general_hardware_client:OnAccessorInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Accessor')
end

function general_hardware_client:OnComponentInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.Component')
end

function general_hardware_client:OnComponentInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.Component')
end

function general_hardware_client:OnFrudataInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
        'bmc.kepler.FrudataService.Frudata')
end

function general_hardware_client:OnFrudataInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.FrudataService.Frudata')
end

function general_hardware_client:OnJtagTargetPropertiesChanged(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_properties_changed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Chip.JtagTarget')
end

function general_hardware_client:OnJtagTargetInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Chip.JtagTarget')
end

function general_hardware_client:OnJtagTargetInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Chip.JtagTarget')
end

function general_hardware_client:OnSystemsPowerOnLockInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.PowerOnLock')
end

function general_hardware_client:OnSystemsPowerOnLockInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.PowerOnLock')
end

function general_hardware_client:OnSmBiosInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.SmBios')
end

function general_hardware_client:OnSmBiosInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.SmBios')
end

function general_hardware_client:GetPmuPmuObject(path_params)
    return mdb.try_get_object(self:get_bus(), path_params and
        ('/bmc/kepler/Systems/' .. path_params['SystemId'] .. '/Pmu') or '/bmc/kepler/Systems/:SystemId/Pmu',
        'bmc.kepler.Systems.Pmu')
end

function general_hardware_client:GetPmuObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.Pmu', true,
        {'/bmc/kepler/Systems/:SystemId/Pmu'})
end

function general_hardware_client:ForeachPmuObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.Pmu', cb, true,
        {'/bmc/kepler/Systems/:SystemId/Pmu'})
end

function general_hardware_client:OnPmuPropertiesChanged(cb, path_params)
    local path_namespace = path_params and ('/bmc/kepler/Systems/' .. path_params['SystemId'] .. '/Pmu') or
                               '/bmc/kepler/Systems/:SystemId/Pmu'
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_properties_changed(self:get_bus(), path_namespace,
        cb, 'bmc.kepler.Systems.Pmu', {'Status'})
end

function general_hardware_client:OnPmuInterfacesAdded(cb, path_params)
    local path_namespace = path_params and ('/bmc/kepler/Systems/' .. path_params['SystemId'] .. '/Pmu') or
                               '/bmc/kepler/Systems/:SystemId/Pmu'
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), path_namespace, cb,
        'bmc.kepler.Systems.Pmu')
end

function general_hardware_client:OnPmuInterfacesRemoved(cb, path_params)
    local path_namespace = path_params and ('/bmc/kepler/Systems/' .. path_params['SystemId'] .. '/Pmu') or
                               '/bmc/kepler/Systems/:SystemId/Pmu'
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), path_namespace,
        cb, 'bmc.kepler.Systems.Pmu')
end

function general_hardware_client:OnBiosUpgradeServiceInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.BiosUpgradeService')
end

function general_hardware_client:OnBiosUpgradeServiceInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.BiosUpgradeService')
end

function general_hardware_client:GetFirmwareActiveFirmwareActiveObject()
    return mdb.try_get_object(self:get_bus(), '/bmc/kepler/UpdateService/FirmwareActive',
        'bmc.kepler.UpdateService.FirmwareActive')
end

function general_hardware_client:OnFirmwareActiveInterfacesAdded(cb)
    local path_namespace = '/bmc/kepler/UpdateService/FirmwareActive'
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), path_namespace, cb,
        'bmc.kepler.UpdateService.FirmwareActive')
end

function general_hardware_client:OnFirmwareActiveInterfacesRemoved(cb)
    local path_namespace = '/bmc/kepler/UpdateService/FirmwareActive'
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), path_namespace,
        cb, 'bmc.kepler.UpdateService.FirmwareActive')
end

function general_hardware_client:OnEventsInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.Events')
end

function general_hardware_client:OnEventsInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.Events')
end

function general_hardware_client:GetBmcDfxEyeDiagramObject(path_params)
    return mdb.try_get_object(self:get_bus(),
        path_params and ('/bmc/kepler/Managers/' .. path_params['ManagerId'] .. '/SOC/BmcDfx') or
            '/bmc/kepler/Managers/:ManagerId/SOC/BmcDfx', 'bmc.kepler.Managers.SOC.Serdes.EyeDiagram')
end

function general_hardware_client:GetEyeDiagramObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Managers.SOC.Serdes.EyeDiagram', true,
        {'/bmc/kepler/Managers/:ManagerId/SOC/BmcDfx'})
end

function general_hardware_client:ForeachEyeDiagramObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Managers.SOC.Serdes.EyeDiagram', cb, true,
        {'/bmc/kepler/Managers/:ManagerId/SOC/BmcDfx'})
end

function general_hardware_client:OnEyeDiagramInterfacesAdded(cb, path_params)
    local path_namespace = path_params and ('/bmc/kepler/Managers/' .. path_params['ManagerId'] .. '/SOC/BmcDfx') or
                               '/bmc/kepler/Managers/:ManagerId/SOC/BmcDfx'
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), path_namespace, cb,
        'bmc.kepler.Managers.SOC.Serdes.EyeDiagram')
end

function general_hardware_client:OnEyeDiagramInterfacesRemoved(cb, path_params)
    local path_namespace = path_params and ('/bmc/kepler/Managers/' .. path_params['ManagerId'] .. '/SOC/BmcDfx') or
                               '/bmc/kepler/Managers/:ManagerId/SOC/BmcDfx'
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), path_namespace,
        cb, 'bmc.kepler.Managers.SOC.Serdes.EyeDiagram')
end

function general_hardware_client:OnChassisPowerOnLockInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Chassis.PowerOnLock')
end

function general_hardware_client:OnChassisPowerOnLockInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Chassis.PowerOnLock')
end

function general_hardware_client:OnChassisFruCtrlInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Chassis.FruCtrl')
end

function general_hardware_client:OnChassisFruCtrlInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Chassis.FruCtrl')
end

function general_hardware_client:OnMultihostInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Managers.Multihost')
end

function general_hardware_client:OnMultihostInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Managers.Multihost')
end

function general_hardware_client:GetFileFileObject(path_params)
    return mdb.try_get_object(self:get_bus(),
        path_params and ('/bmc/kepler/Managers/' .. path_params['Id'] .. '/Security/File') or
            '/bmc/kepler/Managers/:Id/Security/File', 'bmc.kepler.Managers.Security.File')
end

function general_hardware_client:GetFileObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Managers.Security.File', true,
        {'/bmc/kepler/Managers/:Id/Security/File'})
end

function general_hardware_client:ForeachFileObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Managers.Security.File', cb, true,
        {'/bmc/kepler/Managers/:Id/Security/File'})
end

function general_hardware_client:OnFileInterfacesAdded(cb, path_params)
    local path_namespace = path_params and ('/bmc/kepler/Managers/' .. path_params['Id'] .. '/Security/File') or
                               '/bmc/kepler/Managers/:Id/Security/File'
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), path_namespace, cb,
        'bmc.kepler.Managers.Security.File')
end

function general_hardware_client:OnFileInterfacesRemoved(cb, path_params)
    local path_namespace = path_params and ('/bmc/kepler/Managers/' .. path_params['Id'] .. '/Security/File') or
                               '/bmc/kepler/Managers/:Id/Security/File'
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), path_namespace,
        cb, 'bmc.kepler.Managers.Security.File')
end

function general_hardware_client:GetSmsSmsStatusObject(path_params)
    return mdb.try_get_object(self:get_bus(), path_params and
        ('/bmc/kepler/Systems/' .. path_params['SystemId'] .. '/Sms') or '/bmc/kepler/Systems/${SystemId}/Sms',
        'bmc.kepler.Systems.Sms.SmsStatus')
end

function general_hardware_client:GetSmsStatusObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.Sms.SmsStatus', true,
        {'/bmc/kepler/Systems/${SystemId}/Sms'})
end

function general_hardware_client:ForeachSmsStatusObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.Sms.SmsStatus', cb, true,
        {'/bmc/kepler/Systems/${SystemId}/Sms'})
end

function general_hardware_client:OnSmsStatusPropertiesChanged(cb, path_params)
    local path_namespace = path_params and ('/bmc/kepler/Systems/' .. path_params['SystemId'] .. '/Sms') or
                               '/bmc/kepler/Systems/${SystemId}/Sms'
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_properties_changed(self:get_bus(), path_namespace,
        cb, 'bmc.kepler.Systems.Sms.SmsStatus', {'State'})
end

function general_hardware_client:OnSmsStatusInterfacesAdded(cb, path_params)
    local path_namespace = path_params and ('/bmc/kepler/Systems/' .. path_params['SystemId'] .. '/Sms') or
                               '/bmc/kepler/Systems/${SystemId}/Sms'
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), path_namespace, cb,
        'bmc.kepler.Systems.Sms.SmsStatus')
end

function general_hardware_client:OnSmsStatusInterfacesRemoved(cb, path_params)
    local path_namespace = path_params and ('/bmc/kepler/Systems/' .. path_params['SystemId'] .. '/Sms') or
                               '/bmc/kepler/Systems/${SystemId}/Sms'
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), path_namespace,
        cb, 'bmc.kepler.Systems.Sms.SmsStatus')
end

function general_hardware_client:GetSmsSmsObject(path_params)
    return mdb.try_get_object(self:get_bus(), path_params and
        ('/bmc/kepler/Systems/' .. path_params['SystemId'] .. '/Sms') or '/bmc/kepler/Systems/${SystemId}/Sms',
        'bmc.kepler.Systems.Sms')
end

function general_hardware_client:GetSmsObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.Sms', true,
        {'/bmc/kepler/Systems/${SystemId}/Sms'})
end

function general_hardware_client:ForeachSmsObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Systems.Sms', cb, true,
        {'/bmc/kepler/Systems/${SystemId}/Sms'})
end

function general_hardware_client:OnSmsInterfacesAdded(cb, path_params)
    local path_namespace = path_params and ('/bmc/kepler/Systems/' .. path_params['SystemId'] .. '/Sms') or
                               '/bmc/kepler/Systems/${SystemId}/Sms'
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), path_namespace, cb,
        'bmc.kepler.Systems.Sms')
end

function general_hardware_client:OnSmsInterfacesRemoved(cb, path_params)
    local path_namespace = path_params and ('/bmc/kepler/Systems/' .. path_params['SystemId'] .. '/Sms') or
                               '/bmc/kepler/Systems/${SystemId}/Sms'
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), path_namespace,
        cb, 'bmc.kepler.Systems.Sms')
end

function general_hardware_client:OnConnectorPropertiesChanged(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_properties_changed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Connector')
end

function general_hardware_client:OnConnectorInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Connector')
end

function general_hardware_client:OnConnectorInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Connector')
end

function general_hardware_client:OnSystemsProcessorPropertiesChanged(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_properties_changed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.Processor')
end

function general_hardware_client:OnSystemsProcessorInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.Processor')
end

function general_hardware_client:OnSystemsProcessorInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.Processor')
end

function general_hardware_client:OnCPUPropertiesChanged(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_properties_changed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.Processor.CPU')
end

function general_hardware_client:OnCPUInterfacesAdded(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.Processor.CPU')
end

function general_hardware_client:OnCPUInterfacesRemoved(cb)
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), '/bmc', cb,
        'bmc.kepler.Systems.Processor.CPU')
end

function general_hardware_client:GetProxyDriverObject(path_params)
    return mdb.try_get_object(self:get_bus(),
        path_params and ('/bmc/kepler/Managers/' .. path_params['Id'] .. '/Security/Proxy') or
            '/bmc/kepler/Managers/:Id/Security/Proxy', 'bmc.kepler.Managers.Security.Proxy.Driver')
end

function general_hardware_client:GetDriverObjects()
    return get_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Managers.Security.Proxy.Driver', true,
        {'/bmc/kepler/Managers/:Id/Security/Proxy'})
end

function general_hardware_client:ForeachDriverObjects(cb)
    return foreach_non_virtual_interface_objects(self:get_bus(), 'bmc.kepler.Managers.Security.Proxy.Driver', cb, true,
        {'/bmc/kepler/Managers/:Id/Security/Proxy'})
end

function general_hardware_client:OnDriverInterfacesAdded(cb, path_params)
    local path_namespace = path_params and ('/bmc/kepler/Managers/' .. path_params['Id'] .. '/Security/Proxy') or
                               '/bmc/kepler/Managers/:Id/Security/Proxy'
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_added(self:get_bus(), path_namespace, cb,
        'bmc.kepler.Managers.Security.Proxy.Driver')
end

function general_hardware_client:OnDriverInterfacesRemoved(cb, path_params)
    local path_namespace = path_params and ('/bmc/kepler/Managers/' .. path_params['Id'] .. '/Security/Proxy') or
                               '/bmc/kepler/Managers/:Id/Security/Proxy'
    self.signal_slots[#self.signal_slots + 1] = subscribe_signal.on_interfaces_removed(self:get_bus(), path_namespace,
        cb, 'bmc.kepler.Managers.Security.Proxy.Driver')
end

---@param ResetType integer
---@return SystemControl.ForceResetRsp
function general_hardware_client:MacaServiceSystemControlForceReset(ctx, ResetType)
    local req = SystemControl.ForceResetReq.new(ResetType):validate()
    local obj = self:GetMacaServiceSystemControlObject()

    return SystemControl.ForceResetRsp.new(obj:ForceReset(ctx, req:unpack(true)))
end

function general_hardware_client:PMacaServiceSystemControlForceReset(ctx, ResetType)
    return pcall(function()
        local req = SystemControl.ForceResetReq.new(ResetType):validate()
        local obj = self:GetMacaServiceSystemControlObject()

        return SystemControl.ForceResetRsp.new(obj:ForceReset(ctx, req:unpack(true)))
    end)
end

---@param ResetType integer
---@return SystemControl.GracefulResetRsp
function general_hardware_client:MacaServiceSystemControlGracefulReset(ctx, ResetType)
    local req = SystemControl.GracefulResetReq.new(ResetType):validate()
    local obj = self:GetMacaServiceSystemControlObject()

    return SystemControl.GracefulResetRsp.new(obj:GracefulReset(ctx, req:unpack(true)))
end

function general_hardware_client:PMacaServiceSystemControlGracefulReset(ctx, ResetType)
    return pcall(function()
        local req = SystemControl.GracefulResetReq.new(ResetType):validate()
        local obj = self:GetMacaServiceSystemControlObject()

        return SystemControl.GracefulResetRsp.new(obj:GracefulReset(ctx, req:unpack(true)))
    end)
end

---@return SystemControl.WarmResetRsp
function general_hardware_client:MacaServiceSystemControlWarmReset(ctx)
    local obj = self:GetMacaServiceSystemControlObject()

    return SystemControl.WarmResetRsp.new(obj:WarmReset(ctx))
end

function general_hardware_client:PMacaServiceSystemControlWarmReset(ctx)
    return pcall(function()
        local obj = self:GetMacaServiceSystemControlObject()

        return SystemControl.WarmResetRsp.new(obj:WarmReset(ctx))
    end)
end

---@param ResetType integer
---@param ResetCause integer
---@return SystemControl.ForceResetWithCauseRsp
function general_hardware_client:MacaServiceSystemControlForceResetWithCause(ctx, ResetType, ResetCause)
    local req = SystemControl.ForceResetWithCauseReq.new(ResetType, ResetCause):validate()
    local obj = self:GetMacaServiceSystemControlObject()

    return SystemControl.ForceResetWithCauseRsp.new(obj:ForceResetWithCause(ctx, req:unpack(true)))
end

function general_hardware_client:PMacaServiceSystemControlForceResetWithCause(ctx, ResetType, ResetCause)
    return pcall(function()
        local req = SystemControl.ForceResetWithCauseReq.new(ResetType, ResetCause):validate()
        local obj = self:GetMacaServiceSystemControlObject()

        return SystemControl.ForceResetWithCauseRsp.new(obj:ForceResetWithCause(ctx, req:unpack(true)))
    end)
end

---@param ResetType integer
---@param ResetCause integer
---@return SystemControl.GracefulResetWithCauseRsp
function general_hardware_client:MacaServiceSystemControlGracefulResetWithCause(ctx, ResetType, ResetCause)
    local req = SystemControl.GracefulResetWithCauseReq.new(ResetType, ResetCause):validate()
    local obj = self:GetMacaServiceSystemControlObject()

    return SystemControl.GracefulResetWithCauseRsp.new(obj:GracefulResetWithCause(ctx, req:unpack(true)))
end

function general_hardware_client:PMacaServiceSystemControlGracefulResetWithCause(ctx, ResetType, ResetCause)
    return pcall(function()
        local req = SystemControl.GracefulResetWithCauseReq.new(ResetType, ResetCause):validate()
        local obj = self:GetMacaServiceSystemControlObject()

        return SystemControl.GracefulResetWithCauseRsp.new(obj:GracefulResetWithCause(ctx, req:unpack(true)))
    end)
end

---@param ResetMode string
---@param OperationType string
---@param TimeoutSeconds integer
---@return SystemControl.SetResetLockStatusRsp
function general_hardware_client:MacaServiceSystemControlSetResetLockStatus(ctx, ResetMode, OperationType,
    TimeoutSeconds)
    local req = SystemControl.SetResetLockStatusReq.new(ResetMode, OperationType, TimeoutSeconds):validate()
    local obj = self:GetMacaServiceSystemControlObject()

    return SystemControl.SetResetLockStatusRsp.new(obj:SetResetLockStatus(ctx, req:unpack(true)))
end

function general_hardware_client:PMacaServiceSystemControlSetResetLockStatus(ctx, ResetMode, OperationType,
    TimeoutSeconds)
    return pcall(function()
        local req = SystemControl.SetResetLockStatusReq.new(ResetMode, OperationType, TimeoutSeconds):validate()
        local obj = self:GetMacaServiceSystemControlObject()

        return SystemControl.SetResetLockStatusRsp.new(obj:SetResetLockStatus(ctx, req:unpack(true)))
    end)
end

---@param FilePath string
---@param AdditionalOptions UpdateService.Object
---@return UpdateService.StartUpgradeRsp
function general_hardware_client:UpdateServiceUpdateServiceStartUpgrade(ctx, FilePath, AdditionalOptions)
    local req = UpdateService.StartUpgradeReq.new(FilePath, AdditionalOptions):validate()
    local obj = self:GetUpdateServiceUpdateServiceObject()

    return UpdateService.StartUpgradeRsp.new(obj:StartUpgrade(ctx, req:unpack(true)))
end

function general_hardware_client:PUpdateServiceUpdateServiceStartUpgrade(ctx, FilePath, AdditionalOptions)
    return pcall(function()
        local req = UpdateService.StartUpgradeReq.new(FilePath, AdditionalOptions):validate()
        local obj = self:GetUpdateServiceUpdateServiceObject()

        return UpdateService.StartUpgradeRsp.new(obj:StartUpgrade(ctx, req:unpack(true)))
    end)
end

---@param UpgradeOptions UpdateService.Object
---@param Packages UpdateService.ParallelUpgradePackages[]
---@return UpdateService.ParallelUpdateRsp
function general_hardware_client:UpdateServiceUpdateServiceParallelUpdate(ctx, UpgradeOptions, Packages)
    local req = UpdateService.ParallelUpdateReq.new(UpgradeOptions, Packages):validate()
    local obj = self:GetUpdateServiceUpdateServiceObject()

    return UpdateService.ParallelUpdateRsp.new(obj:ParallelUpdate(ctx, req:unpack(true)))
end

function general_hardware_client:PUpdateServiceUpdateServiceParallelUpdate(ctx, UpgradeOptions, Packages)
    return pcall(function()
        local req = UpdateService.ParallelUpdateReq.new(UpgradeOptions, Packages):validate()
        local obj = self:GetUpdateServiceUpdateServiceObject()

        return UpdateService.ParallelUpdateRsp.new(obj:ParallelUpdate(ctx, req:unpack(true)))
    end)
end

---@param SystemId integer
---@param FirmwareType string
---@param VersionStr string
---@param PrepareResult integer
---@param AdditionalOptions UpdateService.Object
---@return UpdateService.PrepareReplyRsp
function general_hardware_client:UpdateServiceUpdateServicePrepareReply(ctx, SystemId, FirmwareType, VersionStr,
    PrepareResult, AdditionalOptions)
    local req =
        UpdateService.PrepareReplyReq.new(SystemId, FirmwareType,
            VersionStr, PrepareResult, AdditionalOptions):validate()
    local obj = self:GetUpdateServiceUpdateServiceObject()

    return UpdateService.PrepareReplyRsp.new(obj:PrepareReply(ctx, req:unpack(true)))
end

function general_hardware_client:PUpdateServiceUpdateServicePrepareReply(ctx, SystemId, FirmwareType, VersionStr,
    PrepareResult, AdditionalOptions)
    return pcall(function()
        local req = UpdateService.PrepareReplyReq.new(SystemId, FirmwareType, VersionStr, PrepareResult,
            AdditionalOptions):validate()
        local obj = self:GetUpdateServiceUpdateServiceObject()

        return UpdateService.PrepareReplyRsp.new(obj:PrepareReply(ctx, req:unpack(true)))
    end)
end

---@param SystemId integer
---@param FirmwareType string
---@param ProcessResult integer
---@param AdditionalOptions UpdateService.Object
---@return UpdateService.ProcessReplyRsp
function general_hardware_client:UpdateServiceUpdateServiceProcessReply(ctx, SystemId, FirmwareType, ProcessResult,
    AdditionalOptions)
    local req = UpdateService.ProcessReplyReq.new(SystemId, FirmwareType, ProcessResult, AdditionalOptions):validate()
    local obj = self:GetUpdateServiceUpdateServiceObject()

    return UpdateService.ProcessReplyRsp.new(obj:ProcessReply(ctx, req:unpack(true)))
end

function general_hardware_client:PUpdateServiceUpdateServiceProcessReply(ctx, SystemId, FirmwareType, ProcessResult,
    AdditionalOptions)
    return pcall(function()
        local req =
            UpdateService.ProcessReplyReq.new(SystemId, FirmwareType, ProcessResult, AdditionalOptions):validate()
        local obj = self:GetUpdateServiceUpdateServiceObject()

        return UpdateService.ProcessReplyRsp.new(obj:ProcessReply(ctx, req:unpack(true)))
    end)
end

---@param SystemId integer
---@param FirmwareType string
---@param FinishResult integer
---@param AdditionalOptions UpdateService.Object
---@return UpdateService.FinishReplyRsp
function general_hardware_client:UpdateServiceUpdateServiceFinishReply(ctx, SystemId, FirmwareType, FinishResult,
    AdditionalOptions)
    local req = UpdateService.FinishReplyReq.new(SystemId, FirmwareType, FinishResult, AdditionalOptions):validate()
    local obj = self:GetUpdateServiceUpdateServiceObject()

    return UpdateService.FinishReplyRsp.new(obj:FinishReply(ctx, req:unpack(true)))
end

function general_hardware_client:PUpdateServiceUpdateServiceFinishReply(ctx, SystemId, FirmwareType, FinishResult,
    AdditionalOptions)
    return pcall(function()
        local req = UpdateService.FinishReplyReq.new(SystemId, FirmwareType, FinishResult, AdditionalOptions):validate()
        local obj = self:GetUpdateServiceUpdateServiceObject()

        return UpdateService.FinishReplyRsp.new(obj:FinishReply(ctx, req:unpack(true)))
    end)
end

---@param SystemId integer
---@param FirmwareType string
---@param ResultCode integer
---@param Progress integer
---@param ResultDetail string
---@param AdditionalOptions UpdateService.Object
---@return UpdateService.UpdateUpgradeStatusRsp
function general_hardware_client:UpdateServiceUpdateServiceUpdateUpgradeStatus(ctx, SystemId, FirmwareType, ResultCode,
    Progress, ResultDetail, AdditionalOptions)
    local req = UpdateService.UpdateUpgradeStatusReq.new(SystemId, FirmwareType, ResultCode, Progress, ResultDetail,
        AdditionalOptions):validate()
    local obj = self:GetUpdateServiceUpdateServiceObject()

    return UpdateService.UpdateUpgradeStatusRsp.new(obj:UpdateUpgradeStatus(ctx, req:unpack(true)))
end

function general_hardware_client:PUpdateServiceUpdateServiceUpdateUpgradeStatus(ctx, SystemId, FirmwareType, ResultCode,
    Progress, ResultDetail, AdditionalOptions)
    return pcall(function()
        local req = UpdateService.UpdateUpgradeStatusReq.new(SystemId, FirmwareType, ResultCode, Progress, ResultDetail,
            AdditionalOptions):validate()
        local obj = self:GetUpdateServiceUpdateServiceObject()

        return UpdateService.UpdateUpgradeStatusRsp.new(obj:UpdateUpgradeStatus(ctx, req:unpack(true)))
    end)
end

---@param TaskId integer
---@return UpdateService.GetUpgradeStatusRsp
function general_hardware_client:UpdateServiceUpdateServiceGetUpgradeStatus(ctx, TaskId)
    local req = UpdateService.GetUpgradeStatusReq.new(TaskId):validate()
    local obj = self:GetUpdateServiceUpdateServiceObject()

    return UpdateService.GetUpgradeStatusRsp.new(obj:GetUpgradeStatus(ctx, req:unpack(true)))
end

function general_hardware_client:PUpdateServiceUpdateServiceGetUpgradeStatus(ctx, TaskId)
    return pcall(function()
        local req = UpdateService.GetUpgradeStatusReq.new(TaskId):validate()
        local obj = self:GetUpdateServiceUpdateServiceObject()

        return UpdateService.GetUpgradeStatusRsp.new(obj:GetUpgradeStatus(ctx, req:unpack(true)))
    end)
end

---@return UpdateService.GetUpdateProgressRsp
function general_hardware_client:UpdateServiceUpdateServiceGetUpdateProgress(ctx)
    local obj = self:GetUpdateServiceUpdateServiceObject()

    return UpdateService.GetUpdateProgressRsp.new(obj:GetUpdateProgress(ctx))
end

function general_hardware_client:PUpdateServiceUpdateServiceGetUpdateProgress(ctx)
    return pcall(function()
        local obj = self:GetUpdateServiceUpdateServiceObject()

        return UpdateService.GetUpdateProgressRsp.new(obj:GetUpdateProgress(ctx))
    end)
end

---@return UpdateService.GetPendingFirmwareListRsp
function general_hardware_client:UpdateServiceUpdateServiceGetPendingFirmwareList(ctx)
    local obj = self:GetUpdateServiceUpdateServiceObject()

    return UpdateService.GetPendingFirmwareListRsp.new(obj:GetPendingFirmwareList(ctx))
end

function general_hardware_client:PUpdateServiceUpdateServiceGetPendingFirmwareList(ctx)
    return pcall(function()
        local obj = self:GetUpdateServiceUpdateServiceObject()

        return UpdateService.GetPendingFirmwareListRsp.new(obj:GetPendingFirmwareList(ctx))
    end)
end

---@param SystemId integer
---@param Id string
---@return UpdateService.RollbackRsp
function general_hardware_client:UpdateServiceUpdateServiceRollback(ctx, SystemId, Id)
    local req = UpdateService.RollbackReq.new(SystemId, Id):validate()
    local obj = self:GetUpdateServiceUpdateServiceObject()

    return UpdateService.RollbackRsp.new(obj:Rollback(ctx, req:unpack(true)))
end

function general_hardware_client:PUpdateServiceUpdateServiceRollback(ctx, SystemId, Id)
    return pcall(function()
        local req = UpdateService.RollbackReq.new(SystemId, Id):validate()
        local obj = self:GetUpdateServiceUpdateServiceObject()

        return UpdateService.RollbackRsp.new(obj:Rollback(ctx, req:unpack(true)))
    end)
end

---@param SystemId integer
---@param Id string
---@return UpdateService.ExportRsp
function general_hardware_client:UpdateServiceUpdateServiceExport(ctx, SystemId, Id)
    local req = UpdateService.ExportReq.new(SystemId, Id):validate()
    local obj = self:GetUpdateServiceUpdateServiceObject()

    return UpdateService.ExportRsp.new(obj:Export(ctx, req:unpack(true)))
end

function general_hardware_client:PUpdateServiceUpdateServiceExport(ctx, SystemId, Id)
    return pcall(function()
        local req = UpdateService.ExportReq.new(SystemId, Id):validate()
        local obj = self:GetUpdateServiceUpdateServiceObject()

        return UpdateService.ExportRsp.new(obj:Export(ctx, req:unpack(true)))
    end)
end

---@param FilePath string
---@param FirmwareType string
---@return UpdateService.CheckFirmwareTypeRsp
function general_hardware_client:UpdateServiceUpdateServiceCheckFirmwareType(ctx, FilePath, FirmwareType)
    local req = UpdateService.CheckFirmwareTypeReq.new(FilePath, FirmwareType):validate()
    local obj = self:GetUpdateServiceUpdateServiceObject()

    return UpdateService.CheckFirmwareTypeRsp.new(obj:CheckFirmwareType(ctx, req:unpack(true)))
end

function general_hardware_client:PUpdateServiceUpdateServiceCheckFirmwareType(ctx, FilePath, FirmwareType)
    return pcall(function()
        local req = UpdateService.CheckFirmwareTypeReq.new(FilePath, FirmwareType):validate()
        local obj = self:GetUpdateServiceUpdateServiceObject()

        return UpdateService.CheckFirmwareTypeRsp.new(obj:CheckFirmwareType(ctx, req:unpack(true)))
    end)
end

---@param FilePath string
---@return UpdateService.ParseFirmwarePackageRsp
function general_hardware_client:UpdateServiceUpdateServiceParseFirmwarePackage(ctx, FilePath)
    local req = UpdateService.ParseFirmwarePackageReq.new(FilePath):validate()
    local obj = self:GetUpdateServiceUpdateServiceObject()

    return UpdateService.ParseFirmwarePackageRsp.new(obj:ParseFirmwarePackage(ctx, req:unpack(true)))
end

function general_hardware_client:PUpdateServiceUpdateServiceParseFirmwarePackage(ctx, FilePath)
    return pcall(function()
        local req = UpdateService.ParseFirmwarePackageReq.new(FilePath):validate()
        local obj = self:GetUpdateServiceUpdateServiceObject()

        return UpdateService.ParseFirmwarePackageRsp.new(obj:ParseFirmwarePackage(ctx, req:unpack(true)))
    end)
end

---@param FirmwareInfoTable FirmwareInventory.Object
---@param Updateable boolean
---@param Capability integer
---@param MaxPackageSizeMiB integer
---@return FirmwareInventory.AddRsp
function general_hardware_client:FirmwareInventoryFirmwareInventoryAdd(ctx, FirmwareInfoTable, Updateable, Capability,
    MaxPackageSizeMiB)
    local req = FirmwareInventory.AddReq.new(FirmwareInfoTable, Updateable, Capability, MaxPackageSizeMiB):validate()
    local obj = self:GetFirmwareInventoryFirmwareInventoryObject()

    return FirmwareInventory.AddRsp.new(obj:Add(ctx, req:unpack(true)))
end

function general_hardware_client:PFirmwareInventoryFirmwareInventoryAdd(ctx, FirmwareInfoTable, Updateable, Capability,
    MaxPackageSizeMiB)
    return pcall(function()
        local req =
            FirmwareInventory.AddReq.new(FirmwareInfoTable, Updateable, Capability, MaxPackageSizeMiB):validate()
        local obj = self:GetFirmwareInventoryFirmwareInventoryObject()

        return FirmwareInventory.AddRsp.new(obj:Add(ctx, req:unpack(true)))
    end)
end

---@param Id string
---@return FirmwareInventory.DeleteRsp
function general_hardware_client:FirmwareInventoryFirmwareInventoryDelete(ctx, Id)
    local req = FirmwareInventory.DeleteReq.new(Id):validate()
    local obj = self:GetFirmwareInventoryFirmwareInventoryObject()

    return FirmwareInventory.DeleteRsp.new(obj:Delete(ctx, req:unpack(true)))
end

function general_hardware_client:PFirmwareInventoryFirmwareInventoryDelete(ctx, Id)
    return pcall(function()
        local req = FirmwareInventory.DeleteReq.new(Id):validate()
        local obj = self:GetFirmwareInventoryFirmwareInventoryObject()

        return FirmwareInventory.DeleteRsp.new(obj:Delete(ctx, req:unpack(true)))
    end)
end

---@return FirmwareInventory.GetAllFirmwareInfoRsp
function general_hardware_client:FirmwareInventoryFirmwareInventoryGetAllFirmwareInfo(ctx)
    local obj = self:GetFirmwareInventoryFirmwareInventoryObject()

    return FirmwareInventory.GetAllFirmwareInfoRsp.new(obj:GetAllFirmwareInfo(ctx))
end

function general_hardware_client:PFirmwareInventoryFirmwareInventoryGetAllFirmwareInfo(ctx)
    return pcall(function()
        local obj = self:GetFirmwareInventoryFirmwareInventoryObject()

        return FirmwareInventory.GetAllFirmwareInfoRsp.new(obj:GetAllFirmwareInfo(ctx))
    end)
end

---@return EthernetInterfaces.GetAllPortRsp
function general_hardware_client:EthernetInterfacesEthernetInterfacesGetAllPort(ctx, path_params)
    local obj = self:GetEthernetInterfacesEthernetInterfacesObject(path_params)

    return EthernetInterfaces.GetAllPortRsp.new(obj:GetAllPort(ctx))
end

function general_hardware_client:PEthernetInterfacesEthernetInterfacesGetAllPort(ctx, path_params)
    return pcall(function()
        local obj = self:GetEthernetInterfacesEthernetInterfacesObject(path_params)

        return EthernetInterfaces.GetAllPortRsp.new(obj:GetAllPort(ctx))
    end)
end

---@param NetMode string
---@param PortId integer
---@param VLANEnable boolean
---@param VLANId integer
---@return EthernetInterfaces.SetNetworkConfigRsp
function general_hardware_client:EthernetInterfacesEthernetInterfacesSetNetworkConfig(ctx, path_params, NetMode, PortId,
    VLANEnable, VLANId)
    local req = EthernetInterfaces.SetNetworkConfigReq.new(NetMode, PortId, VLANEnable, VLANId):validate()
    local obj = self:GetEthernetInterfacesEthernetInterfacesObject(path_params)

    return EthernetInterfaces.SetNetworkConfigRsp.new(obj:SetNetworkConfig(ctx, req:unpack(true)))
end

function general_hardware_client:PEthernetInterfacesEthernetInterfacesSetNetworkConfig(ctx, path_params, NetMode,
    PortId, VLANEnable, VLANId)
    return pcall(function()
        local req = EthernetInterfaces.SetNetworkConfigReq.new(NetMode, PortId, VLANEnable, VLANId):validate()
        local obj = self:GetEthernetInterfacesEthernetInterfacesObject(path_params)

        return EthernetInterfaces.SetNetworkConfigRsp.new(obj:SetNetworkConfig(ctx, req:unpack(true)))
    end)
end

---@param DeviceId integer
---@param DevicePortId integer
---@param Silkscreen string
---@param EthId integer
---@param Type string
---@param PortId integer
---@return EthernetInterfaces.AddMgmtPortRsp
function general_hardware_client:EthernetInterfacesEthernetInterfacesAddMgmtPort(ctx, path_params, DeviceId,
    DevicePortId, Silkscreen, EthId, Type, PortId)
    local req =
        EthernetInterfaces.AddMgmtPortReq.new(DeviceId, DevicePortId, Silkscreen, EthId, Type, PortId):validate()
    local obj = self:GetEthernetInterfacesEthernetInterfacesObject(path_params)

    return EthernetInterfaces.AddMgmtPortRsp.new(obj:AddMgmtPort(ctx, req:unpack(true)))
end

function general_hardware_client:PEthernetInterfacesEthernetInterfacesAddMgmtPort(ctx, path_params, DeviceId,
    DevicePortId, Silkscreen, EthId, Type, PortId)
    return pcall(function()
        local req =
            EthernetInterfaces.AddMgmtPortReq.new(DeviceId, DevicePortId, Silkscreen, EthId, Type, PortId):validate()
        local obj = self:GetEthernetInterfacesEthernetInterfacesObject(path_params)

        return EthernetInterfaces.AddMgmtPortRsp.new(obj:AddMgmtPort(ctx, req:unpack(true)))
    end)
end

---@param PortId integer
---@return EthernetInterfaces.DeleteMgmtPortRsp
function general_hardware_client:EthernetInterfacesEthernetInterfacesDeleteMgmtPort(ctx, path_params, PortId)
    local req = EthernetInterfaces.DeleteMgmtPortReq.new(PortId):validate()
    local obj = self:GetEthernetInterfacesEthernetInterfacesObject(path_params)

    return EthernetInterfaces.DeleteMgmtPortRsp.new(obj:DeleteMgmtPort(ctx, req:unpack(true)))
end

function general_hardware_client:PEthernetInterfacesEthernetInterfacesDeleteMgmtPort(ctx, path_params, PortId)
    return pcall(function()
        local req = EthernetInterfaces.DeleteMgmtPortReq.new(PortId):validate()
        local obj = self:GetEthernetInterfacesEthernetInterfacesObject(path_params)

        return EthernetInterfaces.DeleteMgmtPortRsp.new(obj:DeleteMgmtPort(ctx, req:unpack(true)))
    end)
end

---@param InterfaceName string
---@param IpAddr string
---@return EthernetInterfaces.AddIptablesRuleRsp
function general_hardware_client:EthernetInterfacesEthernetInterfacesAddIptablesRule(ctx, path_params, InterfaceName,
    IpAddr)
    local req = EthernetInterfaces.AddIptablesRuleReq.new(InterfaceName, IpAddr):validate()
    local obj = self:GetEthernetInterfacesEthernetInterfacesObject(path_params)

    return EthernetInterfaces.AddIptablesRuleRsp.new(obj:AddIptablesRule(ctx, req:unpack(true)))
end

function general_hardware_client:PEthernetInterfacesEthernetInterfacesAddIptablesRule(ctx, path_params, InterfaceName,
    IpAddr)
    return pcall(function()
        local req = EthernetInterfaces.AddIptablesRuleReq.new(InterfaceName, IpAddr):validate()
        local obj = self:GetEthernetInterfacesEthernetInterfacesObject(path_params)

        return EthernetInterfaces.AddIptablesRuleRsp.new(obj:AddIptablesRule(ctx, req:unpack(true)))
    end)
end

---@param InterfaceName string
---@param Ipv6Addr string
---@return EthernetInterfaces.AddIp6tablesRuleRsp
function general_hardware_client:EthernetInterfacesEthernetInterfacesAddIp6tablesRule(ctx, path_params, InterfaceName,
    Ipv6Addr)
    local req = EthernetInterfaces.AddIp6tablesRuleReq.new(InterfaceName, Ipv6Addr):validate()
    local obj = self:GetEthernetInterfacesEthernetInterfacesObject(path_params)

    return EthernetInterfaces.AddIp6tablesRuleRsp.new(obj:AddIp6tablesRule(ctx, req:unpack(true)))
end

function general_hardware_client:PEthernetInterfacesEthernetInterfacesAddIp6tablesRule(ctx, path_params, InterfaceName,
    Ipv6Addr)
    return pcall(function()
        local req = EthernetInterfaces.AddIp6tablesRuleReq.new(InterfaceName, Ipv6Addr):validate()
        local obj = self:GetEthernetInterfacesEthernetInterfacesObject(path_params)

        return EthernetInterfaces.AddIp6tablesRuleRsp.new(obj:AddIp6tablesRule(ctx, req:unpack(true)))
    end)
end

---@param VLANEnabled boolean
---@param VLANId integer
---@param PortType integer
---@return EthernetInterfaces.SetVLANConfigRsp
function general_hardware_client:EthernetInterfacesEthernetInterfacesSetVLANConfig(ctx, path_params, VLANEnabled,
    VLANId, PortType)
    local req = EthernetInterfaces.SetVLANConfigReq.new(VLANEnabled, VLANId, PortType):validate()
    local obj = self:GetEthernetInterfacesEthernetInterfacesObject(path_params)

    return EthernetInterfaces.SetVLANConfigRsp.new(obj:SetVLANConfig(ctx, req:unpack(true)))
end

function general_hardware_client:PEthernetInterfacesEthernetInterfacesSetVLANConfig(ctx, path_params, VLANEnabled,
    VLANId, PortType)
    return pcall(function()
        local req = EthernetInterfaces.SetVLANConfigReq.new(VLANEnabled, VLANId, PortType):validate()
        local obj = self:GetEthernetInterfacesEthernetInterfacesObject(path_params)

        return EthernetInterfaces.SetVLANConfigRsp.new(obj:SetVLANConfig(ctx, req:unpack(true)))
    end)
end

---@param EthName string
---@param EthEnabled boolean
---@return EthernetInterfaces.SetEthStateRsp
function general_hardware_client:EthernetInterfacesEthernetInterfacesSetEthState(ctx, path_params, EthName, EthEnabled)
    local req = EthernetInterfaces.SetEthStateReq.new(EthName, EthEnabled):validate()
    local obj = self:GetEthernetInterfacesEthernetInterfacesObject(path_params)

    return EthernetInterfaces.SetEthStateRsp.new(obj:SetEthState(ctx, req:unpack(true)))
end

function general_hardware_client:PEthernetInterfacesEthernetInterfacesSetEthState(ctx, path_params, EthName, EthEnabled)
    return pcall(function()
        local req = EthernetInterfaces.SetEthStateReq.new(EthName, EthEnabled):validate()
        local obj = self:GetEthernetInterfacesEthernetInterfacesObject(path_params)

        return EthernetInterfaces.SetEthStateRsp.new(obj:SetEthState(ctx, req:unpack(true)))
    end)
end

---@param From string
---@param To string
---@return EthernetInterfaces.NetworkFailoverRsp
function general_hardware_client:EthernetInterfacesEthernetInterfacesNetworkFailover(ctx, path_params, From, To)
    local req = EthernetInterfaces.NetworkFailoverReq.new(From, To):validate()
    local obj = self:GetEthernetInterfacesEthernetInterfacesObject(path_params)

    return EthernetInterfaces.NetworkFailoverRsp.new(obj:NetworkFailover(ctx, req:unpack(true)))
end

function general_hardware_client:PEthernetInterfacesEthernetInterfacesNetworkFailover(ctx, path_params, From, To)
    return pcall(function()
        local req = EthernetInterfaces.NetworkFailoverReq.new(From, To):validate()
        local obj = self:GetEthernetInterfacesEthernetInterfacesObject(path_params)

        return EthernetInterfaces.NetworkFailoverRsp.new(obj:NetworkFailover(ctx, req:unpack(true)))
    end)
end

---@return Pmu.ResetRsp
function general_hardware_client:PmuPmuReset(ctx, path_params)
    local obj = self:GetPmuPmuObject(path_params)

    return Pmu.ResetRsp.new(obj:Reset(ctx))
end

function general_hardware_client:PPmuPmuReset(ctx, path_params)
    return pcall(function()
        local obj = self:GetPmuPmuObject(path_params)

        return Pmu.ResetRsp.new(obj:Reset(ctx))
    end)
end

---@param SystemId integer
---@param IsLocal boolean
---@param CpuId integer
---@param Bus integer
---@param Device integer
---@param Function integer
---@return Pmu.GetPcieVidDidRsp
function general_hardware_client:PmuPmuGetPcieVidDid(ctx, path_params, SystemId, IsLocal, CpuId, Bus, Device, Function)
    local req = Pmu.GetPcieVidDidReq.new(SystemId, IsLocal, CpuId, Bus, Device, Function):validate()
    local obj = self:GetPmuPmuObject(path_params)

    return Pmu.GetPcieVidDidRsp.new(obj:GetPcieVidDid(ctx, req:unpack(true)))
end

function general_hardware_client:PPmuPmuGetPcieVidDid(ctx, path_params, SystemId, IsLocal, CpuId, Bus, Device, Function)
    return pcall(function()
        local req = Pmu.GetPcieVidDidReq.new(SystemId, IsLocal, CpuId, Bus, Device, Function):validate()
        local obj = self:GetPmuPmuObject(path_params)

        return Pmu.GetPcieVidDidRsp.new(obj:GetPcieVidDid(ctx, req:unpack(true)))
    end)
end

---@param SystemId integer
---@param IsLocal boolean
---@param CpuId integer
---@param Bus integer
---@param Device integer
---@param Function integer
---@return Pmu.GetPcieSubVidDidRsp
function general_hardware_client:PmuPmuGetPcieSubVidDid(ctx, path_params, SystemId, IsLocal, CpuId, Bus, Device,
    Function)
    local req = Pmu.GetPcieSubVidDidReq.new(SystemId, IsLocal, CpuId, Bus, Device, Function):validate()
    local obj = self:GetPmuPmuObject(path_params)

    return Pmu.GetPcieSubVidDidRsp.new(obj:GetPcieSubVidDid(ctx, req:unpack(true)))
end

function general_hardware_client:PPmuPmuGetPcieSubVidDid(ctx, path_params, SystemId, IsLocal, CpuId, Bus, Device,
    Function)
    return pcall(function()
        local req = Pmu.GetPcieSubVidDidReq.new(SystemId, IsLocal, CpuId, Bus, Device, Function):validate()
        local obj = self:GetPmuPmuObject(path_params)

        return Pmu.GetPcieSubVidDidRsp.new(obj:GetPcieSubVidDid(ctx, req:unpack(true)))
    end)
end

---@param SystemId integer
---@return FirmwareActive.StartActiveRsp
function general_hardware_client:FirmwareActiveFirmwareActiveStartActive(ctx, SystemId)
    local req = FirmwareActive.StartActiveReq.new(SystemId):validate()
    local obj = self:GetFirmwareActiveFirmwareActiveObject()

    return FirmwareActive.StartActiveRsp.new(obj:StartActive(ctx, req:unpack(true)))
end

function general_hardware_client:PFirmwareActiveFirmwareActiveStartActive(ctx, SystemId)
    return pcall(function()
        local req = FirmwareActive.StartActiveReq.new(SystemId):validate()
        local obj = self:GetFirmwareActiveFirmwareActiveObject()

        return FirmwareActive.StartActiveRsp.new(obj:StartActive(ctx, req:unpack(true)))
    end)
end

---@param SystemId integer
---@param AdditionalOptions FirmwareActive.Object
---@return FirmwareActive.StartActiveWithPolicyRsp
function general_hardware_client:FirmwareActiveFirmwareActiveStartActiveWithPolicy(ctx, SystemId, AdditionalOptions)
    local req = FirmwareActive.StartActiveWithPolicyReq.new(SystemId, AdditionalOptions):validate()
    local obj = self:GetFirmwareActiveFirmwareActiveObject()

    return FirmwareActive.StartActiveWithPolicyRsp.new(obj:StartActiveWithPolicy(ctx, req:unpack(true)))
end

function general_hardware_client:PFirmwareActiveFirmwareActiveStartActiveWithPolicy(ctx, SystemId, AdditionalOptions)
    return pcall(function()
        local req = FirmwareActive.StartActiveWithPolicyReq.new(SystemId, AdditionalOptions):validate()
        local obj = self:GetFirmwareActiveFirmwareActiveObject()

        return FirmwareActive.StartActiveWithPolicyRsp.new(obj:StartActiveWithPolicy(ctx, req:unpack(true)))
    end)
end

---@param ActiveAction FirmwareActive.KeyValueTable[]
---@return FirmwareActive.RegisterActiveActionRsp
function general_hardware_client:FirmwareActiveFirmwareActiveRegisterActiveAction(ctx, ActiveAction)
    local req = FirmwareActive.RegisterActiveActionReq.new(ActiveAction):validate()
    local obj = self:GetFirmwareActiveFirmwareActiveObject()

    return FirmwareActive.RegisterActiveActionRsp.new(obj:RegisterActiveAction(ctx, req:unpack(true)))
end

function general_hardware_client:PFirmwareActiveFirmwareActiveRegisterActiveAction(ctx, ActiveAction)
    return pcall(function()
        local req = FirmwareActive.RegisterActiveActionReq.new(ActiveAction):validate()
        local obj = self:GetFirmwareActiveFirmwareActiveObject()

        return FirmwareActive.RegisterActiveActionRsp.new(obj:RegisterActiveAction(ctx, req:unpack(true)))
    end)
end

---@param ActiveStatus FirmwareActive.KeyValueTable[]
---@return FirmwareActive.UpdateActiveStatusRsp
function general_hardware_client:FirmwareActiveFirmwareActiveUpdateActiveStatus(ctx, ActiveStatus)
    local req = FirmwareActive.UpdateActiveStatusReq.new(ActiveStatus):validate()
    local obj = self:GetFirmwareActiveFirmwareActiveObject()

    return FirmwareActive.UpdateActiveStatusRsp.new(obj:UpdateActiveStatus(ctx, req:unpack(true)))
end

function general_hardware_client:PFirmwareActiveFirmwareActiveUpdateActiveStatus(ctx, ActiveStatus)
    return pcall(function()
        local req = FirmwareActive.UpdateActiveStatusReq.new(ActiveStatus):validate()
        local obj = self:GetFirmwareActiveFirmwareActiveObject()

        return FirmwareActive.UpdateActiveStatusRsp.new(obj:UpdateActiveStatus(ctx, req:unpack(true)))
    end)
end

---@param SystemId integer
---@param FirmwareType string
---@param ProcessResult integer
---@return FirmwareActive.ActiveProcessReplyRsp
function general_hardware_client:FirmwareActiveFirmwareActiveActiveProcessReply(ctx, SystemId, FirmwareType,
    ProcessResult)
    local req = FirmwareActive.ActiveProcessReplyReq.new(SystemId, FirmwareType, ProcessResult):validate()
    local obj = self:GetFirmwareActiveFirmwareActiveObject()

    return FirmwareActive.ActiveProcessReplyRsp.new(obj:ActiveProcessReply(ctx, req:unpack(true)))
end

function general_hardware_client:PFirmwareActiveFirmwareActiveActiveProcessReply(ctx, SystemId, FirmwareType,
    ProcessResult)
    return pcall(function()
        local req = FirmwareActive.ActiveProcessReplyReq.new(SystemId, FirmwareType, ProcessResult):validate()
        local obj = self:GetFirmwareActiveFirmwareActiveObject()

        return FirmwareActive.ActiveProcessReplyRsp.new(obj:ActiveProcessReply(ctx, req:unpack(true)))
    end)
end

---@param Type EyeDiagram.PortType
---@param ScopeSel EyeDiagram.ScopeSel
---@return EyeDiagram.StartTestRsp
function general_hardware_client:BmcDfxEyeDiagramStartTest(ctx, path_params, Type, ScopeSel)
    local req = EyeDiagram.StartTestReq.new(Type, ScopeSel):validate()
    local obj = self:GetBmcDfxEyeDiagramObject(path_params)

    return EyeDiagram.StartTestRsp.new(obj:StartTest(ctx, req:unpack(true)))
end

function general_hardware_client:PBmcDfxEyeDiagramStartTest(ctx, path_params, Type, ScopeSel)
    return pcall(function()
        local req = EyeDiagram.StartTestReq.new(Type, ScopeSel):validate()
        local obj = self:GetBmcDfxEyeDiagramObject(path_params)

        return EyeDiagram.StartTestRsp.new(obj:StartTest(ctx, req:unpack(true)))
    end)
end

---@param Row integer
---@param Cols integer
---@param CompareValue integer
---@return EyeDiagram.GetResultRsp
function general_hardware_client:BmcDfxEyeDiagramGetResult(ctx, path_params, Row, Cols, CompareValue)
    local req = EyeDiagram.GetResultReq.new(Row, Cols, CompareValue):validate()
    local obj = self:GetBmcDfxEyeDiagramObject(path_params)

    return EyeDiagram.GetResultRsp.new(obj:GetResult(ctx, req:unpack(true)))
end

function general_hardware_client:PBmcDfxEyeDiagramGetResult(ctx, path_params, Row, Cols, CompareValue)
    return pcall(function()
        local req = EyeDiagram.GetResultReq.new(Row, Cols, CompareValue):validate()
        local obj = self:GetBmcDfxEyeDiagramObject(path_params)

        return EyeDiagram.GetResultRsp.new(obj:GetResult(ctx, req:unpack(true)))
    end)
end

---@param SrcFile string
---@param DstFile string
---@param Uid integer
---@param Gid integer
---@return File.CopyRsp
function general_hardware_client:FileFileCopy(ctx, path_params, SrcFile, DstFile, Uid, Gid)
    local req = File.CopyReq.new(SrcFile, DstFile, Uid, Gid):validate()
    local obj = self:GetFileFileObject(path_params)

    return File.CopyRsp.new(obj:Copy(ctx, req:unpack(true)))
end

function general_hardware_client:PFileFileCopy(ctx, path_params, SrcFile, DstFile, Uid, Gid)
    return pcall(function()
        local req = File.CopyReq.new(SrcFile, DstFile, Uid, Gid):validate()
        local obj = self:GetFileFileObject(path_params)

        return File.CopyRsp.new(obj:Copy(ctx, req:unpack(true)))
    end)
end

---@param SrcFile string
---@param DstFile string
---@param Uid integer
---@param Gid integer
---@return File.MoveRsp
function general_hardware_client:FileFileMove(ctx, path_params, SrcFile, DstFile, Uid, Gid)
    local req = File.MoveReq.new(SrcFile, DstFile, Uid, Gid):validate()
    local obj = self:GetFileFileObject(path_params)

    return File.MoveRsp.new(obj:Move(ctx, req:unpack(true)))
end

function general_hardware_client:PFileFileMove(ctx, path_params, SrcFile, DstFile, Uid, Gid)
    return pcall(function()
        local req = File.MoveReq.new(SrcFile, DstFile, Uid, Gid):validate()
        local obj = self:GetFileFileObject(path_params)

        return File.MoveRsp.new(obj:Move(ctx, req:unpack(true)))
    end)
end

---@param DstFile string
---@param FileMode integer
---@return File.ChmodRsp
function general_hardware_client:FileFileChmod(ctx, path_params, DstFile, FileMode)
    local req = File.ChmodReq.new(DstFile, FileMode):validate()
    local obj = self:GetFileFileObject(path_params)

    return File.ChmodRsp.new(obj:Chmod(ctx, req:unpack(true)))
end

function general_hardware_client:PFileFileChmod(ctx, path_params, DstFile, FileMode)
    return pcall(function()
        local req = File.ChmodReq.new(DstFile, FileMode):validate()
        local obj = self:GetFileFileObject(path_params)

        return File.ChmodRsp.new(obj:Chmod(ctx, req:unpack(true)))
    end)
end

---@param DstFile string
---@param Uid integer
---@param Gid integer
---@return File.ChownRsp
function general_hardware_client:FileFileChown(ctx, path_params, DstFile, Uid, Gid)
    local req = File.ChownReq.new(DstFile, Uid, Gid):validate()
    local obj = self:GetFileFileObject(path_params)

    return File.ChownRsp.new(obj:Chown(ctx, req:unpack(true)))
end

function general_hardware_client:PFileFileChown(ctx, path_params, DstFile, Uid, Gid)
    return pcall(function()
        local req = File.ChownReq.new(DstFile, Uid, Gid):validate()
        local obj = self:GetFileFileObject(path_params)

        return File.ChownRsp.new(obj:Chown(ctx, req:unpack(true)))
    end)
end

---@param DstFile string
---@return File.DeleteRsp
function general_hardware_client:FileFileDelete(ctx, path_params, DstFile)
    local req = File.DeleteReq.new(DstFile):validate()
    local obj = self:GetFileFileObject(path_params)

    return File.DeleteRsp.new(obj:Delete(ctx, req:unpack(true)))
end

function general_hardware_client:PFileFileDelete(ctx, path_params, DstFile)
    return pcall(function()
        local req = File.DeleteReq.new(DstFile):validate()
        local obj = self:GetFileFileObject(path_params)

        return File.DeleteRsp.new(obj:Delete(ctx, req:unpack(true)))
    end)
end

---@param DstFile string
---@param OpenMode string
---@param FileMode integer
---@param Uid integer
---@param Gid integer
---@return File.CreateRsp
function general_hardware_client:FileFileCreate(ctx, path_params, DstFile, OpenMode, FileMode, Uid, Gid)
    local req = File.CreateReq.new(DstFile, OpenMode, FileMode, Uid, Gid):validate()
    local obj = self:GetFileFileObject(path_params)

    return File.CreateRsp.new(obj:Create(ctx, req:unpack(true)))
end

function general_hardware_client:PFileFileCreate(ctx, path_params, DstFile, OpenMode, FileMode, Uid, Gid)
    return pcall(function()
        local req = File.CreateReq.new(DstFile, OpenMode, FileMode, Uid, Gid):validate()
        local obj = self:GetFileFileObject(path_params)

        return File.CreateRsp.new(obj:Create(ctx, req:unpack(true)))
    end)
end

---@param DstFile string
---@return File.ChangeOwnerRsp
function general_hardware_client:FileFileChangeOwner(ctx, path_params, DstFile)
    local req = File.ChangeOwnerReq.new(DstFile):validate()
    local obj = self:GetFileFileObject(path_params)

    return File.ChangeOwnerRsp.new(obj:ChangeOwner(ctx, req:unpack(true)))
end

function general_hardware_client:PFileFileChangeOwner(ctx, path_params, DstFile)
    return pcall(function()
        local req = File.ChangeOwnerReq.new(DstFile):validate()
        local obj = self:GetFileFileObject(path_params)

        return File.ChangeOwnerRsp.new(obj:ChangeOwner(ctx, req:unpack(true)))
    end)
end

---@param DstFile string
---@param Permission string
---@return File.IsPermittedRsp
function general_hardware_client:FileFileIsPermitted(ctx, path_params, DstFile, Permission)
    local req = File.IsPermittedReq.new(DstFile, Permission):validate()
    local obj = self:GetFileFileObject(path_params)

    return File.IsPermittedRsp.new(obj:IsPermitted(ctx, req:unpack(true)))
end

function general_hardware_client:PFileFileIsPermitted(ctx, path_params, DstFile, Permission)
    return pcall(function()
        local req = File.IsPermittedReq.new(DstFile, Permission):validate()
        local obj = self:GetFileFileObject(path_params)

        return File.IsPermittedRsp.new(obj:IsPermitted(ctx, req:unpack(true)))
    end)
end

---@param Request string
---@return Sms.SetForwardWhiteListRsp
function general_hardware_client:SmsSmsSetForwardWhiteList(ctx, path_params, Request)
    local req = Sms.SetForwardWhiteListReq.new(Request):validate()
    local obj = self:GetSmsSmsObject(path_params)

    return Sms.SetForwardWhiteListRsp.new(obj:SetForwardWhiteList(ctx, req:unpack(true)))
end

function general_hardware_client:PSmsSmsSetForwardWhiteList(ctx, path_params, Request)
    return pcall(function()
        local req = Sms.SetForwardWhiteListReq.new(Request):validate()
        local obj = self:GetSmsSmsObject(path_params)

        return Sms.SetForwardWhiteListRsp.new(obj:SetForwardWhiteList(ctx, req:unpack(true)))
    end)
end

---@param Request string
---@param RetryTimes integer
---@return Sms.ForwardRequestRsp
function general_hardware_client:SmsSmsForwardRequest(ctx, path_params, Request, RetryTimes)
    local req = Sms.ForwardRequestReq.new(Request, RetryTimes):validate()
    local obj = self:GetSmsSmsObject(path_params)

    return Sms.ForwardRequestRsp.new(obj:ForwardRequest(ctx, req:unpack(true)))
end

function general_hardware_client:PSmsSmsForwardRequest(ctx, path_params, Request, RetryTimes)
    return pcall(function()
        local req = Sms.ForwardRequestReq.new(Request, RetryTimes):validate()
        local obj = self:GetSmsSmsObject(path_params)

        return Sms.ForwardRequestRsp.new(obj:ForwardRequest(ctx, req:unpack(true)))
    end)
end

---@param RequestMethod string
---@param RequestToken string
---@param RequestBody string
---@return Sms.ForwardEventRsp
function general_hardware_client:SmsSmsForwardEvent(ctx, path_params, RequestMethod, RequestToken, RequestBody)
    local req = Sms.ForwardEventReq.new(RequestMethod, RequestToken, RequestBody):validate()
    local obj = self:GetSmsSmsObject(path_params)

    return Sms.ForwardEventRsp.new(obj:ForwardEvent(ctx, req:unpack(true)))
end

function general_hardware_client:PSmsSmsForwardEvent(ctx, path_params, RequestMethod, RequestToken, RequestBody)
    return pcall(function()
        local req = Sms.ForwardEventReq.new(RequestMethod, RequestToken, RequestBody):validate()
        local obj = self:GetSmsSmsObject(path_params)

        return Sms.ForwardEventRsp.new(obj:ForwardEvent(ctx, req:unpack(true)))
    end)
end

---@param ResourceType string
---@return Sms.UpdateSpecificResourceRsp
function general_hardware_client:SmsSmsUpdateSpecificResource(ctx, path_params, ResourceType)
    local req = Sms.UpdateSpecificResourceReq.new(ResourceType):validate()
    local obj = self:GetSmsSmsObject(path_params)

    return Sms.UpdateSpecificResourceRsp.new(obj:UpdateSpecificResource(ctx, req:unpack(true)))
end

function general_hardware_client:PSmsSmsUpdateSpecificResource(ctx, path_params, ResourceType)
    return pcall(function()
        local req = Sms.UpdateSpecificResourceReq.new(ResourceType):validate()
        local obj = self:GetSmsSmsObject(path_params)

        return Sms.UpdateSpecificResourceRsp.new(obj:UpdateSpecificResource(ctx, req:unpack(true)))
    end)
end

---@param ChannelType integer
---@param IsPowerOffPer boolean
---@return Sms.SetChannelTypeRsp
function general_hardware_client:SmsSmsSetChannelType(ctx, path_params, ChannelType, IsPowerOffPer)
    local req = Sms.SetChannelTypeReq.new(ChannelType, IsPowerOffPer):validate()
    local obj = self:GetSmsSmsObject(path_params)

    return Sms.SetChannelTypeRsp.new(obj:SetChannelType(ctx, req:unpack(true)))
end

function general_hardware_client:PSmsSmsSetChannelType(ctx, path_params, ChannelType, IsPowerOffPer)
    return pcall(function()
        local req = Sms.SetChannelTypeReq.new(ChannelType, IsPowerOffPer):validate()
        local obj = self:GetSmsSmsObject(path_params)

        return Sms.SetChannelTypeRsp.new(obj:SetChannelType(ctx, req:unpack(true)))
    end)
end

---@param IsPowerOffPer boolean
---@return Sms.GetChannelTypeRsp
function general_hardware_client:SmsSmsGetChannelType(ctx, path_params, IsPowerOffPer)
    local req = Sms.GetChannelTypeReq.new(IsPowerOffPer):validate()
    local obj = self:GetSmsSmsObject(path_params)

    return Sms.GetChannelTypeRsp.new(obj:GetChannelType(ctx, req:unpack(true)))
end

function general_hardware_client:PSmsSmsGetChannelType(ctx, path_params, IsPowerOffPer)
    return pcall(function()
        local req = Sms.GetChannelTypeReq.new(IsPowerOffPer):validate()
        local obj = self:GetSmsSmsObject(path_params)

        return Sms.GetChannelTypeRsp.new(obj:GetChannelType(ctx, req:unpack(true)))
    end)
end

---@param Enabled boolean
---@return Sms.SetCdevChannelEnabledRsp
function general_hardware_client:SmsSmsSetCdevChannelEnabled(ctx, path_params, Enabled)
    local req = Sms.SetCdevChannelEnabledReq.new(Enabled):validate()
    local obj = self:GetSmsSmsObject(path_params)

    return Sms.SetCdevChannelEnabledRsp.new(obj:SetCdevChannelEnabled(ctx, req:unpack(true)))
end

function general_hardware_client:PSmsSmsSetCdevChannelEnabled(ctx, path_params, Enabled)
    return pcall(function()
        local req = Sms.SetCdevChannelEnabledReq.new(Enabled):validate()
        local obj = self:GetSmsSmsObject(path_params)

        return Sms.SetCdevChannelEnabledRsp.new(obj:SetCdevChannelEnabled(ctx, req:unpack(true)))
    end)
end

---@param ModulePath string
---@param ExtendParams Driver.ExtendParams
---@return Driver.InsmodRsp
function general_hardware_client:ProxyDriverInsmod(ctx, path_params, ModulePath, ExtendParams)
    local req = Driver.InsmodReq.new(ModulePath, ExtendParams):validate()
    local obj = self:GetProxyDriverObject(path_params)

    return Driver.InsmodRsp.new(obj:Insmod(ctx, req:unpack(true)))
end

function general_hardware_client:PProxyDriverInsmod(ctx, path_params, ModulePath, ExtendParams)
    return pcall(function()
        local req = Driver.InsmodReq.new(ModulePath, ExtendParams):validate()
        local obj = self:GetProxyDriverObject(path_params)

        return Driver.InsmodRsp.new(obj:Insmod(ctx, req:unpack(true)))
    end)
end

---@param ModulePath string
---@param ExtendParams Driver.ExtendParams
---@return Driver.RmmodRsp
function general_hardware_client:ProxyDriverRmmod(ctx, path_params, ModulePath, ExtendParams)
    local req = Driver.RmmodReq.new(ModulePath, ExtendParams):validate()
    local obj = self:GetProxyDriverObject(path_params)

    return Driver.RmmodRsp.new(obj:Rmmod(ctx, req:unpack(true)))
end

function general_hardware_client:PProxyDriverRmmod(ctx, path_params, ModulePath, ExtendParams)
    return pcall(function()
        local req = Driver.RmmodReq.new(ModulePath, ExtendParams):validate()
        local obj = self:GetProxyDriverObject(path_params)

        return Driver.RmmodRsp.new(obj:Rmmod(ctx, req:unpack(true)))
    end)
end

function general_hardware_client:SubscribeSystemsFruCtrlBeforePowerOnSignal(cb)
    local sig = match_rule.signal('BeforePowerOnSignal', 'bmc.kepler.Systems.FruCtrl')
    self.signal_slots[#self.signal_slots + 1] = self:get_bus():match(sig, function(msg)
        cb(msg:read())
    end)
end

function general_hardware_client:SubscribeUpdateServiceUpdateServiceUpgradePrepareSignal(cb)
    local sig = match_rule.signal('UpgradePrepareSignal', 'bmc.kepler.UpdateService'):with_path(
        '/bmc/kepler/UpdateService')
    self.signal_slots[#self.signal_slots + 1] = self:get_bus():match(sig, function(msg)
        cb(msg:read())
    end)
end

function general_hardware_client:SubscribeUpdateServiceUpdateServiceUpgradeProcessSignal(cb)
    local sig = match_rule.signal('UpgradeProcessSignal', 'bmc.kepler.UpdateService'):with_path(
        '/bmc/kepler/UpdateService')
    self.signal_slots[#self.signal_slots + 1] = self:get_bus():match(sig, function(msg)
        cb(msg:read())
    end)
end

function general_hardware_client:SubscribeUpdateServiceUpdateServiceUpgradeFinishSignal(cb)
    local sig = match_rule.signal('UpgradeFinishSignal', 'bmc.kepler.UpdateService'):with_path(
        '/bmc/kepler/UpdateService')
    self.signal_slots[#self.signal_slots + 1] = self:get_bus():match(sig, function(msg)
        cb(msg:read())
    end)
end

function general_hardware_client:SubscribeUpdateServiceUpdateServiceRollbackSignal(cb)
    local sig = match_rule.signal('RollbackSignal', 'bmc.kepler.UpdateService'):with_path('/bmc/kepler/UpdateService')
    self.signal_slots[#self.signal_slots + 1] = self:get_bus():match(sig, function(msg)
        cb(msg:read())
    end)
end

function general_hardware_client:SubscribeUpdateServiceUpdateServiceExportSignal(cb)
    local sig = match_rule.signal('ExportSignal', 'bmc.kepler.UpdateService'):with_path('/bmc/kepler/UpdateService')
    self.signal_slots[#self.signal_slots + 1] = self:get_bus():match(sig, function(msg)
        cb(msg:read())
    end)
end

function general_hardware_client:SubscribeEthernetInterfacesEthernetInterfacesActivePortChangedSignal(cb)
    local sig = match_rule.signal('ActivePortChangedSignal', 'bmc.kepler.Managers.EthernetInterfaces'):with_path(
        '/bmc/kepler/Managers/:Id/EthernetInterfaces')
    self.signal_slots[#self.signal_slots + 1] = self:get_bus():match(sig, function(msg)
        cb(msg:read())
    end)
end

function general_hardware_client:SubscribeEthernetInterfacesEthernetInterfacesNCSIInfoChangedSignal(cb)
    local sig = match_rule.signal('NCSIInfoChangedSignal', 'bmc.kepler.Managers.EthernetInterfaces'):with_path(
        '/bmc/kepler/Managers/:Id/EthernetInterfaces')
    self.signal_slots[#self.signal_slots + 1] = self:get_bus():match(sig, function(msg)
        cb(msg:read())
    end)
end

function general_hardware_client:SubscribeEthernetInterfacesEthernetInterfacesEthMacChangedSignal(cb)
    local sig = match_rule.signal('EthMacChangedSignal', 'bmc.kepler.Managers.EthernetInterfaces'):with_path(
        '/bmc/kepler/Managers/:Id/EthernetInterfaces')
    self.signal_slots[#self.signal_slots + 1] = self:get_bus():match(sig, function(msg)
        cb(msg:read())
    end)
end

function general_hardware_client:SubscribeIpv4ChangedSignal(cb)
    local sig = match_rule.signal('ChangedSignal', 'bmc.kepler.Managers.EthernetInterfaces.Ipv4')
    self.signal_slots[#self.signal_slots + 1] = self:get_bus():match(sig, function(msg)
        cb(msg:read())
    end)
end

function general_hardware_client:SubscribeIpv6ChangedSignal(cb)
    local sig = match_rule.signal('ChangedSignal', 'bmc.kepler.Managers.EthernetInterfaces.Ipv6')
    self.signal_slots[#self.signal_slots + 1] = self:get_bus():match(sig, function(msg)
        cb(msg:read())
    end)
end

function general_hardware_client:SubscribeFirmwareActiveFirmwareActiveActiveProcessSignal(cb)
    local sig = match_rule.signal('ActiveProcessSignal', 'bmc.kepler.UpdateService.FirmwareActive'):with_path(
        '/bmc/kepler/UpdateService/FirmwareActive')
    self.signal_slots[#self.signal_slots + 1] = self:get_bus():match(sig, function(msg)
        cb(msg:read())
    end)
end

function general_hardware_client:ctor()
    self.signal_slots = {}
end

---@type general_hardware_client
return general_hardware_client.new('general_hardware')
