-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at:
--         http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

local log = require 'mc.logging'
local json = require 'cjson'
local mc_ctx = require 'mc.context'

local configuration = {}
local ctx = mc_ctx.new('IT', 'Admin', '127.0.0.1')

local function test_config_datas()
    local datas = {
        SystemSerialNumber = {Import = true, Value = 'sys_test'},
        SystemProductName = {Import = true, Value = 'TestS'},
        AssetTag = {Import = true, Value = 'test_a'},
        ManufacturerName = {Import = true, Value = 'Test-P'},
        ProductCustomInfo = {Import = true, Value = 'DES=789'},
        ProductName = {Import = true, Value = 'TestP'},
        ProductPartNumber = {Import = true, Value = '03456'},
        ProductSerialNumber = {Import = true, Value = '789456'},
        BoardCustomInfo = {Import = true, Value = 'EX=123'},
        BoardManufacturer = {Import = true, Value = 'Test-B'},
        BoardPartNumber = {Import = true, Value = '234512'},
        BoardProductName = {Import = true, Value = 'TestB'},
        BoardSerialNumber = {Import = true, Value = '234456'},
        ProductVersion = {Import = true, Value = 'test.106.01'},
        ChassisType = {Import = true, Value = 'Rack Mount Chassis'},
        ChassisPartNumber = {Import = true, Value = '123456'},
        ChassisType = {Import = true, Value = 'Other'}
    }
    return datas
end

local service = 'bmc.kepler.frudata'
local path = '/bmc/kepler/frudata/MicroComponent'
local intf = 'bmc.kepler.MicroComponent.ConfigManage'

local function test_on_import(bus, config_data)
    local data = json.encode( { ConfigData = { FRU = config_data } } )
    bus:call(service, path, intf, 'Import', 'a{ss}ss', ctx, data, 'configuration')
end

local function test_on_backup(bus, config_data)
    local data = json.encode( { ConfigData = { FRU = config_data } } )
    bus:call(service, path, intf, 'Backup', 'a{ss}s', ctx, data, 'configuration')
end

local function test_on_export(bus, config_data)
    local datas = bus:call(service, path, intf, 'Export', 'a{ss}s', ctx, 'configuration')
    local export_data = json.decode(datas)
    local import_data = config_data

    for key, export_value in pairs(export_data.ConfigData.FRU) do
        local import_value = import_data[key].Value
        local ok = false
        if key == 'ProductCustomInfo' or key == 'BoardCustomInfo' then
            ok = string.find(export_value, import_value) and true or false
        else
            ok = import_value == export_value
        end
        assert(ok, string.format('import %s failed, import value(%s), export value(%s)',
            key, import_value, export_value))
    end
end

function configuration:test_configuration(bus)
    log:notice('============== test configuration start ==============')

    test_on_import(bus, test_config_datas())
    test_on_export(bus, test_config_datas())
    test_on_backup(bus, test_config_datas())
    -- 测试非法参数
    local ok = pcall(function ()
        test_on_import(bus, {ChassisType = {Import = true, Value = 'XXXX'}})
    end)
    assert(not ok)

    log:notice('============== test configuration end ==============')
end

return configuration