-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at:
--         http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

local skynet = require 'skynet'
local sdbus = require 'sd_bus'
local log = require 'mc.logging'
local utils = require 'mc.utils'
local lu = require 'luaunit'
local test_common = require 'test_common.utils'
local mdb = require 'mc.mdb'
local context = require 'mc.context'
require 'skynet.manager'
require 'chassis.json_types.Led'
local bs = require 'mc.bitstring'
local ipmi = require 'ipmi'
require 'ipmi'
local test_ipmi = require 'test_ipmi'
local test_manufacture = require 'test_manufacture.test_manufacture'
local test_metric = require 'test_metric'
local test_business_config = require 'test_config_mgmt.test_business_config'
local LED_PATH = '/bmc/kepler/Systems/1/Leds/Led_UIDLed_0101'
local LED_INTERFACE = 'bmc.kepler.Systems.Led'

local project_dir = os.getenv('PROJECT_DIR')
local test_dir = project_dir .. '/test/integration/.test_temp_data/'
skynet.setenv('PROG_APP_PATH', test_dir .. 'apps')
skynet.setenv('PROG_CSR_PATH', test_dir .. 'sr')
skynet.setenv('DATA_CSR_PATH', test_dir .. 'data/sr')

-- 准备测试专用路径
local function prepare_test_data()
    log:info('- prepare test data')
    local test_data_dir = skynet.getenv('TEST_DATA_DIR')
    local dir_list = {'apps/chassis/mds', 'apps/hwdiscovery/mds', 'apps/frudata/mds', 'apps/sensor/mds',
        'sr', 'data', 'usr/lib64', '/apps/ipmi_core/mds', 'apps/fructrl/mds'}
    for _, path in pairs(dir_list) do
        os.execute('mkdir -p ' .. test_data_dir .. '/' .. path)
    end
    os.execute('tar -xzvf temp/test_data/apps/hwproxy/mockdata.tar.gz -C ' .. test_data_dir .. 'data')
    utils.copy_file('temp/usr/lib64/mock/libsoc_adapter_it.so',
        test_data_dir .. '/usr/lib64/libsoc_adapter.so')
    utils.copy_file('test/integration/test_data/14100513_BCU_01.csr',
        test_data_dir .. '/sr/14100513_BCU_01.sr')
    utils.copy_file('test/integration/test_data/14100513_EXU_01.csr',
        test_data_dir .. '/sr/14100513_EXU_01.sr')
    utils.copy_file('test/integration/test_data/platform.sr', test_data_dir .. '/sr/platform.sr')    
    utils.copy_file('test/integration/test_data/root.csr', test_data_dir .. '/sr/root.sr')
    utils.copy_file('mds/schema.json', test_data_dir .. '/apps/chassis/mds/schema.json')
    utils.copy_file('temp/opt/bmc/apps/hwdiscovery/mds/schema.json',
        test_data_dir .. '/apps/hwdiscovery/mds/schema.json')
    utils.copy_file('temp/opt/bmc/apps/frudata/mds/schema.json',
        test_data_dir .. '/apps/frudata/mds/schema.json')
        utils.copy_file('temp/opt/bmc/apps/fructrl/mds/schema.json',
            test_data_dir .. '/apps/fructrl/mds/schema.json')
    utils.copy_file('temp/opt/bmc/apps/sensor/mds/schema.json',
        test_data_dir .. '/apps/sensor/mds/schema.json')
    utils.copy_file('temp/opt/bmc/apps/ipmi_core/mds/schema.json',
        test_data_dir .. '/apps/ipmi_core/mds/schema.json')
end

-- 删除路径
local function clear_test_data(exit_test)
    log:info('clear test data')
    local test_data_dir = skynet.getenv('TEST_DATA_DIR')
    if not exit_test then
        return utils.remove_file(test_data_dir)
    end

    skynet.timeout(0, function()
        skynet.sleep(20)
        skynet.abort()
        utils.remove_file(test_data_dir)
    end)
end

-- 测试设置uid灯的状态
local function test_set_uidled_identify(bus)
    log:info('- test_set_uidled_identify start')
    local obj = mdb.get_object(bus, LED_PATH, LED_INTERFACE)
    -- 设置uid灯为闪烁状态
    local resp1 = obj:SetState_PACKED(context.new(), 5, 0xFF)
    lu.assertEquals(resp1.State, 5)
    lu.assertEquals(resp1.OnDuration, 50)
    lu.assertEquals(resp1.OffDuration, 50)
    -- 测试uid灯为点亮状态
    local resp2 = obj:SetState_PACKED(context.new(), 0xFF, 0)
    lu.assertEquals(resp2.State, 255)
    lu.assertEquals(resp2.OnDuration, 255)
    lu.assertEquals(resp2.OffDuration, 0)
    local resp3 = obj:SetState_PACKED(context.new(), 0, 0)
    lu.assertEquals(resp3.State, 0)
    lu.assertEquals(resp3.OnDuration, 0)
    lu.assertEquals(resp3.OffDuration, 0)
    log:info('- test_set_uidled_identify finish')
end

local function test_run()
    log:info('================ test start ================')
    local bus = sdbus.open_user(true)
    test_set_uidled_identify(bus)
    test_ipmi.main(bus)
    test_manufacture.test_all_dft_method(bus)
    test_metric.test_all_metric_method(bus)
    test_business_config.main(bus)
    skynet.call('chassis', 'lua', 'exit')
    log:info('================ test complete ================')
    skynet.sleep(20)
    skynet.abort()
end

skynet.start(function()
    clear_test_data(false)
    prepare_test_data()
    test_common.dbus_launch()
    skynet.uniqueservice('sd_bus')
    skynet.uniqueservice('persistence/service/main')
    skynet.uniqueservice('maca/service/main')
    skynet.uniqueservice('main')
    skynet.uniqueservice('hwproxy/service/main')
    skynet.uniqueservice('hwdiscovery/service/main')
    skynet.uniqueservice('ipmi_core/service/main')
    skynet.uniqueservice('frudata/service/main')
    skynet.uniqueservice('fructrl/service/main')
    skynet.uniqueservice('sensor/service/main')
    skynet.sleep(200)
    skynet.fork(function()
        local ok, err = pcall(test_run)
        clear_test_data(true)
        if not ok then
            log:error('=== test failed ===: %s', err)
        end
    end)
end)
