-- Copyright (c) 2025 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at: http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

local bs_util = require 'util.base_util'
local file_service = require 'service.file_service'
local bios_service = require 'service.bios_service'
local ipmi = require 'ipmi'
local comp_code = ipmi.types.Cc
local lu = require 'luaunit'
local open_db = require 'bios.db'

local test_common = require 'test_common.utils'
local bus = test_common.dbus_launch('../.dbus', nil, true)

TestIpmiConfiguration = {}

local function construct_db()
    local ok, datas = pcall(require, 'bios.datas')
    if not ok then
        -- 如果没有datas配置，证明当前组件不需要datas，仅打开数据库
        datas = nil
    end
    local db = open_db(':memory:', datas)
    return db
end

local function construct_ctx()
    local ctx = {}
    ctx.ChanType = 1
    ctx.get_initiator = function()
        return {}
    end
    ctx.HostId = 1
    return ctx
end

function TestIpmiConfiguration:test_get_bios_configuration()
    local bios_ser = bios_service.new(bus, construct_db())
    local req = {SystemId = 1}
    req.ManufactureId = ''
    local bios_object_collection = bios_ser.bios_object_collection
    bios_ser.bios_object_collection = {[1] = {}}
    local result = bios_ser:get_bios_configuration(req, construct_ctx())
    lu.assertEquals(result, comp_code.InvalidCommand)
    req.ManufactureId = bios_ser:get_manu_id()
    req.Type = 4
    result = bios_ser:get_bios_configuration(req, construct_ctx())
    lu.assertEquals(result, comp_code.ParmOutOfRange)
    req.Type = 0
    req.Info = '\x05\x41\x42\x43\x44'
    result = bios_ser:get_bios_configuration(req, construct_ctx())
    lu.assertEquals(result, comp_code.ReqDataLenInvalid)
    req.Info = '\x05\x41\x42\x43\x44\x45'
    local get_file_path = file_service.get_file_path
    local get_file_json = bs_util.get_file_json
    file_service.get_file_path = function ()
    end
    bs_util.get_file_json = function ()
    end
    result = bios_ser:get_bios_configuration(req, construct_ctx())
    lu.assertEquals(result, comp_code.ParmOutOfRange)

    bs_util.get_file_json = function ()
        return  {
            ABCDE = 'AAA'
        }
    end
    local data
    result, _, _, data = bios_ser:get_bios_configuration(req, construct_ctx())
    lu.assertEquals(result, comp_code.Success)
    lu.assertEquals(data, '\x00\x03\x41\x41\x41')
    req.Type = 1
    bs_util.get_file_json = function ()
        return  {
            ABCDE = 256
        }
    end
    result, _, _, data = bios_ser:get_bios_configuration(req, construct_ctx())
    lu.assertEquals(result, comp_code.Success)
    lu.assertEquals(data, '\x01\x02\x00\x01')

    file_service.get_file_path = get_file_path
    bs_util.get_file_json = get_file_json
    bios_ser.bios_object_collection = bios_object_collection
end

function TestIpmiConfiguration:test_set_bios_configuration()
    local bios_ser = bios_service.new(bus, construct_db())
    local req = {SystemId = 1}
    req.ManufactureId = ''
    local bios_object_collection = bios_ser.bios_object_collection
    bios_ser.bios_object_collection = {[1] = {}}
    local result = bios_ser:set_bios_configuration(req, construct_ctx())
    lu.assertEquals(result, comp_code.InvalidCommand)
    req.ManufactureId = bios_ser:get_manu_id()
    req.Info = '\x01\x02'
    result = bios_ser:set_bios_configuration(req, construct_ctx())
    lu.assertEquals(result, comp_code.ReqDataLenInvalid)
    req.Info = '\x01\x41\x00\x02\x41'
    result = bios_ser:set_bios_configuration(req, construct_ctx())
    lu.assertEquals(result, comp_code.ReqDataLenInvalid)
    req.Info = '\x01\x41\x02\x01\x41'
    result = bios_ser:set_bios_configuration(req, construct_ctx())
    lu.assertEquals(result, comp_code.ParmOutOfRange)
    req.Info = '\x01\x41\x00\x01\x41'
    local import_setup_json = file_service.import_setup_json
    file_service.import_setup_json = function ()
        error()
    end
    result = bios_ser:set_bios_configuration(req, construct_ctx())
    lu.assertEquals(result, comp_code.ParmOutOfRange)

    file_service.import_setup_json = function (_, ctx, file_type, context, system_id)
        if context ~= '{"Attributes":{"A":"A"}}' then
            error()
        end
    end
    result = bios_ser:set_bios_configuration(req, construct_ctx())
    lu.assertEquals(result, comp_code.Success)

    req.Info = '\x01\x41\x01\x02\x00\x01'
    file_service.import_setup_json = function (_, ctx, file_type, context, system_id)
        if context ~= '{"Attributes":{"A":256}}' then
            error()
        end
    end
    result = bios_ser:set_bios_configuration(req, construct_ctx())
    lu.assertEquals(result, comp_code.Success)

    file_service.import_setup_json = import_setup_json
    bios_ser.bios_object_collection = bios_object_collection
end