-- Copyright (c) 2024 Huawei Technologies Co., Ltd.
-- openUBMC is licensed under Mulan PSL v2.
-- You can use this software according to the terms and conditions of the Mulan PSL v2.
-- You may obtain a copy of Mulan PSL v2 at:
--         http://license.coscl.org.cn/MulanPSL2
-- THIS SOFTWARE IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OF ANY KIND,
-- EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO NON-INFRINGEMENT,
-- MERCHANTABILITY OR FIT FOR A PARTICULAR PURPOSE.
-- See the Mulan PSL v2 for more details.

local bios_service = require 'service.bios_service'
local boot_service = require 'service.boot_service'
local boot_option_service = require 'service.boot_options_service'
local bios_factory = require 'factory.bios_factory'
local ipmi = require 'ipmi'
local comp_code = ipmi.types.Cc
local lu = require 'luaunit'
local prop_hander = require 'handler.bios_prop_handler'

local test_common = require 'test_common.utils'
local bus = test_common.dbus_launch('../.dbus', nil, true)

TestBiosIpmi = {}

local function save_func()
end

function TestBiosIpmi:test_bean()
    local option_db_obj = {}
    option_db_obj.BootFlags = '/0/0/3/4/5'
    option_db_obj.ValidFlag = '/0/0/0/0/0/0/7/8/9/10/11/12/13/14/15/16'
    option_db_obj.save = save_func

    local boot_db_obj = {}
    boot_db_obj.BootSourceOverrideMode = 'UEFIBoot'
    boot_db_obj.BootOrder = '/HardDiskDrive/PXE/DVDROMDrive/Others'
    boot_db_obj.save = save_func

    local bios_ser = bios_service.get_instance(bus, {})
    local boot_ser = boot_service.get_instance(boot_db_obj)
    local boot_option_ser = boot_option_service.get_instance(option_db_obj)

    bios_ser:set_prop('RegistryVersion', '1.1')
    bios_factory.register_bean('bios_service', bios_ser)
    bios_factory.register_bean('boot_service', boot_ser)
    bios_factory.register_bean('boot_option_service', boot_option_ser)
end

local function construct_ctx()
    local ctx = {}
    ctx.get_initiator = function()
        return {}
    end

    return ctx
end

-- ipmi:cmos
function TestBiosIpmi:test_update_bios_status()
    local req = {}
    local ctx = construct_ctx()
    req.ManufactureId = 2012

    local bios_ser = bios_factory.get_service('bios_service')
    local res = bios_ser:update_bios_status(req, ctx)
    lu.assertEquals(res.CompletionCode, comp_code.InvalidFieldRequest)

    req.ManufactureId = 2011
    res = bios_ser:update_bios_status(req, ctx)
    lu.assertEquals(res.CompletionCode, comp_code.InvalidFieldRequest)

    req.Status = 0
    res = bios_ser:update_bios_status(req, ctx)
    lu.assertEquals(res.CompletionCode, comp_code.InvalidFieldRequest)

    req.Status = 1
    res = bios_ser:update_bios_status(req, ctx)
    lu.assertEquals(res.CompletionCode, comp_code.Success)
end

-- ipmi:prop handler
function TestBiosIpmi:test_prop_hander()
    local res = prop_hander.get(3)
    lu.assertEquals(res:byte(), 0)

    res = prop_hander.get(4)
    lu.assertEquals(res:byte(), 1)

    res = prop_hander.get(5)
    lu.assertEquals(res:byte(), 0)

    res = prop_hander.get(6)
    lu.assertEquals(res, '1.1')
end

-- ipmi:获取bios属性
function TestBiosIpmi:test_get_bios_boot_info()
    local req = {}
    local ctx = construct_ctx()
    req.ManufactureId = 2012

    local bios_ser = bios_factory.get_service('bios_service')
    local res = bios_ser:get_bios_boot_info(req, ctx)
    lu.assertEquals(res.CompletionCode, comp_code.InvalidFieldRequest)

    req.ManufactureId = 2011
    res = bios_ser:get_bios_boot_info(req, ctx)
    lu.assertEquals(res.CompletionCode, comp_code.InvalidFieldRequest)

    req.PropOption = 0
    res = bios_ser:get_bios_boot_info(req, ctx)
    lu.assertEquals(res.CompletionCode, comp_code.InvalidFieldRequest)

    req.PropOption = 3
    req.BiosId = 0
    res = bios_ser:get_bios_boot_info(req, ctx)
    lu.assertEquals(res.CompletionCode, comp_code.Success)
    lu.assertEquals(res.ManufactureId, 2011)
    lu.assertEquals(res.PropLen, 1)
    lu.assertEquals(res.PropValue:byte(), 0)

    req.PropOption = 6
    req.BiosId = 0
    res = bios_ser:get_bios_boot_info(req, ctx)
    lu.assertEquals(res.CompletionCode, comp_code.Success)
    lu.assertEquals(res.ManufactureId, 2011)
    lu.assertEquals(res.PropLen, 3)
    lu.assertEquals(res.PropValue, '1.1')
end

-- ipmi:设置bios属性
function TestBiosIpmi:test_set_bios_boot_info()
    local req = {}
    local ctx = construct_ctx()
    req.ManufactureId = 2012

    local bios_ser = bios_factory.get_service('bios_service')
    local res = bios_ser:set_bios_boot_info(req, ctx)
    lu.assertEquals(res.CompletionCode, comp_code.InvalidFieldRequest)

    req.ManufactureId = 2011
    res = bios_ser:set_bios_boot_info(req, ctx)
    lu.assertEquals(res.CompletionCode, comp_code.ReqDataLenInvalid)

    req.PropOption = 0
    req.PropLen = 0
    req.Information = '111'
    res = bios_ser:set_bios_boot_info(req, ctx)
    lu.assertEquals(res.CompletionCode, comp_code.ReqDataLenInvalid)

    req.PropOption = 0
    req.PropLen = 1
    req.Information = '1'
    res = bios_ser:set_bios_boot_info(req, ctx)
    lu.assertEquals(res.CompletionCode, comp_code.InvalidFieldRequest)

    req.PropOption = 6
    req.PropLen = 3
    req.BiosId = 0
    req.Information = '2.2'
    res = bios_ser:set_bios_boot_info(req, ctx)
    lu.assertEquals(res.CompletionCode, comp_code.Success)
    res = bios_ser:get_bios_boot_info(req, ctx)
    lu.assertEquals(res.PropValue, '2.2')

    req.PropOption = 4
    req.PropLen = 1
    req.BiosId = 0
    req.Information = string.char('0')
    res = bios_ser:set_bios_boot_info(req, ctx)
    lu.assertEquals(res.CompletionCode, comp_code.Success)

    req.PropOption = 5
    req.PropLen = 1
    req.BiosId = 0
    req.Information = string.char('0')
    res = bios_ser:set_bios_boot_info(req, ctx)
    lu.assertEquals(res.CompletionCode, comp_code.InvalidFieldRequest)
end

function TestBiosIpmi:test_bios_set_file_changed()
    local bios_ser = bios_factory.get_service('bios_service')
    bios_ser:bios_set_file_changed()
end
